# GIMP - The GNU Image Manipulation Program
# Copyright (C) 1998-2003 Manish Singh <yosh@gimp.org>

# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.

package Gimp::CodeGen::pdb;

%arg_types = (
    size        => { name            => 'SIZE',
		     gtype           => 'G_TYPE_SIZE',
		     type            => 'gsize ',
		     const_type      => 'gsize ',
		     init_value      => '0',
		     get_value_func  => '$var = g_value_get_size ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_GET_INT ($value)',
		     set_value_func  => 'g_value_set_size ($value, $var)',
		     take_value_func => 'g_value_set_size ($value, $var)' },

    int32       => { name            => 'INT32',
		     gtype           => 'G_TYPE_INT',
		     type            => 'gint ',
		     const_type      => 'gint ',
		     init_value      => '0',
		     get_value_func  => '$var = g_value_get_int ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_GET_INT ($value)',
		     set_value_func  => 'g_value_set_int ($value, $var)',
		     take_value_func => 'g_value_set_int ($value, $var)' },

    double      => { name            => 'DOUBLE',
		     gtype           => 'G_TYPE_DOUBLE',
		     type            => 'gdouble ',
		     const_type      => 'gdouble ',
		     init_value      => '0.0',
		     get_value_func  => '$var = g_value_get_double ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_GET_DOUBLE ($value)',
		     set_value_func  => 'g_value_set_double ($value, $var)',
		     take_value_func => 'g_value_set_double ($value, $var)' },

    string      => { name            => 'STRING',
		     gtype           => 'G_TYPE_STRING',
		     type            => 'gchar *',
		     const_type      => 'const gchar *',
		     init_value      => 'NULL',
		     out_annotate    => '(transfer full)',
		     get_value_func  => '$var = g_value_get_string ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_DUP_STRING ($value)',
		     set_value_func  => 'g_value_set_string ($value, $var)',
		     take_value_func => 'g_value_take_string ($value, $var)' },

    strv        => { name            => 'STRV',
		     gtype           => 'G_TYPE_STRV',
		     type            => 'gchar **',
		     const_type      => 'const gchar **',
		     init_value      => 'NULL',
		     in_annotate     => '(array zero-terminated=1)',
		     out_annotate    => '(array zero-terminated=1) (transfer full)',
		     get_value_func  => '$var = g_value_get_boxed ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_DUP_STRV ($value)',
		     set_value_func  => 'g_value_set_boxed ($value, $var)',
		     take_value_func => 'g_value_take_boxed ($value, $var)' },

    bytes       => { name            => 'BYTES',
		     gtype           => 'G_TYPE_BYTES',
		     type            => 'GBytes *',
		     const_type      => 'GBytes *',
		     init_value      => 'NULL',
		     out_annotate    => '(transfer full)',
		     get_value_func  => '$var = g_value_get_boxed ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_DUP_BYTES ($value)',
		     set_value_func  => 'g_value_set_boxed ($value, $var)',
		     take_value_func => 'g_value_take_boxed ($value, $var)' },

    int32array  => { name            => 'INT32ARRAY',
		     gtype           => 'GIMP_TYPE_INT32_ARRAY',
		     type            => 'gint32 *',
		     const_type      => 'const gint32 *',
		     array           => 1,
		     init_value      => 'NULL',
		     in_annotate     => '(element-type gint32)',
		     out_annotate    => '(element-type gint32) (transfer full)',
		     get_value_func  => '$var = gimp_value_get_int32_array ($value, &$var_len)',
		     dup_value_func  => '$var = GIMP_VALUES_DUP_INT32_ARRAY ($value, $var_len)',
		     set_value_func  => 'gimp_value_set_int32_array ($value, $var, $var_len)',
		     take_value_func => 'gimp_value_take_int32_array ($value, $var, $var_len)' },

    doublearray  => { name           => 'DOUBLEARRAY',
		     gtype           => 'GIMP_TYPE_DOUBLE_ARRAY',
		     type            => 'gdouble *',
		     const_type      => 'const gdouble *',
		     array           => 1,
		     init_value      => 'NULL',
		     in_annotate     => '(element-type gdouble)',
		     out_annotate    => '(element-type gdouble) (transfer full)',
		     get_value_func  => '$var = gimp_value_get_double_array ($value, &$var_len)',
		     dup_value_func  => '$var = GIMP_VALUES_DUP_DOUBLE_ARRAY ($value, $var_len)',
		     set_value_func  => 'gimp_value_set_double_array ($value, $var, $var_len)',
		     take_value_func => 'gimp_value_take_double_array ($value, $var, $var_len)' },

    colorarray  => { name            => 'COLOR_ARRAY',
		     gtype           => 'GIMP_TYPE_COLOR_ARRAY',
		     type            => 'GeglColor **',
		     const_type      => 'const GeglColor **',
		     init_value      => 'NULL',
		     in_annotate     => '(array zero-terminated=1)',
		     out_annotate    => '(array zero-terminated=1) (transfer full)',
		     get_value_func  => '$var = g_value_get_boxed ($value)',
		     dup_value_func  => '$var = gimp_color_array_copy (g_value_get_boxed (gimp_value_array_index ($value)))',
		     set_value_func  => 'g_value_set_boxed ($value, $var)',
		     take_value_func => 'g_value_take_boxed ($value, $var)' },

    imagearray  => { name            => 'IMAGEARRAY',
		     gtype           => 'GIMP_TYPE_CORE_OBJECT_ARRAY',
		     type            => 'GimpImage **',
		     const_type      => 'const GimpImage **',
		     app_const_type  => 'GimpImage **',
		     init_value      => 'NULL',
		     in_annotate     => '(element-type GimpImage) (array zero-terminated=1)',
		     out_annotate    => '(element-type GimpImage) (array zero-terminated=1) (transfer container)',
		     get_value_func  => '$var = g_value_get_boxed ($value)',
		     dup_value_func  => '$var = g_value_dup_boxed (gimp_value_array_index ($value))',
		     set_value_func  => 'g_value_set_boxed ($value, $var)',
		     take_value_func => 'g_value_take_boxed ($value, $var)' },

    itemarray   => { name            => 'ITEMARRAY',
		     gtype           => 'GIMP_TYPE_CORE_OBJECT_ARRAY',
		     type            => 'GimpItem **',
		     const_type      => 'const GimpItem **',
		     app_const_type  => 'GimpItem **',
		     init_value      => 'NULL',
		     in_annotate     => '(element-type GimpItem) (array zero-terminated=1)',
		     out_annotate    => '(element-type GimpItem) (array zero-terminated=1) (transfer container)',
		     get_value_func  => '$var = g_value_get_boxed ($value)',
		     dup_value_func  => '$var = g_value_dup_boxed (gimp_value_array_index ($value))',
		     set_value_func  => 'g_value_set_boxed ($value, $var)',
		     take_value_func => 'g_value_take_boxed ($value, $var)' },

    drawablearray  => { name            => 'DRAWABLEARRAY',
		     gtype           => 'GIMP_TYPE_CORE_OBJECT_ARRAY',
		     type            => 'GimpDrawable **',
		     const_type      => 'const GimpDrawable **',
		     app_const_type  => 'GimpDrawable **',
		     init_value      => 'NULL',
		     in_annotate     => '(element-type GimpDrawable) (array zero-terminated=1)',
		     out_annotate    => '(element-type GimpDrawable) (array zero-terminated=1) (transfer container)',
		     get_value_func  => '$var = g_value_get_boxed ($value)',
		     dup_value_func  => '$var = g_value_dup_boxed (gimp_value_array_index ($value))',
		     set_value_func  => 'g_value_set_boxed ($value, $var)',
		     take_value_func => 'g_value_take_boxed ($value, $var)' },

    layerarray  => { name            => 'LAYERARRAY',
		     gtype           => 'GIMP_TYPE_CORE_OBJECT_ARRAY',
		     type            => 'GimpLayer **',
		     const_type      => 'const GimpLayer **',
		     app_const_type  => 'GimpLayer **',
		     init_value      => 'NULL',
		     in_annotate     => '(element-type GimpLayer) (array zero-terminated=1)',
		     out_annotate    => '(element-type GimpLayer) (array zero-terminated=1) (transfer container)',
		     get_value_func  => '$var = g_value_get_boxed ($value)',
		     dup_value_func  => '$var = g_value_dup_boxed (gimp_value_array_index ($value))',
		     set_value_func  => 'g_value_set_boxed ($value, $var)',
		     take_value_func => 'g_value_take_boxed ($value, $var)' },

    channelarray => { name           => 'CHANNELARRAY',
		      gtype          => 'GIMP_TYPE_CORE_OBJECT_ARRAY',
		      type           => 'GimpChannel **',
		      const_type     => 'const GimpChannel **',
		      app_const_type => 'GimpChannel **',
		      init_value     => 'NULL',
		      in_annotate    => '(element-type GimpChannel) (array zero-terminated=1)',
		     out_annotate    => '(element-type GimpChannel) (array zero-terminated=1) (transfer container)',
		     get_value_func  => '$var = g_value_get_boxed ($value)',
		     dup_value_func  => '$var = g_value_dup_boxed (gimp_value_array_index ($value))',
		     set_value_func  => 'g_value_set_boxed ($value, $var)',
		     take_value_func => 'g_value_take_boxed ($value, $var)' },

    patharray    => { name            => 'PATHARRAY',
		      gtype           => 'GIMP_TYPE_CORE_OBJECT_ARRAY',
		      type            => 'GimpPath **',
		      const_type      => 'const GimpPath **',
		      app_const_type  => 'GimpPath **',
		      init_value      => 'NULL',
		      in_annotate     => '(element-type GimpPath) (array zero-terminated=1)',
		     out_annotate    => '(element-type GimpPath) (array zero-terminated=1) (transfer container)',
		     get_value_func  => '$var = g_value_get_boxed ($value)',
		     dup_value_func  => '$var = g_value_dup_boxed (gimp_value_array_index ($value))',
		     set_value_func  => 'g_value_set_boxed ($value, $var)',
		     take_value_func => 'g_value_take_boxed ($value, $var)' },

    filterarray => { name           => 'FILTERARRAY',
		     gtype           => 'GIMP_TYPE_CORE_OBJECT_ARRAY',
		     type            => 'GimpDrawableFilter **',
		     const_type      => 'const GimpDrawableFilter **',
		     app_const_type  => 'GimpDrawableFilter **',
		     init_value      => 'NULL',
		     in_annotate     => '(element-type GimpDrawableFilter) (array zero-terminated=1)',
		     out_annotate    => '(element-type GimpDrawableFilter) (array zero-terminated=1) (transfer container)',
		     get_value_func  => '$var = g_value_get_boxed ($value)',
		     dup_value_func  => '$var = g_value_dup_boxed (gimp_value_array_index ($value))',
		     set_value_func  => 'g_value_set_boxed ($value, $var)',
		     take_value_func => 'g_value_take_boxed ($value, $var)' },

    resourcearray => { name          => 'RESOURCEARRAY',
		     gtype           => 'GIMP_TYPE_CORE_OBJECT_ARRAY',
		     type            => 'GimpResource **',
		     const_type      => 'const GimpResource **',
		     app_const_type  => 'GimpResource **',
		     init_value      => 'NULL',
		     in_annotate     => '(element-type GimpResource) (array zero-terminated=1)',
		     out_annotate    => '(element-type GimpResource) (array zero-terminated=1) (transfer container)',
		     get_value_func  => '$var = g_value_get_boxed ($value)',
		     dup_value_func  => '$var = g_value_dup_boxed (gimp_value_array_index ($value))',
		     set_value_func  => 'g_value_set_boxed ($value, $var)',
		     take_value_func => 'g_value_take_boxed ($value, $var)' },

    brusharray  => { name            => 'BRUSHARRAY',
		     gtype           => 'GIMP_TYPE_CORE_OBJECT_ARRAY',
		     type            => 'GimpBrush **',
		     const_type      => 'const GimpBrush **',
		     init_value      => 'NULL',
		     in_annotate     => '(element-type GimpBrush) (array zero-terminated=1)',
		     out_annotate    => '(element-type GimpBrush) (array zero-terminated=1) (transfer container)',
		     get_value_func  => '$var = g_value_get_boxed ($value)',
		     dup_value_func  => '$var = g_value_dup_boxed (gimp_value_array_index ($value))',
		     set_value_func  => 'g_value_set_boxed ($value, $var)',
		     take_value_func => 'g_value_take_boxed ($value, $var)' },

    fontarray   => { name            => 'FONTARRAY',
		     gtype           => 'GIMP_TYPE_CORE_OBJECT_ARRAY',
		     type            => 'GimpFont **',
		     const_type      => 'const GimpFont **',
		     app_const_type  => 'GimpFont **',
		     init_value      => 'NULL',
		     in_annotate     => '(element-type GimpFont) (array zero-terminated=1)',
		     out_annotate    => '(element-type GimpFont) (array zero-terminated=1) (transfer container)',
		     get_value_func  => '$var = g_value_get_boxed ($value)',
		     dup_value_func  => '$var = g_value_dup_boxed (gimp_value_array_index ($value))',
		     set_value_func  => 'g_value_set_boxed ($value, $var)',
		     take_value_func => 'g_value_take_boxed ($value, $var)' },

    gradientarray => { name          => 'GRADIENTARRAY',
		     gtype           => 'GIMP_TYPE_CORE_OBJECT_ARRAY',
		     type            => 'GimpGradient **',
		     const_type      => 'const GimpGradient **',
		     app_const_type  => 'GimpGradient **',
		     init_value      => 'NULL',
		     in_annotate     => '(element-type GimpGradient) (array zero-terminated=1)',
		     out_annotate    => '(element-type GimpGradient) (array zero-terminated=1) (transfer container)',
		     get_value_func  => '$var = g_value_get_boxed ($value)',
		     dup_value_func  => '$var = g_value_dup_boxed (gimp_value_array_index ($value))',
		     set_value_func  => 'g_value_set_boxed ($value, $var)',
		     take_value_func => 'g_value_take_boxed ($value, $var)' },

    palettearray => { name           => 'PALETTEARRAY',
		     gtype           => 'GIMP_TYPE_CORE_OBJECT_ARRAY',
		     type            => 'GimpPalette **',
		     const_type      => 'const GimpPalette **',
		     app_const_type  => 'GimpPalette **',
		     init_value      => 'NULL',
		     in_annotate     => '(element-type GimpPalette) (array zero-terminated=1)',
		     out_annotate    => '(element-type GimpPalette) (array zero-terminated=1) (transfer container)',
		     get_value_func  => '$var = g_value_get_boxed ($value)',
		     dup_value_func  => '$var = g_value_dup_boxed (gimp_value_array_index ($value))',
		     set_value_func  => 'g_value_set_boxed ($value, $var)',
		     take_value_func => 'g_value_take_boxed ($value, $var)' },

    patternarray => { name           => 'PATTERNARRAY',
		     gtype           => 'GIMP_TYPE_CORE_OBJECT_ARRAY',
		     type            => 'GimpPattern **',
		     const_type      => 'const GimpPattern **',
		     app_const_type  => 'GimpPattern **',
		     init_value      => 'NULL',
		     in_annotate     => '(element-type GimpPattern) (array zero-terminated=1)',
		     out_annotate    => '(element-type GimpPattern) (array zero-terminated=1) (transfer container)',
		     get_value_func  => '$var = g_value_get_boxed ($value)',
		     dup_value_func  => '$var = g_value_dup_boxed (gimp_value_array_index ($value))',
		     set_value_func  => 'g_value_set_boxed ($value, $var)',
		     take_value_func => 'g_value_take_boxed ($value, $var)' },

    geglcolor   => { name            => 'GEGLCOLOR',
		     gtype           => 'GEGL_TYPE_COLOR',
		     type            => 'GeglColor *',
		     const_type      => 'GeglColor *',
		     init_value      => 'NULL',
		     out_annotate    => '(transfer full)',
		     get_value_func  => '$var = g_value_get_object ($value)',
		     dup_value_func  => '$var = g_value_dup_object (gimp_value_array_index ($value))',
		     set_value_func  => 'g_value_set_object ($value, $var)',
		     take_value_func => 'g_value_take_object ($value, $var)',
		     headers         => [ qw(<cairo.h> "libgimpcolor/gimpcolor.h") ] },

    format      => { name            => 'FORMAT',
		     gtype           => 'GIMP_TYPE_BABL_FORMAT',
		     type            => 'const Babl *',
		     const_type      => 'const Babl *',
		     init_value      => 'NULL',
		     out_annotate    => '(transfer none)',
		     get_value_func  => '$var = g_value_get_boxed ($value)',
		     dup_value_func  => '$var = g_value_get_boxed (gimp_value_array_index ($value))',
		     set_value_func  => 'g_value_set_boxed ($value, $var)',
		     take_value_func => 'g_value_set_boxed ($value, $var)' },

    display     => { name            => 'DISPLAY',
		     gtype           => 'GIMP_TYPE_DISPLAY',
		     type            => 'GimpDisplay *',
		     const_type      => 'GimpDisplay *',
		     app_type        => 'GimpDisplay *',
		     app_const_type  => 'GimpDisplay *',
		     init_value      => 'NULL',
		     out_annotate    => '(transfer none)',
		     get_value_func  => '$var = g_value_get_object ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_GET_DISPLAY ($value)',
		     set_value_func  => 'g_value_set_object ($value, $var)',
		     take_value_func => 'g_value_set_object ($value, $var)' },

    image       => { name            => 'IMAGE',
		     gtype           => 'GIMP_TYPE_IMAGE',
		     type            => 'GimpImage *',
		     const_type      => 'GimpImage *',
		     init_value      => 'NULL',
		     out_annotate    => '(transfer none)',
		     get_value_func  => '$var = g_value_get_object ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_GET_IMAGE ($value)',
		     set_value_func  => 'g_value_set_object ($value, $var)',
		     take_value_func => 'g_value_set_object ($value, $var)',
		     headers         => [ qw("core/gimpimage.h") ] },

    item        => { name            => 'ITEM',
		     gtype           => 'GIMP_TYPE_ITEM',
		     type            => 'GimpItem *',
		     const_type      => 'GimpItem *',
		     init_value      => 'NULL',
		     out_annotate    => '(transfer none)',
		     get_value_func  => '$var = g_value_get_object ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_GET_ITEM ($value)',
		     set_value_func  => 'g_value_set_object ($value, $var)',
		     take_value_func => 'g_value_set_object ($value, $var)',
		     headers         => [ qw("core/gimpitem.h") ] },

    layer       => { name            => 'LAYER',
		     gtype           => 'GIMP_TYPE_LAYER',
		     type            => 'GimpLayer *',
		     const_type      => 'GimpLayer *',
		     init_value      => 'NULL',
		     out_annotate    => '(transfer none)',
		     get_value_func  => '$var = g_value_get_object ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_GET_LAYER ($value)',
		     set_value_func  => 'g_value_set_object ($value, $var)',
		     take_value_func => 'g_value_set_object ($value, $var)',
		     headers         => [ qw("core/gimplayer.h") ] },

    text_layer  => { name            => 'TEXT_LAYER',
		     gtype           => 'GIMP_TYPE_TEXT_LAYER',
		     type            => 'GimpTextLayer *',
		     const_type      => 'GimpTextLayer *',
		     init_value      => 'NULL',
		     out_annotate    => '(transfer none)',
		     get_value_func  => '$var = g_value_get_object ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_GET_TEXT_LAYER ($value)',
		     set_value_func  => 'g_value_set_object ($value, $var)',
		     take_value_func => 'g_value_set_object ($value, $var)',
		     headers         => [ qw("text/gimptextlayer.h") ] },

    group_layer => { name            => 'GROUP_LAYER',
		     gtype           => 'GIMP_TYPE_GROUP_LAYER',
		     type            => 'GimpGroupLayer *',
		     const_type      => 'GimpGroupLayer *',
		     init_value      => 'NULL',
		     out_annotate    => '(transfer none)',
		     get_value_func  => '$var = g_value_get_object ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_GET_GROUP_LAYER ($value)',
		     set_value_func  => 'g_value_set_object ($value, $var)',
		     take_value_func => 'g_value_set_object ($value, $var)',
		     headers         => [ qw("core/gimpgrouplayer.h") ] },

    channel     => { name            => 'CHANNEL',
		     gtype           => 'GIMP_TYPE_CHANNEL',
		     type            => 'GimpChannel *',
		     const_type      => 'GimpChannel *',
		     init_value      => 'NULL',
		     out_annotate    => '(transfer none)',
		     get_value_func  => '$var = g_value_get_object ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_GET_CHANNEL ($value)',
		     set_value_func  => 'g_value_set_object ($value, $var)',
		     take_value_func => 'g_value_set_object ($value, $var)',
		     headers         => [ qw("core/gimpchannel.h") ] },

    drawable    => { name            => 'DRAWABLE',
		     gtype           => 'GIMP_TYPE_DRAWABLE',
		     type            => 'GimpDrawable *',
		     const_type      => 'GimpDrawable *',
		     init_value      => 'NULL',
		     out_annotate    => '(transfer none)',
		     get_value_func  => '$var = g_value_get_object ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_GET_DRAWABLE ($value)',
		     set_value_func  => 'g_value_set_object ($value, $var)',
		     take_value_func => 'g_value_set_object ($value, $var)',
		     headers         => [ qw("core/gimpdrawable.h") ] },

    selection   => { name            => 'SELECTION',
		     gtype           => 'GIMP_TYPE_SELECTION',
		     type            => 'GimpSelection *',
		     const_type      => 'GimpSelection *',
		     init_value      => 'NULL',
		     out_annotate    => '(transfer none)',
		     get_value_func  => '$var = g_value_get_object ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_GET_SELECTION ($value)',
		     set_value_func  => 'g_value_set_object ($value, $var)',
		     take_value_func => 'g_value_set_object ($value, $var)',
		     headers         => [ qw("core/gimpselection.h") ] },

    layer_mask  => { name            => 'CHANNEL',
		     gtype           => 'GIMP_TYPE_LAYER_MASK',
		     type            => 'GimpLayerMask *',
		     const_type      => 'GimpLayerMask *',
		     init_value      => 'NULL',
		     out_annotate    => '(transfer none)',
		     get_value_func  => '$var = g_value_get_object ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_GET_LAYER_MASK ($value)',
		     set_value_func  => 'g_value_set_object ($value, $var)',
		     take_value_func => 'g_value_set_object ($value, $var)',
		     headers         => [ qw("core/gimplayermask.h") ] },

    path        => { name            => 'PATH',
		     gtype           => 'GIMP_TYPE_PATH',
		     type            => 'GimpPath *',
		     const_type      => 'GimpPath *',
		     init_value      => 'NULL',
		     out_annotate    => '(transfer none)',
		     get_value_func  => '$var = g_value_get_object ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_GET_PATH ($value)',
		     set_value_func  => 'g_value_set_object ($value, $var)',
		     take_value_func => 'g_value_set_object ($value, $var)',
		     headers         => [ qw("vectors/gimppath.h") ] },

    filter      => { name            => 'DRAWABLE_FILTER',
		     gtype           => 'GIMP_TYPE_DRAWABLE_FILTER',
		     type            => 'GimpDrawableFilter *',
		     const_type      => 'GimpDrawableFilter *',
		     init_value      => 'NULL',
		     out_annotate    => '(transfer none)',
		     get_value_func  => '$var = g_value_get_object ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_GET_DRAWABLE_FILTER ($value)',
		     set_value_func  => 'g_value_set_object ($value, $var)',
		     take_value_func => 'g_value_set_object ($value, $var)',
		     headers         => [ qw("core/gimpdrawablefilter.h") ] },

    file        => { name            => 'FILE',
		     gtype           => 'G_TYPE_FILE',
		     type            => 'GFile *',
		     const_type      => 'GFile *',
		     init_value      => 'NULL',
		     out_annotate    => '(transfer full)',
		     get_value_func  => '$var = g_value_get_object ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_DUP_FILE ($value)',
		     set_value_func  => 'g_value_set_object ($value, $var)',
		     take_value_func => 'g_value_take_object ($value, $var)' },

    parasite    => { name            => 'PARASITE',
		     gtype           => 'GIMP_TYPE_PARASITE',
		     type            => 'GimpParasite *',
		     const_type      => 'const GimpParasite *',
		     init_value      => 'NULL',
		     out_annotate    => '(transfer full)',
		     get_value_func  => '$var = g_value_get_boxed ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_DUP_PARASITE ($value)',
		     set_value_func  => 'g_value_set_boxed ($value, $var)',
		     take_value_func => 'g_value_take_boxed ($value, $var)',
		     headers         => [ qw("libgimpbase/gimpbase.h") ] },

    param       => { name            => 'PARAM',
		     gtype           => 'G_TYPE_PARAM_SPEC',
		     type            => 'GParamSpec *',
		     const_type      => 'GParamSpec *',
		     init_value      => 'NULL',
		     out_annotate    => '(transfer full)',
		     get_value_func  => '$var = g_value_get_param ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_DUP_PARAM ($value)',
		     set_value_func  => 'g_value_set_param ($value, $var)',
		     take_value_func => 'g_value_take_param ($value, $var)' },

    # Special cases
    enum        => { name            => 'ENUM',
		     gtype           => 'G_TYPE_ENUM',
		     type            => 'gint ',
		     const_type      => 'gint ',
		     init_value      => '0',
		     get_value_func  => '$var = g_value_get_enum ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_GET_ENUM ($value)',
		     set_value_func  => 'g_value_set_enum ($value, $var)',
		     take_value_func => 'g_value_set_enum ($value, $var)' },

    boolean     => { name            => 'BOOLEAN',
		     gtype           => 'G_TYPE_BOOLEAN',
		     type            => 'gboolean ',
		     const_type      => 'gboolean ',
		     init_value      => 'FALSE',
		     get_value_func  => '$var = g_value_get_boolean ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_GET_BOOLEAN ($value)',
		     set_value_func  => 'g_value_set_boolean ($value, $var)',
		     take_value_func => 'g_value_set_boolean ($value, $var)' },

    tattoo      => { name            => 'TATTOO',
		     gtype           => 'G_TYPE_UINT',
		     type            => 'guint ',
		     const_type      => 'guint ',
		     init_value      => '0',
		     get_value_func  => '$var = g_value_get_uint ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_GET_UINT ($value)',
		     set_value_func  => 'g_value_set_uint ($value, $var)',
		     take_value_func => 'g_value_set_uint ($value, $var)' },

    guide       => { name            => 'GUIDE',
		     gtype           => 'G_TYPE_UINT',
		     type            => 'guint ',
		     const_type      => 'guint ',
		     init_value      => '0',
		     get_value_func  => '$var = g_value_get_uint ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_GET_UINT ($value)',
		     set_value_func  => 'g_value_set_uint ($value, $var)',
		     take_value_func => 'g_value_set_uint ($value, $var)' },

   sample_point => { name            => 'SAMPLE_POINT',
		     gtype           => 'G_TYPE_UINT',
		     type            => 'guint ',
		     const_type      => 'guint ',
		     init_value      => '0',
		     get_value_func  => '$var = g_value_get_uint ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_GET_UINT ($value)',
		     set_value_func  => 'g_value_set_uint ($value, $var)',
		     take_value_func => 'g_value_set_uint ($value, $var)' },

    export_options => { name         => 'EXPORT_OPTIONS',
		     gtype           => 'GIMP_TYPE_EXPORT_OPTIONS',
		     type            => 'GimpExportOptions *',
		     const_type      => 'GimpExportOptions *',
		     init_value      => 'NULL',
		     in_annotate     => '(nullable)',
		     out_annotate    => '(transfer full)',
		     get_value_func  => '$var = g_value_get_object ($value)',
		     dup_value_func  => '$var = g_value_dup_object (gimp_value_array_index ($value))',
		     set_value_func  => 'g_value_set_object ($value, $var)',
		     take_value_func => 'g_value_set_object ($value, $var)',
		     headers         => [ qw("libgimpbase/gimpbase.h") ] },

    unit        => { name            => 'UNIT',
		     gtype           => 'GIMP_TYPE_UNIT',
		     type            => 'GimpUnit *',
		     const_type      => 'GimpUnit *',
		     init_value      => 'NULL',
		     out_annotate    => '(transfer none)',
		     get_value_func  => '$var = g_value_get_object ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_GET_UNIT ($value)',
		     set_value_func  => 'g_value_set_object ($value, $var)',
		     take_value_func => 'g_value_set_object ($value, $var)',
		     headers         => [ qw("libgimpbase/gimpbase.h") ] },

    resource    => { name            => 'RESOURCE',
		     gtype           => 'GIMP_TYPE_RESOURCE',
		     type            => 'GimpResource *',
		     const_type      => 'GimpResource *',
		     init_value      => 'NULL',
		     out_annotate    => '(transfer none)',
		     get_value_func  => '$var = g_value_get_object ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_GET_RESOURCE ($value)',
		     set_value_func  => 'g_value_set_object ($value, $var)',
		     take_value_func => 'g_value_set_object ($value, $var)',
		     headers         => [ qw("core/gimpresource.h") ] },

    brush       => { name            => 'BRUSH',
		     gtype           => 'GIMP_TYPE_BRUSH',
		     type            => 'GimpBrush *',
		     const_type      => 'GimpBrush *',
		     init_value      => 'NULL',
		     out_annotate    => '(transfer none)',
		     get_value_func  => '$var = g_value_get_object ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_GET_BRUSH ($value)',
		     set_value_func  => 'g_value_set_object ($value, $var)',
		     take_value_func => 'g_value_set_object ($value, $var)',
		     headers         => [ qw("core/gimpbrush.h") ] },


    pattern     => { name            => 'PATTERN',
		     gtype           => 'GIMP_TYPE_PATTERN',
		     type            => 'GimpPattern *',
		     const_type      => 'GimpPattern *',
		     init_value      => 'NULL',
		     out_annotate    => '(transfer none)',
		     get_value_func  => '$var = g_value_get_object ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_GET_PATTERN ($value)',
		     set_value_func  => 'g_value_set_object ($value, $var)',
		     take_value_func => 'g_value_set_object ($value, $var)',
		     headers         => [ qw("core/gimppattern.h") ] },

    gradient    => { name            => 'GRADIENT',
		     gtype           => 'GIMP_TYPE_GRADIENT',
		     type            => 'GimpGradient *',
		     const_type      => 'GimpGradient *',
		     init_value      => 'NULL',
		     out_annotate    => '(transfer none)',
		     get_value_func  => '$var = g_value_get_object ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_GET_GRADIENT ($value)',
		     set_value_func  => 'g_value_set_object ($value, $var)',
		     take_value_func => 'g_value_set_object ($value, $var)',
		     headers         => [ qw("core/gimpgradient.h") ] },

    palette     => { name            => 'PALETTE',
		     gtype           => 'GIMP_TYPE_PALETTE',
		     type            => 'GimpPalette *',
		     const_type      => 'GimpPalette *',
		     init_value      => 'NULL',
		     out_annotate    => '(transfer none)',
		     get_value_func  => '$var = g_value_get_object ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_GET_PALETTE ($value)',
		     set_value_func  => 'g_value_set_object ($value, $var)',
		     take_value_func => 'g_value_set_object ($value, $var)',
		     headers         => [ qw("core/gimppalette.h") ] },

    font        => { name            => 'FONT',
		     gtype           => 'GIMP_TYPE_FONT',
		     type            => 'GimpFont *',
		     const_type      => 'GimpFont *',
		     init_value      => 'NULL',
		     out_annotate    => '(transfer none)',
		     get_value_func  => '$var = g_value_get_object ($value)',
		     dup_value_func  => '$var = GIMP_VALUES_GET_FONT ($value)',
		     set_value_func  => 'g_value_set_object ($value, $var)',
		     take_value_func => 'g_value_set_object ($value, $var)',
		     headers         => [ qw("text/gimpfont.h") ] },

    valuearray  => { name            => 'VALUEARRAY',
		     gtype           => 'GIMP_TYPE_VALUE_ARRAY',
		     type            => 'GimpValueArray *',
		     const_type      => 'const GimpValueArray *',
		     init_value      => 'NULL',
		     get_value_func  => '$var = g_value_get_boxed ($value)',
		     dup_value_func  => '$var = g_value_dup_boxed (gimp_value_array_index ($value))',
		     set_value_func  => 'g_value_set_boxed ($value, $var)',
		     take_value_func => 'g_value_take_boxed ($value, $var)' },
);

# Split out the parts of an arg constraint
sub arg_parse {
    my $arg = shift;

    if ($arg =~ /^enum (\w+)(.*)/) {
	my ($name, $remove) = ($1, $2);
	my @retvals = ('enum', $name);

	if ($remove && $remove =~ m@ \(no @) {
	    chop $remove; ($remove = substr($remove, 5)) =~ s/ $//;
	    push @retvals, split(/,\s*/, $remove);
	}

	return @retvals;
    }
    elsif ($arg =~ /^unit(?: \(min (.*?)\))?/) {
	my @retvals = ('unit');
	push @retvals, $1 if $1;
	return @retvals;
    }
    elsif ($arg =~ /^(?:([+-.\dA-Z_][^\s]*) \s* (<=|<))?
		     \s* (\w+) \s*
		     (?:(<=|<) \s* ([+-.\dA-Z_][^\s]*))?
		   /x) {
	return ($3, $1, $2, $5, $4);
    }
}

1;
