# chat_page.py
#
# Copyright 2024 Christopher Talbot
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.
#
# SPDX-License-Identifier: GPL-3.0-or-later

import json
import base64
from gi.repository import Adw
from gi.repository import Gtk
from gi.repository import GLib
import meshtastic
from meshtastic import BROADCAST_NUM
from datetime import datetime, timezone

import gtk_meshtastic_client.channel_row as channel_row
import gtk_meshtastic_client.message_bar as message_bar
import gtk_meshtastic_client.message_row as message_row
import gtk_meshtastic_client.message_storage as message_storage
import gtk_meshtastic_client.utils as utils

DBL_EPSILON = 2.2204460492503131e-16
SCROLL_THRESHOLD = 2 * DBL_EPSILON
SCROLL_TIMEOUT = 60  #milliseconds

@Gtk.Template(resource_path='/org/kop316/meshtastic/ui/chat_page.ui')
class ChatNavPage(Adw.Bin):
    __gtype_name__ = 'ChatNavPage'

    message_list = Gtk.Template.Child()
    message_bar = Gtk.Template.Child()
    window_title = Gtk.Template.Child()
    scroll_down_button = Gtk.Template.Child()
    scrolled_window = Gtk.Template.Child()
    vadjustment = Gtk.Template.Child()
    scroll_down_revealer = Gtk.Template.Child()

    send_address = meshtastic.BROADCAST_NUM
    title = " "
    index = -1
    message_list_children = 0
    page_loaded = False
    is_bottom = False
    value_adjust = 0
    scroll_bottom_id = 0
    # "MA==" is the default, indicating there is no public key
    # It is the base64 equivalent of Hex 0x00
    publicKey = "MA=="

    def find_message_by_id(self, message_id):
        for x in range(self.message_list_children):
            message_row = self.message_list.get_row_at_index(x)
            if message_id == message_row.message_id:
                return message_row

    def clear_message_list(self):
        self.message_list_children = 0
        self.message_list.remove_all()

    def set_chat_index(self, index):
        self.index = index
        self.message_bar.set_channel_index(self.index)

    def set_public_key(self, publicKey):
        if not publicKey:
            self.logger.warning("publickey is NULL! This is a programming Error")
            return
        elif publicKey == "":
            self.logger.warning("publickey is empty! This is a programming Error")
            return
        self.publicKey = publicKey
        self.message_bar.set_public_key(self.publicKey)

    def set_channel_id(self, title, psk_base64):
        app = Gtk.Application.get_default()
        win = Gtk.Application.get_active_window(app)

        if not title and psk_base64 == "AQ==":
            title = "LongFast"
        elif title == "" and psk_base64 == "AQ==":
            title = "LongFast"

        self.channel_id = message_storage.create_channel_id(title, psk_base64)
        self.message_bar.set_channel_id(self.channel_id)

        win.connection_page_bin.database.load_messages(self, self.channel_id, 0)

    def set_dm_id(self, id):
        app = Gtk.Application.get_default()
        win = Gtk.Application.get_active_window(app)

        self.channel_id = id
        self.message_bar.set_channel_id(self.channel_id)

        win.connection_page_bin.database.load_messages(self, self.channel_id, 0)

    def set_send_address(self, send_address):
        self.send_address = send_address
        self.message_bar.set_send_address(self.send_address)

    def set_interface(self, interface):
        self.interface = interface
        self.message_bar.set_interface(self.interface)

    def set_chat_title(self, title, DirectMessage):
        if not title:
            return
        else:
            self.title = title

        title_string = ""

        if self.publicKey != "MA==":
            title_string = title_string + "Encrypted "

        if not DirectMessage:
            title_string = title_string + "Channel: " + str(self.title)
        else:
            title_string = title_string + "Direct Message: " + str(self.title)

        self.window_title.set_title(title_string)

    def _sort_func(self, message1, message2, data=None):
            return message1.date - message2.date

    def delete_msg_by_id(self, message_id):
        row_to_delete = self.find_message_by_id(message_id)
        if row_to_delete:
            self.message_list.remove(row_to_delete)
            self.message_list_children -= 1
            return

    def update_msg_delivered_by_id(self, message_id):
        row_to_update = self.find_message_by_id(message_id)
        if row_to_update:
            row_to_update.update_delivery(True)
            return

    def focus_cb(self) -> bool:
        self.message_bar.message_input.grab_focus()
        return GLib.SOURCE_REMOVE

    def message_bar_grab_focus(self):
        """
        You need a slight delay for the Message Input to grab focus
        """
        self._focus_timeout_id = GLib.timeout_add(100, self.focus_cb)

    def update_ui(self):
        for x in range(self.message_list_children):
            row_to_update = self.message_list.get_row_at_index(x)
            row_to_update.update_ui()

    def update_ui_periodic(self) -> bool:
        if hasattr(self, 'message_list'):
            self.logger.debug("Update node UI")
            self.update_ui()
            return GLib.SOURCE_CONTINUE
        else:
            return GLib.SOURCE_REMOVE

    """
    Since we added a sort function, the rows will automatically sort when
    appended, no need to manually specifiy
    """
    def add_message(self, row):
        self.message_list.append(row)
        self.message_list_children += 1

    def scroll_to_bottom(self):
        #Temporarily disable kinetic scrolling
        #Otherwise, adjustment value isn't updated if kinetic scrolling is active
        self.scrolled_window.set_kinetic_scrolling (False)
        self.vadjustment.set_value (self.vadjustment.get_upper() - self.vadjustment.get_page_size())
        self.scrolled_window.set_kinetic_scrolling (True)
        self.is_bottom = True

    @Gtk.Template.Callback()
    def page_edge_overshot_cb(self, window, pos):
        if pos == Gtk.PositionType.TOP:
            app = Gtk.Application.get_default()
            win = Gtk.Application.get_active_window(app)

            messages_in_history = win.connection_page_bin.database.get_message_count(self.channel_id)
            if self.message_list_children >= messages_in_history:
                return

            win.connection_page_bin.database.load_messages(self, self.channel_id, self.message_list_children)

    @Gtk.Template.Callback()
    def chat_page_scroll_down_clicked_cb(self, button):
        self.scroll_to_bottom()

    def scroll_to_bottom_cb(self) -> bool:
        self.scroll_to_bottom()
        return GLib.SOURCE_REMOVE

    def scroll_is_bottom(self):
        value = self.vadjustment.get_value()
        upper = self.vadjustment.get_upper()
        page_size = self.vadjustment.get_page_size()

        return abs(upper - value - page_size) <= SCROLL_THRESHOLD

    def update_view_scroll(self) -> bool:
        value = self.vadjustment.get_value()
        upper = self.vadjustment.get_upper()
        page_size = self.vadjustment.get_page_size()

        if abs(upper - page_size) <= SCROLL_THRESHOLD:
            return GLib.SOURCE_REMOVE

        # If close to bottom, scroll to bottom
        # This is useful for incoming messages
        if upper - value < (page_size * 1.75):
            self.vadjustment.set_value (upper - page_size);

        return GLib.SOURCE_REMOVE

    def handle_adjustment_changed(self, vadjustment):
        is_bottom = False
        is_scroll = False
        value = self.vadjustment.get_value()
        if not abs(value - self.value_adjust) <= SCROLL_THRESHOLD:
            is_scroll = True
            self.value_adjust = value

        is_bottom = self.scroll_is_bottom()
        self.scroll_down_revealer.set_reveal_child (not is_bottom)

        if ((is_bottom != self.is_bottom) and not is_scroll):
            if self.scroll_bottom_id != 0:
                GLib.source_remove(self.scroll_bottom_id)
                self.scroll_bottom_id = 0
            self.scroll_bottom_id = GLib.timeout_add(SCROLL_TIMEOUT, self.update_view_scroll)
            self.scroll_down_revealer.set_reveal_child (False) # avoids flickering the button

        self.is_bottom = is_bottom
        return

    @Gtk.Template.Callback()
    def chat_page_adjustment_changed_cb(self, vadjustment):
        if not self.page_loaded:
            self._timeout_id = GLib.timeout_add(50, self.scroll_to_bottom_cb)
            self.page_loaded = True

        self.handle_adjustment_changed(vadjustment)

    @Gtk.Template.Callback()
    def chat_page_adjustment_value_changed_cb(self, vadjustment):
        self.handle_adjustment_changed(vadjustment)

    def __init__(self, **kwargs):
        super().__init__(**kwargs)
        app = Gtk.Application.get_default()

        self.message_bar.set_nav_page(self)
        self.message_list.set_sort_func(sort_func=self._sort_func)
        self.logger = app.logger
        GLib.timeout_add(60000, self.update_ui_periodic)
