/* ************************************************************************
 * Copyright (C) 2016-2024 Advanced Micro Devices, Inc. All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * ************************************************************************ */

#include "hipblas.h"
#include "exceptions.hpp"
#include <cublas_v2.h>
#include <cuda_runtime_api.h>
#include <hip/hip_runtime.h>

#ifdef __cplusplus
extern "C" {
#endif

cublasOperation_t hipblasConvertOperation(hipblasOperation_t op)
{
    switch(op)
    {
    case HIPBLAS_OP_N:
        return CUBLAS_OP_N;

    case HIPBLAS_OP_T:
        return CUBLAS_OP_T;

    case HIPBLAS_OP_C:
        return CUBLAS_OP_C;

    default:
        throw HIPBLAS_STATUS_INVALID_ENUM;
    }
}

cublasFillMode_t hipblasConvertFill(hipblasFillMode_t fill)
{
    switch(fill)
    {
    case HIPBLAS_FILL_MODE_UPPER:
        return CUBLAS_FILL_MODE_UPPER;
    case HIPBLAS_FILL_MODE_LOWER:
        return CUBLAS_FILL_MODE_LOWER;
    case HIPBLAS_FILL_MODE_FULL:
        return CUBLAS_FILL_MODE_FULL;
    default:
        throw HIPBLAS_STATUS_INVALID_ENUM;
    }
}

cublasDiagType_t hipblasConvertDiag(hipblasDiagType_t diagonal)
{
    switch(diagonal)
    {
    case HIPBLAS_DIAG_NON_UNIT:
        return CUBLAS_DIAG_NON_UNIT;
    case HIPBLAS_DIAG_UNIT:
        return CUBLAS_DIAG_UNIT;
    default:
        throw HIPBLAS_STATUS_INVALID_ENUM;
    }
}

cublasSideMode_t hipblasConvertSide(hipblasSideMode_t side)
{
    switch(side)
    {
    case HIPBLAS_SIDE_LEFT:
        return CUBLAS_SIDE_LEFT;
    case HIPBLAS_SIDE_RIGHT:
        return CUBLAS_SIDE_RIGHT;
    default:
        throw HIPBLAS_STATUS_INVALID_ENUM;
    }
}

cublasPointerMode_t hipblasGetCublasPointerMode(hipblasPointerMode_t mode)
{
    switch(mode)
    {
    case HIPBLAS_POINTER_MODE_HOST:
        return CUBLAS_POINTER_MODE_HOST;

    case HIPBLAS_POINTER_MODE_DEVICE:
        return CUBLAS_POINTER_MODE_DEVICE;

    default:
        throw HIPBLAS_STATUS_INVALID_ENUM;
    }
}

hipblasPointerMode_t hipblasConvertPointerMode(cublasPointerMode_t mode)
{
    switch(mode)
    {
    case CUBLAS_POINTER_MODE_HOST:
        return HIPBLAS_POINTER_MODE_HOST;

    case CUBLAS_POINTER_MODE_DEVICE:
        return HIPBLAS_POINTER_MODE_DEVICE;

    default:
        throw HIPBLAS_STATUS_INVALID_ENUM;
    }
}

cudaDataType_t hipblasConvertDatatype(hipblasDatatype_t type)
{
    switch(type)
    {
    case HIPBLAS_R_16F:
        return CUDA_R_16F;

    case HIPBLAS_R_32F:
        return CUDA_R_32F;

    case HIPBLAS_R_64F:
        return CUDA_R_64F;

    case HIPBLAS_C_16F:
        return CUDA_C_16F;

    case HIPBLAS_C_32F:
        return CUDA_C_32F;

    case HIPBLAS_C_64F:
        return CUDA_C_64F;

    case HIPBLAS_R_8I:
        return CUDA_R_8I;

    case HIPBLAS_R_8U:
        return CUDA_R_8U;

    case HIPBLAS_R_32I:
        return CUDA_R_32I;

    case HIPBLAS_C_8I:
        return CUDA_C_8I;

    case HIPBLAS_C_8U:
        return CUDA_C_8U;

    case HIPBLAS_C_32I:
        return CUDA_C_32I;

    case HIPBLAS_R_16B:
        return CUDA_R_16BF;

    case HIPBLAS_C_16B:
        return CUDA_C_16BF;

    default:
        throw HIPBLAS_STATUS_INVALID_ENUM;
    }
}

cudaDataType_t hipblasConvertDatatype_v2(hipDataType type)
{
    switch(type)
    {
    case HIP_R_16F:
        return CUDA_R_16F;

    case HIP_R_32F:
        return CUDA_R_32F;

    case HIP_R_64F:
        return CUDA_R_64F;

    case HIP_C_16F:
        return CUDA_C_16F;

    case HIP_C_32F:
        return CUDA_C_32F;

    case HIP_C_64F:
        return CUDA_C_64F;

    case HIP_R_8I:
        return CUDA_R_8I;

    case HIP_R_8U:
        return CUDA_R_8U;

    case HIP_R_32I:
        return CUDA_R_32I;

    case HIP_C_8I:
        return CUDA_C_8I;

    case HIP_C_8U:
        return CUDA_C_8U;

    case HIP_C_32I:
        return CUDA_C_32I;

    case HIP_R_16BF:
        return CUDA_R_16BF;

    case HIP_C_16BF:
        return CUDA_C_16BF;

    default:
        throw HIPBLAS_STATUS_INVALID_ENUM;
    }
}

cublasComputeType_t hipblasConvertComputeType(hipblasComputeType_t type)
{
    switch(type)
    {
    case HIPBLAS_COMPUTE_16F:
        return CUBLAS_COMPUTE_16F;

    case HIPBLAS_COMPUTE_16F_PEDANTIC:
        return CUBLAS_COMPUTE_16F_PEDANTIC;

    case HIPBLAS_COMPUTE_32F:
        return CUBLAS_COMPUTE_32F;

    case HIPBLAS_COMPUTE_32F_PEDANTIC:
        return CUBLAS_COMPUTE_32F_PEDANTIC;

    case HIPBLAS_COMPUTE_32F_FAST_16F:
        return CUBLAS_COMPUTE_32F_FAST_16F;

    case HIPBLAS_COMPUTE_32F_FAST_16BF:
        return CUBLAS_COMPUTE_32F_FAST_16BF;

    case HIPBLAS_COMPUTE_32F_FAST_TF32:
        return CUBLAS_COMPUTE_32F_FAST_TF32;

    case HIPBLAS_COMPUTE_64F:
        return CUBLAS_COMPUTE_64F;

    case HIPBLAS_COMPUTE_64F_PEDANTIC:
        return CUBLAS_COMPUTE_64F_PEDANTIC;

    case HIPBLAS_COMPUTE_32I:
        return CUBLAS_COMPUTE_32I;

    case HIPBLAS_COMPUTE_32I_PEDANTIC:
        return CUBLAS_COMPUTE_32I_PEDANTIC;

    default:
        throw HIPBLAS_STATUS_INVALID_ENUM;
    }
}

cublasGemmAlgo_t hipblasConvertGemmAlgo(hipblasGemmAlgo_t algo)
{
    // Only support Default Algo for now
    switch(algo)
    {
    case HIPBLAS_GEMM_DEFAULT:
        return CUBLAS_GEMM_DEFAULT;

    default:
        throw HIPBLAS_STATUS_INVALID_ENUM;
    }
}

cublasAtomicsMode_t hipblasConvertAtomicsMode(hipblasAtomicsMode_t mode)

{
    switch(mode)
    {
    case HIPBLAS_ATOMICS_NOT_ALLOWED:
        return CUBLAS_ATOMICS_NOT_ALLOWED;
    case HIPBLAS_ATOMICS_ALLOWED:
        return CUBLAS_ATOMICS_ALLOWED;
    default:
        throw HIPBLAS_STATUS_INVALID_ENUM;
    }
}

cublasMath_t hipblasGetCublasMathMode(hipblasMath_t mode)
{
    switch(mode)
    {
    case HIPBLAS_DEFAULT_MATH:
        return CUBLAS_DEFAULT_MATH;
    case HIPBLAS_PEDANTIC_MATH:
        return CUBLAS_PEDANTIC_MATH;
    case HIPBLAS_TF32_TENSOR_OP_MATH:
        return CUBLAS_TF32_TENSOR_OP_MATH;
    case HIPBLAS_MATH_DISALLOW_REDUCED_PRECISION_REDUCTION:
        return CUBLAS_MATH_DISALLOW_REDUCED_PRECISION_REDUCTION;
    case HIPBLAS_TENSOR_OP_MATH:
        return CUBLAS_TENSOR_OP_MATH;
    }
    throw HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasMath_t hipblasConvertMathMode(cublasMath_t mode)
{
    switch(mode)
    {
    case CUBLAS_DEFAULT_MATH:
        return HIPBLAS_DEFAULT_MATH;
    case CUBLAS_PEDANTIC_MATH:
        return HIPBLAS_PEDANTIC_MATH;
    case CUBLAS_TF32_TENSOR_OP_MATH:
        return HIPBLAS_TF32_TENSOR_OP_MATH;
    case CUBLAS_MATH_DISALLOW_REDUCED_PRECISION_REDUCTION:
        return HIPBLAS_MATH_DISALLOW_REDUCED_PRECISION_REDUCTION;
    case CUBLAS_TENSOR_OP_MATH:
        return HIPBLAS_TENSOR_OP_MATH;
    }
    throw HIPBLAS_STATUS_INVALID_ENUM;
}

hipblasStatus_t hipblasConvertStatus(cublasStatus_t cuStatus)
{
    switch(cuStatus)
    {
    case CUBLAS_STATUS_SUCCESS:
        return HIPBLAS_STATUS_SUCCESS;
    case CUBLAS_STATUS_NOT_INITIALIZED:
        return HIPBLAS_STATUS_NOT_INITIALIZED;
    case CUBLAS_STATUS_ALLOC_FAILED:
        return HIPBLAS_STATUS_ALLOC_FAILED;
    case CUBLAS_STATUS_INVALID_VALUE:
        return HIPBLAS_STATUS_INVALID_VALUE;
    case CUBLAS_STATUS_MAPPING_ERROR:
        return HIPBLAS_STATUS_MAPPING_ERROR;
    case CUBLAS_STATUS_EXECUTION_FAILED:
        return HIPBLAS_STATUS_EXECUTION_FAILED;
    case CUBLAS_STATUS_INTERNAL_ERROR:
        return HIPBLAS_STATUS_INTERNAL_ERROR;
    case CUBLAS_STATUS_NOT_SUPPORTED:
        return HIPBLAS_STATUS_NOT_SUPPORTED;
    case CUBLAS_STATUS_ARCH_MISMATCH:
        return HIPBLAS_STATUS_ARCH_MISMATCH;
    default:
        return HIPBLAS_STATUS_UNKNOWN;
    }
}

hipblasStatus_t hipblasSetStream(hipblasHandle_t handle, hipStream_t streamId)
try
{
    return hipblasConvertStatus(cublasSetStream((cublasHandle_t)handle, streamId));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasGetStream(hipblasHandle_t handle, hipStream_t* streamId)
try
{
    return hipblasConvertStatus(cublasGetStream((cublasHandle_t)handle, streamId));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCreate(hipblasHandle_t* handle)
try
{
    return hipblasConvertStatus(cublasCreate((cublasHandle_t*)handle));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// TODO broke common API semantics, think about this again.
hipblasStatus_t hipblasDestroy(hipblasHandle_t handle)
try
{
    return hipblasConvertStatus(cublasDestroy((cublasHandle_t)handle));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasSetPointerMode(hipblasHandle_t handle, hipblasPointerMode_t mode)
try
{
    return hipblasConvertStatus(
        cublasSetPointerMode((cublasHandle_t)handle, hipblasGetCublasPointerMode(mode)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasGetPointerMode(hipblasHandle_t handle, hipblasPointerMode_t* mode)
try
{
    cublasPointerMode_t cublasMode;
    cublasStatus_t      status = cublasGetPointerMode((cublasHandle_t)handle, &cublasMode);
    *mode                      = hipblasConvertPointerMode(cublasMode);
    return hipblasConvertStatus(status);
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasSetMathMode(hipblasHandle_t handle, hipblasMath_t mode)
try
{
    return hipblasConvertStatus(
        cublasSetMathMode((cublasHandle_t)handle, hipblasGetCublasMathMode(mode)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasGetMathMode(hipblasHandle_t handle, hipblasMath_t* mode)
try
{
    cublasMath_t   cublasMode;
    cublasStatus_t status = cublasGetMathMode((cublasHandle_t)handle, &cublasMode);
    *mode                 = hipblasConvertMathMode(cublasMode);
    return hipblasConvertStatus(status);
}
catch(...)
{
    return hipblas_exception_to_status();
}

// note: no handle
hipblasStatus_t hipblasSetVector(int n, int elemSize, const void* x, int incx, void* y, int incy)
try
{
    return hipblasConvertStatus(
        cublasSetVector(n, elemSize, x, incx, y, incy)); // HGSOS no need for handle
}
catch(...)
{
    return hipblas_exception_to_status();
}

// note: no handle
hipblasStatus_t hipblasGetVector(int n, int elemSize, const void* x, int incx, void* y, int incy)
try
{
    return hipblasConvertStatus(
        cublasGetVector(n, elemSize, x, incx, y, incy)); // HGSOS no need for handle
}
catch(...)
{
    return hipblas_exception_to_status();
}

// note: no handle
hipblasStatus_t
    hipblasSetMatrix(int rows, int cols, int elemSize, const void* A, int lda, void* B, int ldb)
try
{
    return hipblasConvertStatus(cublasSetMatrix(rows, cols, elemSize, A, lda, B, ldb));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// note: no handle
hipblasStatus_t
    hipblasGetMatrix(int rows, int cols, int elemSize, const void* A, int lda, void* B, int ldb)
try
{
    return hipblasConvertStatus(cublasGetMatrix(rows, cols, elemSize, A, lda, B, ldb));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasSetVectorAsync(
    int n, int elemSize, const void* x, int incx, void* y, int incy, hipStream_t stream)
try
{
    return hipblasConvertStatus(cublasSetVectorAsync(n, elemSize, x, incx, y, incy, stream));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasGetVectorAsync(
    int n, int elemSize, const void* x, int incx, void* y, int incy, hipStream_t stream)
try
{
    return hipblasConvertStatus(cublasGetVectorAsync(n, elemSize, x, incx, y, incy, stream));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasSetMatrixAsync(
    int rows, int cols, int elemSize, const void* A, int lda, void* B, int ldb, hipStream_t stream)
try
{
    return hipblasConvertStatus(cublasSetMatrixAsync(rows, cols, elemSize, A, lda, B, ldb, stream));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasGetMatrixAsync(
    int rows, int cols, int elemSize, const void* A, int lda, void* B, int ldb, hipStream_t stream)
try
{
    return hipblasConvertStatus(cublasGetMatrixAsync(rows, cols, elemSize, A, lda, B, ldb, stream));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// atomics mode
hipblasStatus_t hipblasSetAtomicsMode(hipblasHandle_t handle, hipblasAtomicsMode_t atomics_mode)
try
{
    return hipblasConvertStatus(
        cublasSetAtomicsMode((cublasHandle_t)handle, hipblasConvertAtomicsMode(atomics_mode)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasGetAtomicsMode(hipblasHandle_t handle, hipblasAtomicsMode_t* atomics_mode)
try
{
    return hipblasConvertStatus(
        cublasGetAtomicsMode((cublasHandle_t)handle, (cublasAtomicsMode_t*)atomics_mode));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// amax
hipblasStatus_t hipblasIsamax(hipblasHandle_t handle, int n, const float* x, int incx, int* result)
try
{
    return hipblasConvertStatus(cublasIsamax((cublasHandle_t)handle, n, x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIdamax(hipblasHandle_t handle, int n, const double* x, int incx, int* result)
try
{
    return hipblasConvertStatus(cublasIdamax((cublasHandle_t)handle, n, x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasIcamax(hipblasHandle_t handle, int n, const hipblasComplex* x, int incx, int* result)
try
{
    return hipblasConvertStatus(
        cublasIcamax((cublasHandle_t)handle, n, (cuComplex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIzamax(
    hipblasHandle_t handle, int n, const hipblasDoubleComplex* x, int incx, int* result)
try
{
    return hipblasConvertStatus(
        cublasIzamax((cublasHandle_t)handle, n, (cuDoubleComplex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasIcamax_v2(hipblasHandle_t handle, int n, const hipComplex* x, int incx, int* result)
try
{
    return hipblasConvertStatus(
        cublasIcamax((cublasHandle_t)handle, n, (cuComplex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIzamax_v2(
    hipblasHandle_t handle, int n, const hipDoubleComplex* x, int incx, int* result)
try
{
    return hipblasConvertStatus(
        cublasIzamax((cublasHandle_t)handle, n, (cuDoubleComplex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// amax_64
hipblasStatus_t hipblasIsamax_64(
    hipblasHandle_t handle, int64_t n, const float* x, int64_t incx, int64_t* result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasIsamax_64((cublasHandle_t)handle, n, x, incx, result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIdamax_64(
    hipblasHandle_t handle, int64_t n, const double* x, int64_t incx, int64_t* result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasIdamax_64((cublasHandle_t)handle, n, x, incx, result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIcamax_64(
    hipblasHandle_t handle, int64_t n, const hipblasComplex* x, int64_t incx, int64_t* result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasIcamax_64((cublasHandle_t)handle, n, (cuComplex*)x, incx, result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIzamax_64(
    hipblasHandle_t handle, int64_t n, const hipblasDoubleComplex* x, int64_t incx, int64_t* result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasIzamax_64((cublasHandle_t)handle, n, (cuDoubleComplex*)x, incx, result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIcamax_v2_64(
    hipblasHandle_t handle, int64_t n, const hipComplex* x, int64_t incx, int64_t* result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasIcamax_64((cublasHandle_t)handle, n, (cuComplex*)x, incx, result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIzamax_v2_64(
    hipblasHandle_t handle, int64_t n, const hipDoubleComplex* x, int64_t incx, int64_t* result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasIzamax_64((cublasHandle_t)handle, n, (cuDoubleComplex*)x, incx, result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// amax_batched
hipblasStatus_t hipblasIsamaxBatched(
    hipblasHandle_t handle, int n, const float* const x[], int incx, int batchCount, int* result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIdamaxBatched(
    hipblasHandle_t handle, int n, const double* const x[], int incx, int batchCount, int* result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIcamaxBatched(hipblasHandle_t             handle,
                                     int                         n,
                                     const hipblasComplex* const x[],
                                     int                         incx,
                                     int                         batchCount,
                                     int*                        result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIzamaxBatched(hipblasHandle_t                   handle,
                                     int                               n,
                                     const hipblasDoubleComplex* const x[],
                                     int                               incx,
                                     int                               batchCount,
                                     int*                              result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIcamaxBatched_v2(hipblasHandle_t         handle,
                                        int                     n,
                                        const hipComplex* const x[],
                                        int                     incx,
                                        int                     batchCount,
                                        int*                    result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIzamaxBatched_v2(hipblasHandle_t               handle,
                                        int                           n,
                                        const hipDoubleComplex* const x[],
                                        int                           incx,
                                        int                           batchCount,
                                        int*                          result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// amax_batched_64
hipblasStatus_t hipblasIsamaxBatched_64(hipblasHandle_t    handle,
                                        int64_t            n,
                                        const float* const x[],
                                        int64_t            incx,
                                        int64_t            batchCount,
                                        int64_t*           result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIdamaxBatched_64(hipblasHandle_t     handle,
                                        int64_t             n,
                                        const double* const x[],
                                        int64_t             incx,
                                        int64_t             batchCount,
                                        int64_t*            result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIcamaxBatched_64(hipblasHandle_t             handle,
                                        int64_t                     n,
                                        const hipblasComplex* const x[],
                                        int64_t                     incx,
                                        int64_t                     batchCount,
                                        int64_t*                    result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIzamaxBatched_64(hipblasHandle_t                   handle,
                                        int64_t                           n,
                                        const hipblasDoubleComplex* const x[],
                                        int64_t                           incx,
                                        int64_t                           batchCount,
                                        int64_t*                          result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIcamaxBatched_v2_64(hipblasHandle_t         handle,
                                           int64_t                 n,
                                           const hipComplex* const x[],
                                           int64_t                 incx,
                                           int64_t                 batchCount,
                                           int64_t*                result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIzamaxBatched_v2_64(hipblasHandle_t               handle,
                                           int64_t                       n,
                                           const hipDoubleComplex* const x[],
                                           int64_t                       incx,
                                           int64_t                       batchCount,
                                           int64_t*                      result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// amax_strided_batched
hipblasStatus_t hipblasIsamaxStridedBatched(hipblasHandle_t handle,
                                            int             n,
                                            const float*    x,
                                            int             incx,
                                            hipblasStride   stridex,
                                            int             batchCount,
                                            int*            result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIdamaxStridedBatched(hipblasHandle_t handle,
                                            int             n,
                                            const double*   x,
                                            int             incx,
                                            hipblasStride   stridex,
                                            int             batchCount,
                                            int*            result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIcamaxStridedBatched(hipblasHandle_t       handle,
                                            int                   n,
                                            const hipblasComplex* x,
                                            int                   incx,
                                            hipblasStride         stridex,
                                            int                   batchCount,
                                            int*                  result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIzamaxStridedBatched(hipblasHandle_t             handle,
                                            int                         n,
                                            const hipblasDoubleComplex* x,
                                            int                         incx,
                                            hipblasStride               stridex,
                                            int                         batchCount,
                                            int*                        result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIcamaxStridedBatched_v2(hipblasHandle_t   handle,
                                               int               n,
                                               const hipComplex* x,
                                               int               incx,
                                               hipblasStride     stridex,
                                               int               batchCount,
                                               int*              result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIzamaxStridedBatched_v2(hipblasHandle_t         handle,
                                               int                     n,
                                               const hipDoubleComplex* x,
                                               int                     incx,
                                               hipblasStride           stridex,
                                               int                     batchCount,
                                               int*                    result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// amax_strided_batched_64
hipblasStatus_t hipblasIsamaxStridedBatched_64(hipblasHandle_t handle,
                                               int64_t         n,
                                               const float*    x,
                                               int64_t         incx,
                                               hipblasStride   stridex,
                                               int64_t         batchCount,
                                               int64_t*        result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIdamaxStridedBatched_64(hipblasHandle_t handle,
                                               int64_t         n,
                                               const double*   x,
                                               int64_t         incx,
                                               hipblasStride   stridex,
                                               int64_t         batchCount,
                                               int64_t*        result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIcamaxStridedBatched_64(hipblasHandle_t       handle,
                                               int64_t               n,
                                               const hipblasComplex* x,
                                               int64_t               incx,
                                               hipblasStride         stridex,
                                               int64_t               batchCount,
                                               int64_t*              result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIzamaxStridedBatched_64(hipblasHandle_t             handle,
                                               int64_t                     n,
                                               const hipblasDoubleComplex* x,
                                               int64_t                     incx,
                                               hipblasStride               stridex,
                                               int64_t                     batchCount,
                                               int64_t*                    result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIcamaxStridedBatched_v2_64(hipblasHandle_t   handle,
                                                  int64_t           n,
                                                  const hipComplex* x,
                                                  int64_t           incx,
                                                  hipblasStride     stridex,
                                                  int64_t           batchCount,
                                                  int64_t*          result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIzamaxStridedBatched_v2_64(hipblasHandle_t         handle,
                                                  int64_t                 n,
                                                  const hipDoubleComplex* x,
                                                  int64_t                 incx,
                                                  hipblasStride           stridex,
                                                  int64_t                 batchCount,
                                                  int64_t*                result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// amin
hipblasStatus_t hipblasIsamin(hipblasHandle_t handle, int n, const float* x, int incx, int* result)
try
{
    return hipblasConvertStatus(cublasIsamin((cublasHandle_t)handle, n, x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIdamin(hipblasHandle_t handle, int n, const double* x, int incx, int* result)
try
{
    return hipblasConvertStatus(cublasIdamin((cublasHandle_t)handle, n, x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasIcamin(hipblasHandle_t handle, int n, const hipblasComplex* x, int incx, int* result)
try
{
    return hipblasConvertStatus(
        cublasIcamin((cublasHandle_t)handle, n, (cuComplex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIzamin(
    hipblasHandle_t handle, int n, const hipblasDoubleComplex* x, int incx, int* result)
try
{
    return hipblasConvertStatus(
        cublasIzamin((cublasHandle_t)handle, n, (cuDoubleComplex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasIcamin_v2(hipblasHandle_t handle, int n, const hipComplex* x, int incx, int* result)
try
{
    return hipblasConvertStatus(
        cublasIcamin((cublasHandle_t)handle, n, (cuComplex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIzamin_v2(
    hipblasHandle_t handle, int n, const hipDoubleComplex* x, int incx, int* result)
try
{
    return hipblasConvertStatus(
        cublasIzamin((cublasHandle_t)handle, n, (cuDoubleComplex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// amin_64
hipblasStatus_t hipblasIsamin_64(
    hipblasHandle_t handle, int64_t n, const float* x, int64_t incx, int64_t* result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasIsamin_64((cublasHandle_t)handle, n, x, incx, result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIdamin_64(
    hipblasHandle_t handle, int64_t n, const double* x, int64_t incx, int64_t* result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasIdamin_64((cublasHandle_t)handle, n, x, incx, result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIcamin_64(
    hipblasHandle_t handle, int64_t n, const hipblasComplex* x, int64_t incx, int64_t* result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasIcamin_64((cublasHandle_t)handle, n, (cuComplex*)x, incx, result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIzamin_64(
    hipblasHandle_t handle, int64_t n, const hipblasDoubleComplex* x, int64_t incx, int64_t* result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasIzamin_64((cublasHandle_t)handle, n, (cuDoubleComplex*)x, incx, result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIcamin_v2_64(
    hipblasHandle_t handle, int64_t n, const hipComplex* x, int64_t incx, int64_t* result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasIcamin_64((cublasHandle_t)handle, n, (cuComplex*)x, incx, result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIzamin_v2_64(
    hipblasHandle_t handle, int64_t n, const hipDoubleComplex* x, int64_t incx, int64_t* result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasIzamin_64((cublasHandle_t)handle, n, (cuDoubleComplex*)x, incx, result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// amin_batched
hipblasStatus_t hipblasIsaminBatched(
    hipblasHandle_t handle, int n, const float* const x[], int incx, int batchCount, int* result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIdaminBatched(
    hipblasHandle_t handle, int n, const double* const x[], int incx, int batchCount, int* result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIcaminBatched(hipblasHandle_t             handle,
                                     int                         n,
                                     const hipblasComplex* const x[],
                                     int                         incx,
                                     int                         batchCount,
                                     int*                        result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIzaminBatched(hipblasHandle_t                   handle,
                                     int                               n,
                                     const hipblasDoubleComplex* const x[],
                                     int                               incx,
                                     int                               batchCount,
                                     int*                              result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIcaminBatched_v2(hipblasHandle_t         handle,
                                        int                     n,
                                        const hipComplex* const x[],
                                        int                     incx,
                                        int                     batchCount,
                                        int*                    result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIzaminBatched_v2(hipblasHandle_t               handle,
                                        int                           n,
                                        const hipDoubleComplex* const x[],
                                        int                           incx,
                                        int                           batchCount,
                                        int*                          result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// amin_batched_64
hipblasStatus_t hipblasIsaminBatched_64(hipblasHandle_t    handle,
                                        int64_t            n,
                                        const float* const x[],
                                        int64_t            incx,
                                        int64_t            batchCount,
                                        int64_t*           result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIdaminBatched_64(hipblasHandle_t     handle,
                                        int64_t             n,
                                        const double* const x[],
                                        int64_t             incx,
                                        int64_t             batchCount,
                                        int64_t*            result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIcaminBatched_64(hipblasHandle_t             handle,
                                        int64_t                     n,
                                        const hipblasComplex* const x[],
                                        int64_t                     incx,
                                        int64_t                     batchCount,
                                        int64_t*                    result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIzaminBatched_64(hipblasHandle_t                   handle,
                                        int64_t                           n,
                                        const hipblasDoubleComplex* const x[],
                                        int64_t                           incx,
                                        int64_t                           batchCount,
                                        int64_t*                          result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIcaminBatched_v2_64(hipblasHandle_t         handle,
                                           int64_t                 n,
                                           const hipComplex* const x[],
                                           int64_t                 incx,
                                           int64_t                 batchCount,
                                           int64_t*                result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIzaminBatched_v2_64(hipblasHandle_t               handle,
                                           int64_t                       n,
                                           const hipDoubleComplex* const x[],
                                           int64_t                       incx,
                                           int64_t                       batchCount,
                                           int64_t*                      result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// amin_strided_batched
hipblasStatus_t hipblasIsaminStridedBatched(hipblasHandle_t handle,
                                            int             n,
                                            const float*    x,
                                            int             incx,
                                            hipblasStride   stridex,
                                            int             batchCount,
                                            int*            result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIdaminStridedBatched(hipblasHandle_t handle,
                                            int             n,
                                            const double*   x,
                                            int             incx,
                                            hipblasStride   stridex,
                                            int             batchCount,
                                            int*            result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIcaminStridedBatched(hipblasHandle_t       handle,
                                            int                   n,
                                            const hipblasComplex* x,
                                            int                   incx,
                                            hipblasStride         stridex,
                                            int                   batchCount,
                                            int*                  result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIzaminStridedBatched(hipblasHandle_t             handle,
                                            int                         n,
                                            const hipblasDoubleComplex* x,
                                            int                         incx,
                                            hipblasStride               stridex,
                                            int                         batchCount,
                                            int*                        result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIcaminStridedBatched_v2(hipblasHandle_t   handle,
                                               int               n,
                                               const hipComplex* x,
                                               int               incx,
                                               hipblasStride     stridex,
                                               int               batchCount,
                                               int*              result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIzaminStridedBatched_v2(hipblasHandle_t         handle,
                                               int                     n,
                                               const hipDoubleComplex* x,
                                               int                     incx,
                                               hipblasStride           stridex,
                                               int                     batchCount,
                                               int*                    result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// amin_strided_batched_64
hipblasStatus_t hipblasIsaminStridedBatched_64(hipblasHandle_t handle,
                                               int64_t         n,
                                               const float*    x,
                                               int64_t         incx,
                                               hipblasStride   stridex,
                                               int64_t         batchCount,
                                               int64_t*        result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIdaminStridedBatched_64(hipblasHandle_t handle,
                                               int64_t         n,
                                               const double*   x,
                                               int64_t         incx,
                                               hipblasStride   stridex,
                                               int64_t         batchCount,
                                               int64_t*        result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIcaminStridedBatched_64(hipblasHandle_t       handle,
                                               int64_t               n,
                                               const hipblasComplex* x,
                                               int64_t               incx,
                                               hipblasStride         stridex,
                                               int64_t               batchCount,
                                               int64_t*              result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIzaminStridedBatched_64(hipblasHandle_t             handle,
                                               int64_t                     n,
                                               const hipblasDoubleComplex* x,
                                               int64_t                     incx,
                                               hipblasStride               stridex,
                                               int64_t                     batchCount,
                                               int64_t*                    result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIcaminStridedBatched_v2_64(hipblasHandle_t   handle,
                                                  int64_t           n,
                                                  const hipComplex* x,
                                                  int64_t           incx,
                                                  hipblasStride     stridex,
                                                  int64_t           batchCount,
                                                  int64_t*          result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasIzaminStridedBatched_v2_64(hipblasHandle_t         handle,
                                                  int64_t                 n,
                                                  const hipDoubleComplex* x,
                                                  int64_t                 incx,
                                                  hipblasStride           stridex,
                                                  int64_t                 batchCount,
                                                  int64_t*                result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// asum
hipblasStatus_t hipblasSasum(hipblasHandle_t handle, int n, const float* x, int incx, float* result)
try
{
    return hipblasConvertStatus(cublasSasum((cublasHandle_t)handle, n, x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasDasum(hipblasHandle_t handle, int n, const double* x, int incx, double* result)
try
{
    return hipblasConvertStatus(cublasDasum((cublasHandle_t)handle, n, x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasScasum(hipblasHandle_t handle, int n, const hipblasComplex* x, int incx, float* result)
try
{
    return hipblasConvertStatus(
        cublasScasum((cublasHandle_t)handle, n, (cuComplex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDzasum(
    hipblasHandle_t handle, int n, const hipblasDoubleComplex* x, int incx, double* result)
try
{
    return hipblasConvertStatus(
        cublasDzasum((cublasHandle_t)handle, n, (cuDoubleComplex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasScasum_v2(hipblasHandle_t handle, int n, const hipComplex* x, int incx, float* result)
try
{
    return hipblasConvertStatus(
        cublasScasum((cublasHandle_t)handle, n, (cuComplex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDzasum_v2(
    hipblasHandle_t handle, int n, const hipDoubleComplex* x, int incx, double* result)
try
{
    return hipblasConvertStatus(
        cublasDzasum((cublasHandle_t)handle, n, (cuDoubleComplex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// asum_64
hipblasStatus_t
    hipblasSasum_64(hipblasHandle_t handle, int64_t n, const float* x, int64_t incx, float* result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasSasum_64((cublasHandle_t)handle, n, x, incx, result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDasum_64(
    hipblasHandle_t handle, int64_t n, const double* x, int64_t incx, double* result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasDasum_64((cublasHandle_t)handle, n, x, incx, result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasScasum_64(
    hipblasHandle_t handle, int64_t n, const hipblasComplex* x, int64_t incx, float* result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasScasum_64((cublasHandle_t)handle, n, (cuComplex*)x, incx, result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDzasum_64(
    hipblasHandle_t handle, int64_t n, const hipblasDoubleComplex* x, int64_t incx, double* result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasDzasum_64((cublasHandle_t)handle, n, (cuDoubleComplex*)x, incx, result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasScasum_v2_64(
    hipblasHandle_t handle, int64_t n, const hipComplex* x, int64_t incx, float* result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasScasum_64((cublasHandle_t)handle, n, (cuComplex*)x, incx, result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDzasum_v2_64(
    hipblasHandle_t handle, int64_t n, const hipDoubleComplex* x, int64_t incx, double* result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasDzasum_64((cublasHandle_t)handle, n, (cuDoubleComplex*)x, incx, result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// asum_batched
hipblasStatus_t hipblasSasumBatched(
    hipblasHandle_t handle, int n, const float* const x[], int incx, int batchCount, float* result)
try
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
    // // TODO warn user that function was demoted to ignore batch
    // return hipblasConvertStatus(cublasSasum((cublasHandle_t)handle, n, x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDasumBatched(hipblasHandle_t     handle,
                                    int                 n,
                                    const double* const x[],
                                    int                 incx,
                                    int                 batchCount,
                                    double*             result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasScasumBatched(hipblasHandle_t             handle,
                                     int                         n,
                                     const hipblasComplex* const x[],
                                     int                         incx,
                                     int                         batchCount,
                                     float*                      result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDzasumBatched(hipblasHandle_t                   handle,
                                     int                               n,
                                     const hipblasDoubleComplex* const x[],
                                     int                               incx,
                                     int                               batchCount,
                                     double*                           result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasScasumBatched_v2(hipblasHandle_t         handle,
                                        int                     n,
                                        const hipComplex* const x[],
                                        int                     incx,
                                        int                     batchCount,
                                        float*                  result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDzasumBatched_v2(hipblasHandle_t               handle,
                                        int                           n,
                                        const hipDoubleComplex* const x[],
                                        int                           incx,
                                        int                           batchCount,
                                        double*                       result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// asum_batched_64
hipblasStatus_t hipblasSasumBatched_64(hipblasHandle_t    handle,
                                       int64_t            n,
                                       const float* const x[],
                                       int64_t            incx,
                                       int64_t            batchCount,
                                       float*             result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDasumBatched_64(hipblasHandle_t     handle,
                                       int64_t             n,
                                       const double* const x[],
                                       int64_t             incx,
                                       int64_t             batchCount,
                                       double*             result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasScasumBatched_64(hipblasHandle_t             handle,
                                        int64_t                     n,
                                        const hipblasComplex* const x[],
                                        int64_t                     incx,
                                        int64_t                     batchCount,
                                        float*                      result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDzasumBatched_64(hipblasHandle_t                   handle,
                                        int64_t                           n,
                                        const hipblasDoubleComplex* const x[],
                                        int64_t                           incx,
                                        int64_t                           batchCount,
                                        double*                           result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasScasumBatched_v2_64(hipblasHandle_t         handle,
                                           int64_t                 n,
                                           const hipComplex* const x[],
                                           int64_t                 incx,
                                           int64_t                 batchCount,
                                           float*                  result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDzasumBatched_v2_64(hipblasHandle_t               handle,
                                           int64_t                       n,
                                           const hipDoubleComplex* const x[],
                                           int64_t                       incx,
                                           int64_t                       batchCount,
                                           double*                       result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// asum_strided_batched
hipblasStatus_t hipblasSasumStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const float*    x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           int             batchCount,
                                           float*          result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDasumStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const double*   x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           int             batchCount,
                                           double*         result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasScasumStridedBatched(hipblasHandle_t       handle,
                                            int                   n,
                                            const hipblasComplex* x,
                                            int                   incx,
                                            hipblasStride         stridex,
                                            int                   batchCount,
                                            float*                result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDzasumStridedBatched(hipblasHandle_t             handle,
                                            int                         n,
                                            const hipblasDoubleComplex* x,
                                            int                         incx,
                                            hipblasStride               stridex,
                                            int                         batchCount,
                                            double*                     result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasScasumStridedBatched_v2(hipblasHandle_t   handle,
                                               int               n,
                                               const hipComplex* x,
                                               int               incx,
                                               hipblasStride     stridex,
                                               int               batchCount,
                                               float*            result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDzasumStridedBatched_v2(hipblasHandle_t         handle,
                                               int                     n,
                                               const hipDoubleComplex* x,
                                               int                     incx,
                                               hipblasStride           stridex,
                                               int                     batchCount,
                                               double*                 result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// asum_strided_batched_64
hipblasStatus_t hipblasSasumStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const float*    x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              int64_t         batchCount,
                                              float*          result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDasumStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const double*   x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              int64_t         batchCount,
                                              double*         result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasScasumStridedBatched_64(hipblasHandle_t       handle,
                                               int64_t               n,
                                               const hipblasComplex* x,
                                               int64_t               incx,
                                               hipblasStride         stridex,
                                               int64_t               batchCount,
                                               float*                result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDzasumStridedBatched_64(hipblasHandle_t             handle,
                                               int64_t                     n,
                                               const hipblasDoubleComplex* x,
                                               int64_t                     incx,
                                               hipblasStride               stridex,
                                               int64_t                     batchCount,
                                               double*                     result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasScasumStridedBatched_v2_64(hipblasHandle_t   handle,
                                                  int64_t           n,
                                                  const hipComplex* x,
                                                  int64_t           incx,
                                                  hipblasStride     stridex,
                                                  int64_t           batchCount,
                                                  float*            result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDzasumStridedBatched_v2_64(hipblasHandle_t         handle,
                                                  int64_t                 n,
                                                  const hipDoubleComplex* x,
                                                  int64_t                 incx,
                                                  hipblasStride           stridex,
                                                  int64_t                 batchCount,
                                                  double*                 result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// axpy
hipblasStatus_t hipblasHaxpy(hipblasHandle_t    handle,
                             int                n,
                             const hipblasHalf* alpha,
                             const hipblasHalf* x,
                             int                incx,
                             hipblasHalf*       y,
                             int                incy)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasSaxpy(
    hipblasHandle_t handle, int n, const float* alpha, const float* x, int incx, float* y, int incy)
try
{
    return hipblasConvertStatus(cublasSaxpy((cublasHandle_t)handle, n, alpha, x, incx, y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDaxpy(hipblasHandle_t handle,
                             int             n,
                             const double*   alpha,
                             const double*   x,
                             int             incx,
                             double*         y,
                             int             incy)
try
{
    return hipblasConvertStatus(cublasDaxpy((cublasHandle_t)handle, n, alpha, x, incx, y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCaxpy(hipblasHandle_t       handle,
                             int                   n,
                             const hipblasComplex* alpha,
                             const hipblasComplex* x,
                             int                   incx,
                             hipblasComplex*       y,
                             int                   incy)
try
{
    return hipblasConvertStatus(cublasCaxpy(
        (cublasHandle_t)handle, n, (cuComplex*)alpha, (cuComplex*)x, incx, (cuComplex*)y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZaxpy(hipblasHandle_t             handle,
                             int                         n,
                             const hipblasDoubleComplex* alpha,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             hipblasDoubleComplex*       y,
                             int                         incy)
try
{
    return hipblasConvertStatus(cublasZaxpy((cublasHandle_t)handle,
                                            n,
                                            (cuDoubleComplex*)alpha,
                                            (cuDoubleComplex*)x,
                                            incx,
                                            (cuDoubleComplex*)y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCaxpy_v2(hipblasHandle_t   handle,
                                int               n,
                                const hipComplex* alpha,
                                const hipComplex* x,
                                int               incx,
                                hipComplex*       y,
                                int               incy)
try
{
    return hipblasConvertStatus(cublasCaxpy(
        (cublasHandle_t)handle, n, (cuComplex*)alpha, (cuComplex*)x, incx, (cuComplex*)y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZaxpy_v2(hipblasHandle_t         handle,
                                int                     n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* x,
                                int                     incx,
                                hipDoubleComplex*       y,
                                int                     incy)
try
{
    return hipblasConvertStatus(cublasZaxpy((cublasHandle_t)handle,
                                            n,
                                            (cuDoubleComplex*)alpha,
                                            (cuDoubleComplex*)x,
                                            incx,
                                            (cuDoubleComplex*)y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// 64-bit interface
hipblasStatus_t hipblasHaxpy_64(hipblasHandle_t    handle,
                                int64_t            n,
                                const hipblasHalf* alpha,
                                const hipblasHalf* x,
                                int64_t            incx,
                                hipblasHalf*       y,
                                int64_t            incy)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasSaxpy_64(hipblasHandle_t handle,
                                int64_t         n,
                                const float*    alpha,
                                const float*    x,
                                int64_t         incx,
                                float*          y,
                                int64_t         incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasSaxpy_64((cublasHandle_t)handle, n, alpha, x, incx, y, incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDaxpy_64(hipblasHandle_t handle,
                                int64_t         n,
                                const double*   alpha,
                                const double*   x,
                                int64_t         incx,
                                double*         y,
                                int64_t         incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasDaxpy_64((cublasHandle_t)handle, n, alpha, x, incx, y, incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCaxpy_64(hipblasHandle_t       handle,
                                int64_t               n,
                                const hipblasComplex* alpha,
                                const hipblasComplex* x,
                                int64_t               incx,
                                hipblasComplex*       y,
                                int64_t               incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCaxpy_64(
        (cublasHandle_t)handle, n, (cuComplex*)alpha, (cuComplex*)x, incx, (cuComplex*)y, incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZaxpy_64(hipblasHandle_t             handle,
                                int64_t                     n,
                                const hipblasDoubleComplex* alpha,
                                const hipblasDoubleComplex* x,
                                int64_t                     incx,
                                hipblasDoubleComplex*       y,
                                int64_t                     incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZaxpy_64((cublasHandle_t)handle,
                                               n,
                                               (cuDoubleComplex*)alpha,
                                               (cuDoubleComplex*)x,
                                               incx,
                                               (cuDoubleComplex*)y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCaxpy_v2_64(hipblasHandle_t   handle,
                                   int64_t           n,
                                   const hipComplex* alpha,
                                   const hipComplex* x,
                                   int64_t           incx,
                                   hipComplex*       y,
                                   int64_t           incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCaxpy_64(
        (cublasHandle_t)handle, n, (cuComplex*)alpha, (cuComplex*)x, incx, (cuComplex*)y, incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZaxpy_v2_64(hipblasHandle_t         handle,
                                   int64_t                 n,
                                   const hipDoubleComplex* alpha,
                                   const hipDoubleComplex* x,
                                   int64_t                 incx,
                                   hipDoubleComplex*       y,
                                   int64_t                 incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZaxpy_64((cublasHandle_t)handle,
                                               n,
                                               (cuDoubleComplex*)alpha,
                                               (cuDoubleComplex*)x,
                                               incx,
                                               (cuDoubleComplex*)y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// axpy_batched
hipblasStatus_t hipblasHaxpyBatched(hipblasHandle_t          handle,
                                    int                      n,
                                    const hipblasHalf*       alpha,
                                    const hipblasHalf* const x[],
                                    int                      incx,
                                    hipblasHalf* const       y[],
                                    int                      incy,
                                    int                      batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasSaxpyBatched(hipblasHandle_t    handle,
                                    int                n,
                                    const float*       alpha,
                                    const float* const x[],
                                    int                incx,
                                    float* const       y[],
                                    int                incy,
                                    int                batchCount)
try
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDaxpyBatched(hipblasHandle_t     handle,
                                    int                 n,
                                    const double*       alpha,
                                    const double* const x[],
                                    int                 incx,
                                    double* const       y[],
                                    int                 incy,
                                    int                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCaxpyBatched(hipblasHandle_t             handle,
                                    int                         n,
                                    const hipblasComplex*       alpha,
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    hipblasComplex* const       y[],
                                    int                         incy,
                                    int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZaxpyBatched(hipblasHandle_t                   handle,
                                    int                               n,
                                    const hipblasDoubleComplex*       alpha,
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    hipblasDoubleComplex* const       y[],
                                    int                               incy,
                                    int                               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCaxpyBatched_v2(hipblasHandle_t         handle,
                                       int                     n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const x[],
                                       int                     incx,
                                       hipComplex* const       y[],
                                       int                     incy,
                                       int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZaxpyBatched_v2(hipblasHandle_t               handle,
                                       int                           n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       hipDoubleComplex* const       y[],
                                       int                           incy,
                                       int                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// 64-bit interface
hipblasStatus_t hipblasHaxpyBatched_64(hipblasHandle_t          handle,
                                       int64_t                  n,
                                       const hipblasHalf*       alpha,
                                       const hipblasHalf* const x[],
                                       int64_t                  incx,
                                       hipblasHalf* const       y[],
                                       int64_t                  incy,
                                       int64_t                  batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasSaxpyBatched_64(hipblasHandle_t    handle,
                                       int64_t            n,
                                       const float*       alpha,
                                       const float* const x[],
                                       int64_t            incx,
                                       float* const       y[],
                                       int64_t            incy,
                                       int64_t            batchCount)
try
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDaxpyBatched_64(hipblasHandle_t     handle,
                                       int64_t             n,
                                       const double*       alpha,
                                       const double* const x[],
                                       int64_t             incx,
                                       double* const       y[],
                                       int64_t             incy,
                                       int64_t             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCaxpyBatched_64(hipblasHandle_t             handle,
                                       int64_t                     n,
                                       const hipblasComplex*       alpha,
                                       const hipblasComplex* const x[],
                                       int64_t                     incx,
                                       hipblasComplex* const       y[],
                                       int64_t                     incy,
                                       int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZaxpyBatched_64(hipblasHandle_t                   handle,
                                       int64_t                           n,
                                       const hipblasDoubleComplex*       alpha,
                                       const hipblasDoubleComplex* const x[],
                                       int64_t                           incx,
                                       hipblasDoubleComplex* const       y[],
                                       int64_t                           incy,
                                       int64_t                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCaxpyBatched_v2_64(hipblasHandle_t         handle,
                                          int64_t                 n,
                                          const hipComplex*       alpha,
                                          const hipComplex* const x[],
                                          int64_t                 incx,
                                          hipComplex* const       y[],
                                          int64_t                 incy,
                                          int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZaxpyBatched_v2_64(hipblasHandle_t               handle,
                                          int64_t                       n,
                                          const hipDoubleComplex*       alpha,
                                          const hipDoubleComplex* const x[],
                                          int64_t                       incx,
                                          hipDoubleComplex* const       y[],
                                          int64_t                       incy,
                                          int64_t                       batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// axpy_strided_batched
hipblasStatus_t hipblasHaxpyStridedBatched(hipblasHandle_t    handle,
                                           int                n,
                                           const hipblasHalf* alpha,
                                           const hipblasHalf* x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           hipblasHalf*       y,
                                           int                incy,
                                           hipblasStride      stridey,
                                           int                batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasSaxpyStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const float*    alpha,
                                           const float*    x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           float*          y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDaxpyStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const double*   alpha,
                                           const double*   x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           double*         y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCaxpyStridedBatched(hipblasHandle_t       handle,
                                           int                   n,
                                           const hipblasComplex* alpha,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           hipblasComplex*       y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZaxpyStridedBatched(hipblasHandle_t             handle,
                                           int                         n,
                                           const hipblasDoubleComplex* alpha,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           hipblasDoubleComplex*       y,
                                           int                         incy,
                                           hipblasStride               stridey,
                                           int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCaxpyStridedBatched_v2(hipblasHandle_t   handle,
                                              int               n,
                                              const hipComplex* alpha,
                                              const hipComplex* x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              hipComplex*       y,
                                              int               incy,
                                              hipblasStride     stridey,
                                              int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZaxpyStridedBatched_v2(hipblasHandle_t         handle,
                                              int                     n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              hipDoubleComplex*       y,
                                              int                     incy,
                                              hipblasStride           stridey,
                                              int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// 64-bit interface
hipblasStatus_t hipblasHaxpyStridedBatched_64(hipblasHandle_t    handle,
                                              int64_t            n,
                                              const hipblasHalf* alpha,
                                              const hipblasHalf* x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              hipblasHalf*       y,
                                              int64_t            incy,
                                              hipblasStride      stridey,
                                              int64_t            batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasSaxpyStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const float*    alpha,
                                              const float*    x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              float*          y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDaxpyStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const double*   alpha,
                                              const double*   x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              double*         y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCaxpyStridedBatched_64(hipblasHandle_t       handle,
                                              int64_t               n,
                                              const hipblasComplex* alpha,
                                              const hipblasComplex* x,
                                              int64_t               incx,
                                              hipblasStride         stridex,
                                              hipblasComplex*       y,
                                              int64_t               incy,
                                              hipblasStride         stridey,
                                              int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZaxpyStridedBatched_64(hipblasHandle_t             handle,
                                              int64_t                     n,
                                              const hipblasDoubleComplex* alpha,
                                              const hipblasDoubleComplex* x,
                                              int64_t                     incx,
                                              hipblasStride               stridex,
                                              hipblasDoubleComplex*       y,
                                              int64_t                     incy,
                                              hipblasStride               stridey,
                                              int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCaxpyStridedBatched_v2_64(hipblasHandle_t   handle,
                                                 int64_t           n,
                                                 const hipComplex* alpha,
                                                 const hipComplex* x,
                                                 int64_t           incx,
                                                 hipblasStride     stridex,
                                                 hipComplex*       y,
                                                 int64_t           incy,
                                                 hipblasStride     stridey,
                                                 int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZaxpyStridedBatched_v2_64(hipblasHandle_t         handle,
                                                 int64_t                 n,
                                                 const hipDoubleComplex* alpha,
                                                 const hipDoubleComplex* x,
                                                 int64_t                 incx,
                                                 hipblasStride           stridex,
                                                 hipDoubleComplex*       y,
                                                 int64_t                 incy,
                                                 hipblasStride           stridey,
                                                 int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// copy
hipblasStatus_t
    hipblasScopy(hipblasHandle_t handle, int n, const float* x, int incx, float* y, int incy)
try
{
    return hipblasConvertStatus(cublasScopy((cublasHandle_t)handle, n, x, incx, y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasDcopy(hipblasHandle_t handle, int n, const double* x, int incx, double* y, int incy)
try
{
    return hipblasConvertStatus(cublasDcopy((cublasHandle_t)handle, n, x, incx, y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCcopy(
    hipblasHandle_t handle, int n, const hipblasComplex* x, int incx, hipblasComplex* y, int incy)
try
{
    return hipblasConvertStatus(
        cublasCcopy((cublasHandle_t)handle, n, (cuComplex*)x, incx, (cuComplex*)y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZcopy(hipblasHandle_t             handle,
                             int                         n,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             hipblasDoubleComplex*       y,
                             int                         incy)
try
{
    return hipblasConvertStatus(cublasZcopy(
        (cublasHandle_t)handle, n, (cuDoubleComplex*)x, incx, (cuDoubleComplex*)y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCcopy_v2(
    hipblasHandle_t handle, int n, const hipComplex* x, int incx, hipComplex* y, int incy)
try
{
    return hipblasConvertStatus(
        cublasCcopy((cublasHandle_t)handle, n, (cuComplex*)x, incx, (cuComplex*)y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZcopy_v2(hipblasHandle_t         handle,
                                int                     n,
                                const hipDoubleComplex* x,
                                int                     incx,
                                hipDoubleComplex*       y,
                                int                     incy)
try
{
    return hipblasConvertStatus(cublasZcopy(
        (cublasHandle_t)handle, n, (cuDoubleComplex*)x, incx, (cuDoubleComplex*)y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// 64-bit interface
hipblasStatus_t hipblasScopy_64(
    hipblasHandle_t handle, int64_t n, const float* x, int64_t incx, float* y, int64_t incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasScopy_64((cublasHandle_t)handle, n, x, incx, y, incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDcopy_64(
    hipblasHandle_t handle, int64_t n, const double* x, int64_t incx, double* y, int64_t incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasDcopy_64((cublasHandle_t)handle, n, x, incx, y, incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCcopy_64(hipblasHandle_t       handle,
                                int64_t               n,
                                const hipblasComplex* x,
                                int64_t               incx,
                                hipblasComplex*       y,
                                int64_t               incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasCcopy_64((cublasHandle_t)handle, n, (cuComplex*)x, incx, (cuComplex*)y, incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZcopy_64(hipblasHandle_t             handle,
                                int64_t                     n,
                                const hipblasDoubleComplex* x,
                                int64_t                     incx,
                                hipblasDoubleComplex*       y,
                                int64_t                     incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZcopy_64(
        (cublasHandle_t)handle, n, (cuDoubleComplex*)x, incx, (cuDoubleComplex*)y, incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCcopy_v2_64(hipblasHandle_t   handle,
                                   int64_t           n,
                                   const hipComplex* x,
                                   int64_t           incx,
                                   hipComplex*       y,
                                   int64_t           incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasCcopy_64((cublasHandle_t)handle, n, (cuComplex*)x, incx, (cuComplex*)y, incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZcopy_v2_64(hipblasHandle_t         handle,
                                   int64_t                 n,
                                   const hipDoubleComplex* x,
                                   int64_t                 incx,
                                   hipDoubleComplex*       y,
                                   int64_t                 incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZcopy_64(
        (cublasHandle_t)handle, n, (cuDoubleComplex*)x, incx, (cuDoubleComplex*)y, incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// copy_batched
hipblasStatus_t hipblasScopyBatched(hipblasHandle_t    handle,
                                    int                n,
                                    const float* const x[],
                                    int                incx,
                                    float* const       y[],
                                    int                incy,
                                    int                batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDcopyBatched(hipblasHandle_t     handle,
                                    int                 n,
                                    const double* const x[],
                                    int                 incx,
                                    double* const       y[],
                                    int                 incy,
                                    int                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCcopyBatched(hipblasHandle_t             handle,
                                    int                         n,
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    hipblasComplex* const       y[],
                                    int                         incy,
                                    int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZcopyBatched(hipblasHandle_t                   handle,
                                    int                               n,
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    hipblasDoubleComplex* const       y[],
                                    int                               incy,
                                    int                               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCcopyBatched_v2(hipblasHandle_t         handle,
                                       int                     n,
                                       const hipComplex* const x[],
                                       int                     incx,
                                       hipComplex* const       y[],
                                       int                     incy,
                                       int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZcopyBatched_v2(hipblasHandle_t               handle,
                                       int                           n,
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       hipDoubleComplex* const       y[],
                                       int                           incy,
                                       int                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// 64-bit interface
hipblasStatus_t hipblasScopyBatched_64(hipblasHandle_t    handle,
                                       int64_t            n,
                                       const float* const x[],
                                       int64_t            incx,
                                       float* const       y[],
                                       int64_t            incy,
                                       int64_t            batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDcopyBatched_64(hipblasHandle_t     handle,
                                       int64_t             n,
                                       const double* const x[],
                                       int64_t             incx,
                                       double* const       y[],
                                       int64_t             incy,
                                       int64_t             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCcopyBatched_64(hipblasHandle_t             handle,
                                       int64_t                     n,
                                       const hipblasComplex* const x[],
                                       int64_t                     incx,
                                       hipblasComplex* const       y[],
                                       int64_t                     incy,
                                       int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZcopyBatched_64(hipblasHandle_t                   handle,
                                       int64_t                           n,
                                       const hipblasDoubleComplex* const x[],
                                       int64_t                           incx,
                                       hipblasDoubleComplex* const       y[],
                                       int64_t                           incy,
                                       int64_t                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCcopyBatched_v2_64(hipblasHandle_t         handle,
                                          int64_t                 n,
                                          const hipComplex* const x[],
                                          int64_t                 incx,
                                          hipComplex* const       y[],
                                          int64_t                 incy,
                                          int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZcopyBatched_v2_64(hipblasHandle_t               handle,
                                          int64_t                       n,
                                          const hipDoubleComplex* const x[],
                                          int64_t                       incx,
                                          hipDoubleComplex* const       y[],
                                          int64_t                       incy,
                                          int64_t                       batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// copy_strided_batched
hipblasStatus_t hipblasScopyStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const float*    x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           float*          y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDcopyStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const double*   x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           double*         y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCcopyStridedBatched(hipblasHandle_t       handle,
                                           int                   n,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           hipblasComplex*       y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZcopyStridedBatched(hipblasHandle_t             handle,
                                           int                         n,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           hipblasDoubleComplex*       y,
                                           int                         incy,
                                           hipblasStride               stridey,
                                           int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCcopyStridedBatched_v2(hipblasHandle_t   handle,
                                              int               n,
                                              const hipComplex* x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              hipComplex*       y,
                                              int               incy,
                                              hipblasStride     stridey,
                                              int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZcopyStridedBatched_v2(hipblasHandle_t         handle,
                                              int                     n,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              hipDoubleComplex*       y,
                                              int                     incy,
                                              hipblasStride           stridey,
                                              int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// 64-bit interface
hipblasStatus_t hipblasScopyStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const float*    x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              float*          y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDcopyStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const double*   x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              double*         y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCcopyStridedBatched_64(hipblasHandle_t       handle,
                                              int64_t               n,
                                              const hipblasComplex* x,
                                              int64_t               incx,
                                              hipblasStride         stridex,
                                              hipblasComplex*       y,
                                              int64_t               incy,
                                              hipblasStride         stridey,
                                              int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZcopyStridedBatched_64(hipblasHandle_t             handle,
                                              int64_t                     n,
                                              const hipblasDoubleComplex* x,
                                              int64_t                     incx,
                                              hipblasStride               stridex,
                                              hipblasDoubleComplex*       y,
                                              int64_t                     incy,
                                              hipblasStride               stridey,
                                              int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCcopyStridedBatched_v2_64(hipblasHandle_t   handle,
                                                 int64_t           n,
                                                 const hipComplex* x,
                                                 int64_t           incx,
                                                 hipblasStride     stridex,
                                                 hipComplex*       y,
                                                 int64_t           incy,
                                                 hipblasStride     stridey,
                                                 int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZcopyStridedBatched_v2_64(hipblasHandle_t         handle,
                                                 int64_t                 n,
                                                 const hipDoubleComplex* x,
                                                 int64_t                 incx,
                                                 hipblasStride           stridex,
                                                 hipDoubleComplex*       y,
                                                 int64_t                 incy,
                                                 hipblasStride           stridey,
                                                 int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// dot
hipblasStatus_t hipblasHdot(hipblasHandle_t    handle,
                            int                n,
                            const hipblasHalf* x,
                            int                incx,
                            const hipblasHalf* y,
                            int                incy,
                            hipblasHalf*       result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasBfdot(hipblasHandle_t        handle,
                             int                    n,
                             const hipblasBfloat16* x,
                             int                    incx,
                             const hipblasBfloat16* y,
                             int                    incy,
                             hipblasBfloat16*       result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasSdot(hipblasHandle_t handle,
                            int             n,
                            const float*    x,
                            int             incx,
                            const float*    y,
                            int             incy,
                            float*          result)
try
{
    return hipblasConvertStatus(cublasSdot((cublasHandle_t)handle, n, x, incx, y, incy, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDdot(hipblasHandle_t handle,
                            int             n,
                            const double*   x,
                            int             incx,
                            const double*   y,
                            int             incy,
                            double*         result)
try
{
    return hipblasConvertStatus(cublasDdot((cublasHandle_t)handle, n, x, incx, y, incy, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCdotc(hipblasHandle_t       handle,
                             int                   n,
                             const hipblasComplex* x,
                             int                   incx,
                             const hipblasComplex* y,
                             int                   incy,
                             hipblasComplex*       result)
try
{
    return hipblasConvertStatus(cublasCdotc(
        (cublasHandle_t)handle, n, (cuComplex*)x, incx, (cuComplex*)y, incy, (cuComplex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCdotu(hipblasHandle_t       handle,
                             int                   n,
                             const hipblasComplex* x,
                             int                   incx,
                             const hipblasComplex* y,
                             int                   incy,
                             hipblasComplex*       result)
try
{
    return hipblasConvertStatus(cublasCdotu(
        (cublasHandle_t)handle, n, (cuComplex*)x, incx, (cuComplex*)y, incy, (cuComplex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdotc(hipblasHandle_t             handle,
                             int                         n,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             const hipblasDoubleComplex* y,
                             int                         incy,
                             hipblasDoubleComplex*       result)
try
{
    return hipblasConvertStatus(cublasZdotc((cublasHandle_t)handle,
                                            n,
                                            (cuDoubleComplex*)x,
                                            incx,
                                            (cuDoubleComplex*)y,
                                            incy,
                                            (cuDoubleComplex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdotu(hipblasHandle_t             handle,
                             int                         n,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             const hipblasDoubleComplex* y,
                             int                         incy,
                             hipblasDoubleComplex*       result)
try
{
    return hipblasConvertStatus(cublasZdotu((cublasHandle_t)handle,
                                            n,
                                            (cuDoubleComplex*)x,
                                            incx,
                                            (cuDoubleComplex*)y,
                                            incy,
                                            (cuDoubleComplex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCdotc_v2(hipblasHandle_t   handle,
                                int               n,
                                const hipComplex* x,
                                int               incx,
                                const hipComplex* y,
                                int               incy,
                                hipComplex*       result)
try
{
    return hipblasConvertStatus(cublasCdotc(
        (cublasHandle_t)handle, n, (cuComplex*)x, incx, (cuComplex*)y, incy, (cuComplex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCdotu_v2(hipblasHandle_t   handle,
                                int               n,
                                const hipComplex* x,
                                int               incx,
                                const hipComplex* y,
                                int               incy,
                                hipComplex*       result)
try
{
    return hipblasConvertStatus(cublasCdotu(
        (cublasHandle_t)handle, n, (cuComplex*)x, incx, (cuComplex*)y, incy, (cuComplex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdotc_v2(hipblasHandle_t         handle,
                                int                     n,
                                const hipDoubleComplex* x,
                                int                     incx,
                                const hipDoubleComplex* y,
                                int                     incy,
                                hipDoubleComplex*       result)
try
{
    return hipblasConvertStatus(cublasZdotc((cublasHandle_t)handle,
                                            n,
                                            (cuDoubleComplex*)x,
                                            incx,
                                            (cuDoubleComplex*)y,
                                            incy,
                                            (cuDoubleComplex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdotu_v2(hipblasHandle_t         handle,
                                int                     n,
                                const hipDoubleComplex* x,
                                int                     incx,
                                const hipDoubleComplex* y,
                                int                     incy,
                                hipDoubleComplex*       result)
try
{
    return hipblasConvertStatus(cublasZdotu((cublasHandle_t)handle,
                                            n,
                                            (cuDoubleComplex*)x,
                                            incx,
                                            (cuDoubleComplex*)y,
                                            incy,
                                            (cuDoubleComplex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// dot_64
hipblasStatus_t hipblasHdot_64(hipblasHandle_t    handle,
                               int64_t            n,
                               const hipblasHalf* x,
                               int64_t            incx,
                               const hipblasHalf* y,
                               int64_t            incy,
                               hipblasHalf*       result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasBfdot_64(hipblasHandle_t        handle,
                                int64_t                n,
                                const hipblasBfloat16* x,
                                int64_t                incx,
                                const hipblasBfloat16* y,
                                int64_t                incy,
                                hipblasBfloat16*       result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasSdot_64(hipblasHandle_t handle,
                               int64_t         n,
                               const float*    x,
                               int64_t         incx,
                               const float*    y,
                               int64_t         incy,
                               float*          result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasSdot_64((cublasHandle_t)handle, n, x, incx, y, incy, result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDdot_64(hipblasHandle_t handle,
                               int64_t         n,
                               const double*   x,
                               int64_t         incx,
                               const double*   y,
                               int64_t         incy,
                               double*         result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasDdot_64((cublasHandle_t)handle, n, x, incx, y, incy, result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCdotc_64(hipblasHandle_t       handle,
                                int64_t               n,
                                const hipblasComplex* x,
                                int64_t               incx,
                                const hipblasComplex* y,
                                int64_t               incy,
                                hipblasComplex*       result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCdotc_64(
        (cublasHandle_t)handle, n, (cuComplex*)x, incx, (cuComplex*)y, incy, (cuComplex*)result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCdotu_64(hipblasHandle_t       handle,
                                int64_t               n,
                                const hipblasComplex* x,
                                int64_t               incx,
                                const hipblasComplex* y,
                                int64_t               incy,
                                hipblasComplex*       result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCdotu_64(
        (cublasHandle_t)handle, n, (cuComplex*)x, incx, (cuComplex*)y, incy, (cuComplex*)result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdotc_64(hipblasHandle_t             handle,
                                int64_t                     n,
                                const hipblasDoubleComplex* x,
                                int64_t                     incx,
                                const hipblasDoubleComplex* y,
                                int64_t                     incy,
                                hipblasDoubleComplex*       result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZdotc_64((cublasHandle_t)handle,
                                               n,
                                               (cuDoubleComplex*)x,
                                               incx,
                                               (cuDoubleComplex*)y,
                                               incy,
                                               (cuDoubleComplex*)result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdotu_64(hipblasHandle_t             handle,
                                int64_t                     n,
                                const hipblasDoubleComplex* x,
                                int64_t                     incx,
                                const hipblasDoubleComplex* y,
                                int64_t                     incy,
                                hipblasDoubleComplex*       result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZdotu_64((cublasHandle_t)handle,
                                               n,
                                               (cuDoubleComplex*)x,
                                               incx,
                                               (cuDoubleComplex*)y,
                                               incy,
                                               (cuDoubleComplex*)result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCdotc_v2_64(hipblasHandle_t   handle,
                                   int64_t           n,
                                   const hipComplex* x,
                                   int64_t           incx,
                                   const hipComplex* y,
                                   int64_t           incy,
                                   hipComplex*       result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCdotc_64(
        (cublasHandle_t)handle, n, (cuComplex*)x, incx, (cuComplex*)y, incy, (cuComplex*)result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCdotu_v2_64(hipblasHandle_t   handle,
                                   int64_t           n,
                                   const hipComplex* x,
                                   int64_t           incx,
                                   const hipComplex* y,
                                   int64_t           incy,
                                   hipComplex*       result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCdotu_64(
        (cublasHandle_t)handle, n, (cuComplex*)x, incx, (cuComplex*)y, incy, (cuComplex*)result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdotc_v2_64(hipblasHandle_t         handle,
                                   int64_t                 n,
                                   const hipDoubleComplex* x,
                                   int64_t                 incx,
                                   const hipDoubleComplex* y,
                                   int64_t                 incy,
                                   hipDoubleComplex*       result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZdotc_64((cublasHandle_t)handle,
                                               n,
                                               (cuDoubleComplex*)x,
                                               incx,
                                               (cuDoubleComplex*)y,
                                               incy,
                                               (cuDoubleComplex*)result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdotu_v2_64(hipblasHandle_t         handle,
                                   int64_t                 n,
                                   const hipDoubleComplex* x,
                                   int64_t                 incx,
                                   const hipDoubleComplex* y,
                                   int64_t                 incy,
                                   hipDoubleComplex*       result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZdotu_64((cublasHandle_t)handle,
                                               n,
                                               (cuDoubleComplex*)x,
                                               incx,
                                               (cuDoubleComplex*)y,
                                               incy,
                                               (cuDoubleComplex*)result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// dot_batched
hipblasStatus_t hipblasHdotBatched(hipblasHandle_t          handle,
                                   int                      n,
                                   const hipblasHalf* const x[],
                                   int                      incx,
                                   const hipblasHalf* const y[],
                                   int                      incy,
                                   int                      batchCount,
                                   hipblasHalf*             result)
try
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
    // // TODO warn user that function was demoted to ignore batch
    // return hipblasConvertStatus(
    //     cublasSdot((cublasHandle_t)handle, n, x, incx, y, incy, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasBfdotBatched(hipblasHandle_t              handle,
                                    int                          n,
                                    const hipblasBfloat16* const x[],
                                    int                          incx,
                                    const hipblasBfloat16* const y[],
                                    int                          incy,
                                    int                          batchCount,
                                    hipblasBfloat16*             result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasSdotBatched(hipblasHandle_t    handle,
                                   int                n,
                                   const float* const x[],
                                   int                incx,
                                   const float* const y[],
                                   int                incy,
                                   int                batchCount,
                                   float*             result)
try
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
    // // TODO warn user that function was demoted to ignore batch
    // return hipblasConvertStatus(
    //     cublasSdot((cublasHandle_t)handle, n, x, incx, y, incy, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDdotBatched(hipblasHandle_t     handle,
                                   int                 n,
                                   const double* const x[],
                                   int                 incx,
                                   const double* const y[],
                                   int                 incy,
                                   int                 batchCount,
                                   double*             result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCdotcBatched(hipblasHandle_t             handle,
                                    int                         n,
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    const hipblasComplex* const y[],
                                    int                         incy,
                                    int                         batchCount,
                                    hipblasComplex*             result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCdotuBatched(hipblasHandle_t             handle,
                                    int                         n,
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    const hipblasComplex* const y[],
                                    int                         incy,
                                    int                         batchCount,
                                    hipblasComplex*             result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdotcBatched(hipblasHandle_t                   handle,
                                    int                               n,
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    const hipblasDoubleComplex* const y[],
                                    int                               incy,
                                    int                               batchCount,
                                    hipblasDoubleComplex*             result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdotuBatched(hipblasHandle_t                   handle,
                                    int                               n,
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    const hipblasDoubleComplex* const y[],
                                    int                               incy,
                                    int                               batchCount,
                                    hipblasDoubleComplex*             result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCdotcBatched_v2(hipblasHandle_t         handle,
                                       int                     n,
                                       const hipComplex* const x[],
                                       int                     incx,
                                       const hipComplex* const y[],
                                       int                     incy,
                                       int                     batchCount,
                                       hipComplex*             result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCdotuBatched_v2(hipblasHandle_t         handle,
                                       int                     n,
                                       const hipComplex* const x[],
                                       int                     incx,
                                       const hipComplex* const y[],
                                       int                     incy,
                                       int                     batchCount,
                                       hipComplex*             result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdotcBatched_v2(hipblasHandle_t               handle,
                                       int                           n,
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       const hipDoubleComplex* const y[],
                                       int                           incy,
                                       int                           batchCount,
                                       hipDoubleComplex*             result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdotuBatched_v2(hipblasHandle_t               handle,
                                       int                           n,
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       const hipDoubleComplex* const y[],
                                       int                           incy,
                                       int                           batchCount,
                                       hipDoubleComplex*             result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// dot_batched_64
hipblasStatus_t hipblasHdotBatched_64(hipblasHandle_t          handle,
                                      int64_t                  n,
                                      const hipblasHalf* const x[],
                                      int64_t                  incx,
                                      const hipblasHalf* const y[],
                                      int64_t                  incy,
                                      int64_t                  batchCount,
                                      hipblasHalf*             result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasBfdotBatched_64(hipblasHandle_t              handle,
                                       int64_t                      n,
                                       const hipblasBfloat16* const x[],
                                       int64_t                      incx,
                                       const hipblasBfloat16* const y[],
                                       int64_t                      incy,
                                       int64_t                      batchCount,
                                       hipblasBfloat16*             result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasSdotBatched_64(hipblasHandle_t    handle,
                                      int64_t            n,
                                      const float* const x[],
                                      int64_t            incx,
                                      const float* const y[],
                                      int64_t            incy,
                                      int64_t            batchCount,
                                      float*             result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDdotBatched_64(hipblasHandle_t     handle,
                                      int64_t             n,
                                      const double* const x[],
                                      int64_t             incx,
                                      const double* const y[],
                                      int64_t             incy,
                                      int64_t             batchCount,
                                      double*             result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCdotcBatched_64(hipblasHandle_t             handle,
                                       int64_t                     n,
                                       const hipblasComplex* const x[],
                                       int64_t                     incx,
                                       const hipblasComplex* const y[],
                                       int64_t                     incy,
                                       int64_t                     batchCount,
                                       hipblasComplex*             result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCdotuBatched_64(hipblasHandle_t             handle,
                                       int64_t                     n,
                                       const hipblasComplex* const x[],
                                       int64_t                     incx,
                                       const hipblasComplex* const y[],
                                       int64_t                     incy,
                                       int64_t                     batchCount,
                                       hipblasComplex*             result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdotcBatched_64(hipblasHandle_t                   handle,
                                       int64_t                           n,
                                       const hipblasDoubleComplex* const x[],
                                       int64_t                           incx,
                                       const hipblasDoubleComplex* const y[],
                                       int64_t                           incy,
                                       int64_t                           batchCount,
                                       hipblasDoubleComplex*             result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdotuBatched_64(hipblasHandle_t                   handle,
                                       int64_t                           n,
                                       const hipblasDoubleComplex* const x[],
                                       int64_t                           incx,
                                       const hipblasDoubleComplex* const y[],
                                       int64_t                           incy,
                                       int64_t                           batchCount,
                                       hipblasDoubleComplex*             result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCdotcBatched_v2_64(hipblasHandle_t         handle,
                                          int64_t                 n,
                                          const hipComplex* const x[],
                                          int64_t                 incx,
                                          const hipComplex* const y[],
                                          int64_t                 incy,
                                          int64_t                 batchCount,
                                          hipComplex*             result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCdotuBatched_v2_64(hipblasHandle_t         handle,
                                          int64_t                 n,
                                          const hipComplex* const x[],
                                          int64_t                 incx,
                                          const hipComplex* const y[],
                                          int64_t                 incy,
                                          int64_t                 batchCount,
                                          hipComplex*             result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdotcBatched_v2_64(hipblasHandle_t               handle,
                                          int64_t                       n,
                                          const hipDoubleComplex* const x[],
                                          int64_t                       incx,
                                          const hipDoubleComplex* const y[],
                                          int64_t                       incy,
                                          int64_t                       batchCount,
                                          hipDoubleComplex*             result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdotuBatched_v2_64(hipblasHandle_t               handle,
                                          int64_t                       n,
                                          const hipDoubleComplex* const x[],
                                          int64_t                       incx,
                                          const hipDoubleComplex* const y[],
                                          int64_t                       incy,
                                          int64_t                       batchCount,
                                          hipDoubleComplex*             result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// dot_strided_batched
hipblasStatus_t hipblasHdotStridedBatched(hipblasHandle_t    handle,
                                          int                n,
                                          const hipblasHalf* x,
                                          int                incx,
                                          hipblasStride      stridex,
                                          const hipblasHalf* y,
                                          int                incy,
                                          hipblasStride      stridey,
                                          int                batchCount,
                                          hipblasHalf*       result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasBfdotStridedBatched(hipblasHandle_t        handle,
                                           int                    n,
                                           const hipblasBfloat16* x,
                                           int                    incx,
                                           hipblasStride          stridex,
                                           const hipblasBfloat16* y,
                                           int                    incy,
                                           hipblasStride          stridey,
                                           int                    batchCount,
                                           hipblasBfloat16*       result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasSdotStridedBatched(hipblasHandle_t handle,
                                          int             n,
                                          const float*    x,
                                          int             incx,
                                          hipblasStride   stridex,
                                          const float*    y,
                                          int             incy,
                                          hipblasStride   stridey,
                                          int             batchCount,
                                          float*          result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDdotStridedBatched(hipblasHandle_t handle,
                                          int             n,
                                          const double*   x,
                                          int             incx,
                                          hipblasStride   stridex,
                                          const double*   y,
                                          int             incy,
                                          hipblasStride   stridey,
                                          int             batchCount,
                                          double*         result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCdotcStridedBatched(hipblasHandle_t       handle,
                                           int                   n,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           const hipblasComplex* y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           int                   batchCount,
                                           hipblasComplex*       result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCdotuStridedBatched(hipblasHandle_t       handle,
                                           int                   n,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           const hipblasComplex* y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           int                   batchCount,
                                           hipblasComplex*       result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdotcStridedBatched(hipblasHandle_t             handle,
                                           int                         n,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           const hipblasDoubleComplex* y,
                                           int                         incy,
                                           hipblasStride               stridey,
                                           int                         batchCount,
                                           hipblasDoubleComplex*       result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdotuStridedBatched(hipblasHandle_t             handle,
                                           int                         n,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           const hipblasDoubleComplex* y,
                                           int                         incy,
                                           hipblasStride               stridey,
                                           int                         batchCount,
                                           hipblasDoubleComplex*       result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCdotcStridedBatched_v2(hipblasHandle_t   handle,
                                              int               n,
                                              const hipComplex* x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              const hipComplex* y,
                                              int               incy,
                                              hipblasStride     stridey,
                                              int               batchCount,
                                              hipComplex*       result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCdotuStridedBatched_v2(hipblasHandle_t   handle,
                                              int               n,
                                              const hipComplex* x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              const hipComplex* y,
                                              int               incy,
                                              hipblasStride     stridey,
                                              int               batchCount,
                                              hipComplex*       result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdotcStridedBatched_v2(hipblasHandle_t         handle,
                                              int                     n,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* y,
                                              int                     incy,
                                              hipblasStride           stridey,
                                              int                     batchCount,
                                              hipDoubleComplex*       result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdotuStridedBatched_v2(hipblasHandle_t         handle,
                                              int                     n,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* y,
                                              int                     incy,
                                              hipblasStride           stridey,
                                              int                     batchCount,
                                              hipDoubleComplex*       result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// dot_strided_batched_64
hipblasStatus_t hipblasHdotStridedBatched_64(hipblasHandle_t    handle,
                                             int64_t            n,
                                             const hipblasHalf* x,
                                             int64_t            incx,
                                             hipblasStride      stridex,
                                             const hipblasHalf* y,
                                             int64_t            incy,
                                             hipblasStride      stridey,
                                             int64_t            batchCount,
                                             hipblasHalf*       result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasBfdotStridedBatched_64(hipblasHandle_t        handle,
                                              int64_t                n,
                                              const hipblasBfloat16* x,
                                              int64_t                incx,
                                              hipblasStride          stridex,
                                              const hipblasBfloat16* y,
                                              int64_t                incy,
                                              hipblasStride          stridey,
                                              int64_t                batchCount,
                                              hipblasBfloat16*       result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasSdotStridedBatched_64(hipblasHandle_t handle,
                                             int64_t         n,
                                             const float*    x,
                                             int64_t         incx,
                                             hipblasStride   stridex,
                                             const float*    y,
                                             int64_t         incy,
                                             hipblasStride   stridey,
                                             int64_t         batchCount,
                                             float*          result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDdotStridedBatched_64(hipblasHandle_t handle,
                                             int64_t         n,
                                             const double*   x,
                                             int64_t         incx,
                                             hipblasStride   stridex,
                                             const double*   y,
                                             int64_t         incy,
                                             hipblasStride   stridey,
                                             int64_t         batchCount,
                                             double*         result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCdotcStridedBatched_64(hipblasHandle_t       handle,
                                              int64_t               n,
                                              const hipblasComplex* x,
                                              int64_t               incx,
                                              hipblasStride         stridex,
                                              const hipblasComplex* y,
                                              int64_t               incy,
                                              hipblasStride         stridey,
                                              int64_t               batchCount,
                                              hipblasComplex*       result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCdotuStridedBatched_64(hipblasHandle_t       handle,
                                              int64_t               n,
                                              const hipblasComplex* x,
                                              int64_t               incx,
                                              hipblasStride         stridex,
                                              const hipblasComplex* y,
                                              int64_t               incy,
                                              hipblasStride         stridey,
                                              int64_t               batchCount,
                                              hipblasComplex*       result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdotcStridedBatched_64(hipblasHandle_t             handle,
                                              int64_t                     n,
                                              const hipblasDoubleComplex* x,
                                              int64_t                     incx,
                                              hipblasStride               stridex,
                                              const hipblasDoubleComplex* y,
                                              int64_t                     incy,
                                              hipblasStride               stridey,
                                              int64_t                     batchCount,
                                              hipblasDoubleComplex*       result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdotuStridedBatched_64(hipblasHandle_t             handle,
                                              int64_t                     n,
                                              const hipblasDoubleComplex* x,
                                              int64_t                     incx,
                                              hipblasStride               stridex,
                                              const hipblasDoubleComplex* y,
                                              int64_t                     incy,
                                              hipblasStride               stridey,
                                              int64_t                     batchCount,
                                              hipblasDoubleComplex*       result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCdotcStridedBatched_v2_64(hipblasHandle_t   handle,
                                                 int64_t           n,
                                                 const hipComplex* x,
                                                 int64_t           incx,
                                                 hipblasStride     stridex,
                                                 const hipComplex* y,
                                                 int64_t           incy,
                                                 hipblasStride     stridey,
                                                 int64_t           batchCount,
                                                 hipComplex*       result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCdotuStridedBatched_v2_64(hipblasHandle_t   handle,
                                                 int64_t           n,
                                                 const hipComplex* x,
                                                 int64_t           incx,
                                                 hipblasStride     stridex,
                                                 const hipComplex* y,
                                                 int64_t           incy,
                                                 hipblasStride     stridey,
                                                 int64_t           batchCount,
                                                 hipComplex*       result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdotcStridedBatched_v2_64(hipblasHandle_t         handle,
                                                 int64_t                 n,
                                                 const hipDoubleComplex* x,
                                                 int64_t                 incx,
                                                 hipblasStride           stridex,
                                                 const hipDoubleComplex* y,
                                                 int64_t                 incy,
                                                 hipblasStride           stridey,
                                                 int64_t                 batchCount,
                                                 hipDoubleComplex*       result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdotuStridedBatched_v2_64(hipblasHandle_t         handle,
                                                 int64_t                 n,
                                                 const hipDoubleComplex* x,
                                                 int64_t                 incx,
                                                 hipblasStride           stridex,
                                                 const hipDoubleComplex* y,
                                                 int64_t                 incy,
                                                 hipblasStride           stridey,
                                                 int64_t                 batchCount,
                                                 hipDoubleComplex*       result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// nrm2
hipblasStatus_t hipblasSnrm2(hipblasHandle_t handle, int n, const float* x, int incx, float* result)
try
{
    return hipblasConvertStatus(cublasSnrm2((cublasHandle_t)handle, n, x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasDnrm2(hipblasHandle_t handle, int n, const double* x, int incx, double* result)
try
{
    return hipblasConvertStatus(cublasDnrm2((cublasHandle_t)handle, n, x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasScnrm2(hipblasHandle_t handle, int n, const hipblasComplex* x, int incx, float* result)
try
{
    return hipblasConvertStatus(
        cublasScnrm2((cublasHandle_t)handle, n, (cuComplex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDznrm2(
    hipblasHandle_t handle, int n, const hipblasDoubleComplex* x, int incx, double* result)
try
{
    return hipblasConvertStatus(
        cublasDznrm2((cublasHandle_t)handle, n, (cuDoubleComplex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasScnrm2_v2(hipblasHandle_t handle, int n, const hipComplex* x, int incx, float* result)
try
{
    return hipblasConvertStatus(
        cublasScnrm2((cublasHandle_t)handle, n, (cuComplex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDznrm2_v2(
    hipblasHandle_t handle, int n, const hipDoubleComplex* x, int incx, double* result)
try
{
    return hipblasConvertStatus(
        cublasDznrm2((cublasHandle_t)handle, n, (cuDoubleComplex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// nrm2_64
hipblasStatus_t
    hipblasSnrm2_64(hipblasHandle_t handle, int64_t n, const float* x, int64_t incx, float* result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasSnrm2_64((cublasHandle_t)handle, n, x, incx, result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDnrm2_64(
    hipblasHandle_t handle, int64_t n, const double* x, int64_t incx, double* result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasDnrm2_64((cublasHandle_t)handle, n, x, incx, result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasScnrm2_64(
    hipblasHandle_t handle, int64_t n, const hipblasComplex* x, int64_t incx, float* result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasScnrm2_64((cublasHandle_t)handle, n, (cuComplex*)x, incx, result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDznrm2_64(
    hipblasHandle_t handle, int64_t n, const hipblasDoubleComplex* x, int64_t incx, double* result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasDznrm2_64((cublasHandle_t)handle, n, (cuDoubleComplex*)x, incx, result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasScnrm2_v2_64(
    hipblasHandle_t handle, int64_t n, const hipComplex* x, int64_t incx, float* result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasScnrm2_64((cublasHandle_t)handle, n, (cuComplex*)x, incx, result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDznrm2_v2_64(
    hipblasHandle_t handle, int64_t n, const hipDoubleComplex* x, int64_t incx, double* result)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasDznrm2_64((cublasHandle_t)handle, n, (cuDoubleComplex*)x, incx, result));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// nrm2_batched
hipblasStatus_t hipblasSnrm2Batched(
    hipblasHandle_t handle, int n, const float* const x[], int incx, int batchCount, float* result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDnrm2Batched(hipblasHandle_t     handle,
                                    int                 n,
                                    const double* const x[],
                                    int                 incx,
                                    int                 batchCount,
                                    double*             result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasScnrm2Batched(hipblasHandle_t             handle,
                                     int                         n,
                                     const hipblasComplex* const x[],
                                     int                         incx,
                                     int                         batchCount,
                                     float*                      result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDznrm2Batched(hipblasHandle_t                   handle,
                                     int                               n,
                                     const hipblasDoubleComplex* const x[],
                                     int                               incx,
                                     int                               batchCount,
                                     double*                           result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasScnrm2Batched_v2(hipblasHandle_t         handle,
                                        int                     n,
                                        const hipComplex* const x[],
                                        int                     incx,
                                        int                     batchCount,
                                        float*                  result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDznrm2Batched_v2(hipblasHandle_t               handle,
                                        int                           n,
                                        const hipDoubleComplex* const x[],
                                        int                           incx,
                                        int                           batchCount,
                                        double*                       result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// nrm2_batched_64
hipblasStatus_t hipblasSnrm2Batched_64(hipblasHandle_t    handle,
                                       int64_t            n,
                                       const float* const x[],
                                       int64_t            incx,
                                       int64_t            batchCount,
                                       float*             result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDnrm2Batched_64(hipblasHandle_t     handle,
                                       int64_t             n,
                                       const double* const x[],
                                       int64_t             incx,
                                       int64_t             batchCount,
                                       double*             result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasScnrm2Batched_64(hipblasHandle_t             handle,
                                        int64_t                     n,
                                        const hipblasComplex* const x[],
                                        int64_t                     incx,
                                        int64_t                     batchCount,
                                        float*                      result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDznrm2Batched_64(hipblasHandle_t                   handle,
                                        int64_t                           n,
                                        const hipblasDoubleComplex* const x[],
                                        int64_t                           incx,
                                        int64_t                           batchCount,
                                        double*                           result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasScnrm2Batched_v2_64(hipblasHandle_t         handle,
                                           int64_t                 n,
                                           const hipComplex* const x[],
                                           int64_t                 incx,
                                           int64_t                 batchCount,
                                           float*                  result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDznrm2Batched_v2_64(hipblasHandle_t               handle,
                                           int64_t                       n,
                                           const hipDoubleComplex* const x[],
                                           int64_t                       incx,
                                           int64_t                       batchCount,
                                           double*                       result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// nrm2_strided_batched
hipblasStatus_t hipblasSnrm2StridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const float*    x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           int             batchCount,
                                           float*          result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDnrm2StridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const double*   x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           int             batchCount,
                                           double*         result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasScnrm2StridedBatched(hipblasHandle_t       handle,
                                            int                   n,
                                            const hipblasComplex* x,
                                            int                   incx,
                                            hipblasStride         stridex,
                                            int                   batchCount,
                                            float*                result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDznrm2StridedBatched(hipblasHandle_t             handle,
                                            int                         n,
                                            const hipblasDoubleComplex* x,
                                            int                         incx,
                                            hipblasStride               stridex,
                                            int                         batchCount,
                                            double*                     result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasScnrm2StridedBatched_v2(hipblasHandle_t   handle,
                                               int               n,
                                               const hipComplex* x,
                                               int               incx,
                                               hipblasStride     stridex,
                                               int               batchCount,
                                               float*            result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDznrm2StridedBatched_v2(hipblasHandle_t         handle,
                                               int                     n,
                                               const hipDoubleComplex* x,
                                               int                     incx,
                                               hipblasStride           stridex,
                                               int                     batchCount,
                                               double*                 result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// nrm2_strided_batched_64
hipblasStatus_t hipblasSnrm2StridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const float*    x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              int64_t         batchCount,
                                              float*          result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDnrm2StridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const double*   x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              int64_t         batchCount,
                                              double*         result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasScnrm2StridedBatched_64(hipblasHandle_t       handle,
                                               int64_t               n,
                                               const hipblasComplex* x,
                                               int64_t               incx,
                                               hipblasStride         stridex,
                                               int64_t               batchCount,
                                               float*                result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDznrm2StridedBatched_64(hipblasHandle_t             handle,
                                               int64_t                     n,
                                               const hipblasDoubleComplex* x,
                                               int64_t                     incx,
                                               hipblasStride               stridex,
                                               int64_t                     batchCount,
                                               double*                     result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasScnrm2StridedBatched_v2_64(hipblasHandle_t   handle,
                                                  int64_t           n,
                                                  const hipComplex* x,
                                                  int64_t           incx,
                                                  hipblasStride     stridex,
                                                  int64_t           batchCount,
                                                  float*            result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDznrm2StridedBatched_v2_64(hipblasHandle_t         handle,
                                                  int64_t                 n,
                                                  const hipDoubleComplex* x,
                                                  int64_t                 incx,
                                                  hipblasStride           stridex,
                                                  int64_t                 batchCount,
                                                  double*                 result)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// rot
hipblasStatus_t hipblasSrot(hipblasHandle_t handle,
                            int             n,
                            float*          x,
                            int             incx,
                            float*          y,
                            int             incy,
                            const float*    c,
                            const float*    s)
try
{
    return hipblasConvertStatus(cublasSrot((cublasHandle_t)handle, n, x, incx, y, incy, c, s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrot(hipblasHandle_t handle,
                            int             n,
                            double*         x,
                            int             incx,
                            double*         y,
                            int             incy,
                            const double*   c,
                            const double*   s)
try
{
    return hipblasConvertStatus(cublasDrot((cublasHandle_t)handle, n, x, incx, y, incy, c, s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCrot(hipblasHandle_t       handle,
                            int                   n,
                            hipblasComplex*       x,
                            int                   incx,
                            hipblasComplex*       y,
                            int                   incy,
                            const float*          c,
                            const hipblasComplex* s)
try
{
    return hipblasConvertStatus(cublasCrot(
        (cublasHandle_t)handle, n, (cuComplex*)x, incx, (cuComplex*)y, incy, c, (cuComplex*)s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsrot(hipblasHandle_t handle,
                             int             n,
                             hipblasComplex* x,
                             int             incx,
                             hipblasComplex* y,
                             int             incy,
                             const float*    c,
                             const float*    s)
try
{
    return hipblasConvertStatus(
        cublasCsrot((cublasHandle_t)handle, n, (cuComplex*)x, incx, (cuComplex*)y, incy, c, s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZrot(hipblasHandle_t             handle,
                            int                         n,
                            hipblasDoubleComplex*       x,
                            int                         incx,
                            hipblasDoubleComplex*       y,
                            int                         incy,
                            const double*               c,
                            const hipblasDoubleComplex* s)
try
{
    return hipblasConvertStatus(cublasZrot((cublasHandle_t)handle,
                                           n,
                                           (cuDoubleComplex*)x,
                                           incx,
                                           (cuDoubleComplex*)y,
                                           incy,
                                           c,
                                           (cuDoubleComplex*)s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdrot(hipblasHandle_t       handle,
                             int                   n,
                             hipblasDoubleComplex* x,
                             int                   incx,
                             hipblasDoubleComplex* y,
                             int                   incy,
                             const double*         c,
                             const double*         s)
try
{
    return hipblasConvertStatus(cublasZdrot(
        (cublasHandle_t)handle, n, (cuDoubleComplex*)x, incx, (cuDoubleComplex*)y, incy, c, s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCrot_v2(hipblasHandle_t   handle,
                               int               n,
                               hipComplex*       x,
                               int               incx,
                               hipComplex*       y,
                               int               incy,
                               const float*      c,
                               const hipComplex* s)
try
{
    return hipblasConvertStatus(cublasCrot(
        (cublasHandle_t)handle, n, (cuComplex*)x, incx, (cuComplex*)y, incy, c, (cuComplex*)s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsrot_v2(hipblasHandle_t handle,
                                int             n,
                                hipComplex*     x,
                                int             incx,
                                hipComplex*     y,
                                int             incy,
                                const float*    c,
                                const float*    s)
try
{
    return hipblasConvertStatus(
        cublasCsrot((cublasHandle_t)handle, n, (cuComplex*)x, incx, (cuComplex*)y, incy, c, s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZrot_v2(hipblasHandle_t         handle,
                               int                     n,
                               hipDoubleComplex*       x,
                               int                     incx,
                               hipDoubleComplex*       y,
                               int                     incy,
                               const double*           c,
                               const hipDoubleComplex* s)
try
{
    return hipblasConvertStatus(cublasZrot((cublasHandle_t)handle,
                                           n,
                                           (cuDoubleComplex*)x,
                                           incx,
                                           (cuDoubleComplex*)y,
                                           incy,
                                           c,
                                           (cuDoubleComplex*)s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdrot_v2(hipblasHandle_t   handle,
                                int               n,
                                hipDoubleComplex* x,
                                int               incx,
                                hipDoubleComplex* y,
                                int               incy,
                                const double*     c,
                                const double*     s)
try
{
    return hipblasConvertStatus(cublasZdrot(
        (cublasHandle_t)handle, n, (cuDoubleComplex*)x, incx, (cuDoubleComplex*)y, incy, c, s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rot_64
hipblasStatus_t hipblasSrot_64(hipblasHandle_t handle,
                               int64_t         n,
                               float*          x,
                               int64_t         incx,
                               float*          y,
                               int64_t         incy,
                               const float*    c,
                               const float*    s)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasSrot_64((cublasHandle_t)handle, n, x, incx, y, incy, c, s));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrot_64(hipblasHandle_t handle,
                               int64_t         n,
                               double*         x,
                               int64_t         incx,
                               double*         y,
                               int64_t         incy,
                               const double*   c,
                               const double*   s)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasDrot_64((cublasHandle_t)handle, n, x, incx, y, incy, c, s));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCrot_64(hipblasHandle_t       handle,
                               int64_t               n,
                               hipblasComplex*       x,
                               int64_t               incx,
                               hipblasComplex*       y,
                               int64_t               incy,
                               const float*          c,
                               const hipblasComplex* s)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCrot_64(
        (cublasHandle_t)handle, n, (cuComplex*)x, incx, (cuComplex*)y, incy, c, (cuComplex*)s));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsrot_64(hipblasHandle_t handle,
                                int64_t         n,
                                hipblasComplex* x,
                                int64_t         incx,
                                hipblasComplex* y,
                                int64_t         incy,
                                const float*    c,
                                const float*    s)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasCsrot_64((cublasHandle_t)handle, n, (cuComplex*)x, incx, (cuComplex*)y, incy, c, s));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZrot_64(hipblasHandle_t             handle,
                               int64_t                     n,
                               hipblasDoubleComplex*       x,
                               int64_t                     incx,
                               hipblasDoubleComplex*       y,
                               int64_t                     incy,
                               const double*               c,
                               const hipblasDoubleComplex* s)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZrot_64((cublasHandle_t)handle,
                                              n,
                                              (cuDoubleComplex*)x,
                                              incx,
                                              (cuDoubleComplex*)y,
                                              incy,
                                              c,
                                              (cuDoubleComplex*)s));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdrot_64(hipblasHandle_t       handle,
                                int64_t               n,
                                hipblasDoubleComplex* x,
                                int64_t               incx,
                                hipblasDoubleComplex* y,
                                int64_t               incy,
                                const double*         c,
                                const double*         s)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZdrot_64(
        (cublasHandle_t)handle, n, (cuDoubleComplex*)x, incx, (cuDoubleComplex*)y, incy, c, s));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCrot_v2_64(hipblasHandle_t   handle,
                                  int64_t           n,
                                  hipComplex*       x,
                                  int64_t           incx,
                                  hipComplex*       y,
                                  int64_t           incy,
                                  const float*      c,
                                  const hipComplex* s)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCrot_64(
        (cublasHandle_t)handle, n, (cuComplex*)x, incx, (cuComplex*)y, incy, c, (cuComplex*)s));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsrot_v2_64(hipblasHandle_t handle,
                                   int64_t         n,
                                   hipComplex*     x,
                                   int64_t         incx,
                                   hipComplex*     y,
                                   int64_t         incy,
                                   const float*    c,
                                   const float*    s)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasCsrot_64((cublasHandle_t)handle, n, (cuComplex*)x, incx, (cuComplex*)y, incy, c, s));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZrot_v2_64(hipblasHandle_t         handle,
                                  int64_t                 n,
                                  hipDoubleComplex*       x,
                                  int64_t                 incx,
                                  hipDoubleComplex*       y,
                                  int64_t                 incy,
                                  const double*           c,
                                  const hipDoubleComplex* s)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZrot_64((cublasHandle_t)handle,
                                              n,
                                              (cuDoubleComplex*)x,
                                              incx,
                                              (cuDoubleComplex*)y,
                                              incy,
                                              c,
                                              (cuDoubleComplex*)s));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdrot_v2_64(hipblasHandle_t   handle,
                                   int64_t           n,
                                   hipDoubleComplex* x,
                                   int64_t           incx,
                                   hipDoubleComplex* y,
                                   int64_t           incy,
                                   const double*     c,
                                   const double*     s)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZdrot_64(
        (cublasHandle_t)handle, n, (cuDoubleComplex*)x, incx, (cuDoubleComplex*)y, incy, c, s));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rot_batched
hipblasStatus_t hipblasSrotBatched(hipblasHandle_t handle,
                                   int             n,
                                   float* const    x[],
                                   int             incx,
                                   float* const    y[],
                                   int             incy,
                                   const float*    c,
                                   const float*    s,
                                   int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDrotBatched(hipblasHandle_t handle,
                                   int             n,
                                   double* const   x[],
                                   int             incx,
                                   double* const   y[],
                                   int             incy,
                                   const double*   c,
                                   const double*   s,
                                   int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCrotBatched(hipblasHandle_t       handle,
                                   int                   n,
                                   hipblasComplex* const x[],
                                   int                   incx,
                                   hipblasComplex* const y[],
                                   int                   incy,
                                   const float*          c,
                                   const hipblasComplex* s,
                                   int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsrotBatched(hipblasHandle_t       handle,
                                    int                   n,
                                    hipblasComplex* const x[],
                                    int                   incx,
                                    hipblasComplex* const y[],
                                    int                   incy,
                                    const float*          c,
                                    const float*          s,
                                    int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZrotBatched(hipblasHandle_t             handle,
                                   int                         n,
                                   hipblasDoubleComplex* const x[],
                                   int                         incx,
                                   hipblasDoubleComplex* const y[],
                                   int                         incy,
                                   const double*               c,
                                   const hipblasDoubleComplex* s,
                                   int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdrotBatched(hipblasHandle_t             handle,
                                    int                         n,
                                    hipblasDoubleComplex* const x[],
                                    int                         incx,
                                    hipblasDoubleComplex* const y[],
                                    int                         incy,
                                    const double*               c,
                                    const double*               s,
                                    int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCrotBatched_v2(hipblasHandle_t   handle,
                                      int               n,
                                      hipComplex* const x[],
                                      int               incx,
                                      hipComplex* const y[],
                                      int               incy,
                                      const float*      c,
                                      const hipComplex* s,
                                      int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsrotBatched_v2(hipblasHandle_t   handle,
                                       int               n,
                                       hipComplex* const x[],
                                       int               incx,
                                       hipComplex* const y[],
                                       int               incy,
                                       const float*      c,
                                       const float*      s,
                                       int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZrotBatched_v2(hipblasHandle_t         handle,
                                      int                     n,
                                      hipDoubleComplex* const x[],
                                      int                     incx,
                                      hipDoubleComplex* const y[],
                                      int                     incy,
                                      const double*           c,
                                      const hipDoubleComplex* s,
                                      int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdrotBatched_v2(hipblasHandle_t         handle,
                                       int                     n,
                                       hipDoubleComplex* const x[],
                                       int                     incx,
                                       hipDoubleComplex* const y[],
                                       int                     incy,
                                       const double*           c,
                                       const double*           s,
                                       int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// rot_batched_64
hipblasStatus_t hipblasSrotBatched_64(hipblasHandle_t handle,
                                      int64_t         n,
                                      float* const    x[],
                                      int64_t         incx,
                                      float* const    y[],
                                      int64_t         incy,
                                      const float*    c,
                                      const float*    s,
                                      int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDrotBatched_64(hipblasHandle_t handle,
                                      int64_t         n,
                                      double* const   x[],
                                      int64_t         incx,
                                      double* const   y[],
                                      int64_t         incy,
                                      const double*   c,
                                      const double*   s,
                                      int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCrotBatched_64(hipblasHandle_t       handle,
                                      int64_t               n,
                                      hipblasComplex* const x[],
                                      int64_t               incx,
                                      hipblasComplex* const y[],
                                      int64_t               incy,
                                      const float*          c,
                                      const hipblasComplex* s,
                                      int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsrotBatched_64(hipblasHandle_t       handle,
                                       int64_t               n,
                                       hipblasComplex* const x[],
                                       int64_t               incx,
                                       hipblasComplex* const y[],
                                       int64_t               incy,
                                       const float*          c,
                                       const float*          s,
                                       int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZrotBatched_64(hipblasHandle_t             handle,
                                      int64_t                     n,
                                      hipblasDoubleComplex* const x[],
                                      int64_t                     incx,
                                      hipblasDoubleComplex* const y[],
                                      int64_t                     incy,
                                      const double*               c,
                                      const hipblasDoubleComplex* s,
                                      int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdrotBatched_64(hipblasHandle_t             handle,
                                       int64_t                     n,
                                       hipblasDoubleComplex* const x[],
                                       int64_t                     incx,
                                       hipblasDoubleComplex* const y[],
                                       int64_t                     incy,
                                       const double*               c,
                                       const double*               s,
                                       int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCrotBatched_v2_64(hipblasHandle_t   handle,
                                         int64_t           n,
                                         hipComplex* const x[],
                                         int64_t           incx,
                                         hipComplex* const y[],
                                         int64_t           incy,
                                         const float*      c,
                                         const hipComplex* s,
                                         int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsrotBatched_v2_64(hipblasHandle_t   handle,
                                          int64_t           n,
                                          hipComplex* const x[],
                                          int64_t           incx,
                                          hipComplex* const y[],
                                          int64_t           incy,
                                          const float*      c,
                                          const float*      s,
                                          int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZrotBatched_v2_64(hipblasHandle_t         handle,
                                         int64_t                 n,
                                         hipDoubleComplex* const x[],
                                         int64_t                 incx,
                                         hipDoubleComplex* const y[],
                                         int64_t                 incy,
                                         const double*           c,
                                         const hipDoubleComplex* s,
                                         int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdrotBatched_v2_64(hipblasHandle_t         handle,
                                          int64_t                 n,
                                          hipDoubleComplex* const x[],
                                          int64_t                 incx,
                                          hipDoubleComplex* const y[],
                                          int64_t                 incy,
                                          const double*           c,
                                          const double*           s,
                                          int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// rot_strided_batched
hipblasStatus_t hipblasSrotStridedBatched(hipblasHandle_t handle,
                                          int             n,
                                          float*          x,
                                          int             incx,
                                          hipblasStride   stridex,
                                          float*          y,
                                          int             incy,
                                          hipblasStride   stridey,
                                          const float*    c,
                                          const float*    s,
                                          int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDrotStridedBatched(hipblasHandle_t handle,
                                          int             n,
                                          double*         x,
                                          int             incx,
                                          hipblasStride   stridex,
                                          double*         y,
                                          int             incy,
                                          hipblasStride   stridey,
                                          const double*   c,
                                          const double*   s,
                                          int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCrotStridedBatched(hipblasHandle_t       handle,
                                          int                   n,
                                          hipblasComplex*       x,
                                          int                   incx,
                                          hipblasStride         stridex,
                                          hipblasComplex*       y,
                                          int                   incy,
                                          hipblasStride         stridey,
                                          const float*          c,
                                          const hipblasComplex* s,
                                          int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsrotStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           hipblasComplex* x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           hipblasComplex* y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           const float*    c,
                                           const float*    s,
                                           int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZrotStridedBatched(hipblasHandle_t             handle,
                                          int                         n,
                                          hipblasDoubleComplex*       x,
                                          int                         incx,
                                          hipblasStride               stridex,
                                          hipblasDoubleComplex*       y,
                                          int                         incy,
                                          hipblasStride               stridey,
                                          const double*               c,
                                          const hipblasDoubleComplex* s,
                                          int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdrotStridedBatched(hipblasHandle_t       handle,
                                           int                   n,
                                           hipblasDoubleComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           hipblasDoubleComplex* y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           const double*         c,
                                           const double*         s,
                                           int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCrotStridedBatched_v2(hipblasHandle_t   handle,
                                             int               n,
                                             hipComplex*       x,
                                             int               incx,
                                             hipblasStride     stridex,
                                             hipComplex*       y,
                                             int               incy,
                                             hipblasStride     stridey,
                                             const float*      c,
                                             const hipComplex* s,
                                             int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsrotStridedBatched_v2(hipblasHandle_t handle,
                                              int             n,
                                              hipComplex*     x,
                                              int             incx,
                                              hipblasStride   stridex,
                                              hipComplex*     y,
                                              int             incy,
                                              hipblasStride   stridey,
                                              const float*    c,
                                              const float*    s,
                                              int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZrotStridedBatched_v2(hipblasHandle_t         handle,
                                             int                     n,
                                             hipDoubleComplex*       x,
                                             int                     incx,
                                             hipblasStride           stridex,
                                             hipDoubleComplex*       y,
                                             int                     incy,
                                             hipblasStride           stridey,
                                             const double*           c,
                                             const hipDoubleComplex* s,
                                             int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdrotStridedBatched_v2(hipblasHandle_t   handle,
                                              int               n,
                                              hipDoubleComplex* x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              hipDoubleComplex* y,
                                              int               incy,
                                              hipblasStride     stridey,
                                              const double*     c,
                                              const double*     s,
                                              int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// rot_strided_batched_64
hipblasStatus_t hipblasSrotStridedBatched_64(hipblasHandle_t handle,
                                             int64_t         n,
                                             float*          x,
                                             int64_t         incx,
                                             hipblasStride   stridex,
                                             float*          y,
                                             int64_t         incy,
                                             hipblasStride   stridey,
                                             const float*    c,
                                             const float*    s,
                                             int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDrotStridedBatched_64(hipblasHandle_t handle,
                                             int64_t         n,
                                             double*         x,
                                             int64_t         incx,
                                             hipblasStride   stridex,
                                             double*         y,
                                             int64_t         incy,
                                             hipblasStride   stridey,
                                             const double*   c,
                                             const double*   s,
                                             int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCrotStridedBatched_64(hipblasHandle_t       handle,
                                             int64_t               n,
                                             hipblasComplex*       x,
                                             int64_t               incx,
                                             hipblasStride         stridex,
                                             hipblasComplex*       y,
                                             int64_t               incy,
                                             hipblasStride         stridey,
                                             const float*          c,
                                             const hipblasComplex* s,
                                             int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsrotStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              hipblasComplex* x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              hipblasComplex* y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              const float*    c,
                                              const float*    s,
                                              int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZrotStridedBatched_64(hipblasHandle_t             handle,
                                             int64_t                     n,
                                             hipblasDoubleComplex*       x,
                                             int64_t                     incx,
                                             hipblasStride               stridex,
                                             hipblasDoubleComplex*       y,
                                             int64_t                     incy,
                                             hipblasStride               stridey,
                                             const double*               c,
                                             const hipblasDoubleComplex* s,
                                             int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdrotStridedBatched_64(hipblasHandle_t       handle,
                                              int64_t               n,
                                              hipblasDoubleComplex* x,
                                              int64_t               incx,
                                              hipblasStride         stridex,
                                              hipblasDoubleComplex* y,
                                              int64_t               incy,
                                              hipblasStride         stridey,
                                              const double*         c,
                                              const double*         s,
                                              int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCrotStridedBatched_v2_64(hipblasHandle_t   handle,
                                                int64_t           n,
                                                hipComplex*       x,
                                                int64_t           incx,
                                                hipblasStride     stridex,
                                                hipComplex*       y,
                                                int64_t           incy,
                                                hipblasStride     stridey,
                                                const float*      c,
                                                const hipComplex* s,
                                                int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsrotStridedBatched_v2_64(hipblasHandle_t handle,
                                                 int64_t         n,
                                                 hipComplex*     x,
                                                 int64_t         incx,
                                                 hipblasStride   stridex,
                                                 hipComplex*     y,
                                                 int64_t         incy,
                                                 hipblasStride   stridey,
                                                 const float*    c,
                                                 const float*    s,
                                                 int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZrotStridedBatched_v2_64(hipblasHandle_t         handle,
                                                int64_t                 n,
                                                hipDoubleComplex*       x,
                                                int64_t                 incx,
                                                hipblasStride           stridex,
                                                hipDoubleComplex*       y,
                                                int64_t                 incy,
                                                hipblasStride           stridey,
                                                const double*           c,
                                                const hipDoubleComplex* s,
                                                int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdrotStridedBatched_v2_64(hipblasHandle_t   handle,
                                                 int64_t           n,
                                                 hipDoubleComplex* x,
                                                 int64_t           incx,
                                                 hipblasStride     stridex,
                                                 hipDoubleComplex* y,
                                                 int64_t           incy,
                                                 hipblasStride     stridey,
                                                 const double*     c,
                                                 const double*     s,
                                                 int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// rotg
hipblasStatus_t hipblasSrotg(hipblasHandle_t handle, float* a, float* b, float* c, float* s)
try
{
    return hipblasConvertStatus(cublasSrotg((cublasHandle_t)handle, a, b, c, s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrotg(hipblasHandle_t handle, double* a, double* b, double* c, double* s)
try
{
    return hipblasConvertStatus(cublasDrotg((cublasHandle_t)handle, a, b, c, s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCrotg(
    hipblasHandle_t handle, hipblasComplex* a, hipblasComplex* b, float* c, hipblasComplex* s)
try
{
    return hipblasConvertStatus(
        cublasCrotg((cublasHandle_t)handle, (cuComplex*)a, (cuComplex*)b, c, (cuComplex*)s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZrotg(hipblasHandle_t       handle,
                             hipblasDoubleComplex* a,
                             hipblasDoubleComplex* b,
                             double*               c,
                             hipblasDoubleComplex* s)
try
{
    return hipblasConvertStatus(cublasZrotg(
        (cublasHandle_t)handle, (cuDoubleComplex*)a, (cuDoubleComplex*)b, c, (cuDoubleComplex*)s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasCrotg_v2(hipblasHandle_t handle, hipComplex* a, hipComplex* b, float* c, hipComplex* s)
try
{
    return hipblasConvertStatus(
        cublasCrotg((cublasHandle_t)handle, (cuComplex*)a, (cuComplex*)b, c, (cuComplex*)s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZrotg_v2(hipblasHandle_t   handle,
                                hipDoubleComplex* a,
                                hipDoubleComplex* b,
                                double*           c,
                                hipDoubleComplex* s)
try
{
    return hipblasConvertStatus(cublasZrotg(
        (cublasHandle_t)handle, (cuDoubleComplex*)a, (cuDoubleComplex*)b, c, (cuDoubleComplex*)s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rotg
hipblasStatus_t hipblasSrotg_64(hipblasHandle_t handle, float* a, float* b, float* c, float* s)
try
{
    // _64 interface is same as regular interface
    return hipblasConvertStatus(cublasSrotg((cublasHandle_t)handle, a, b, c, s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrotg_64(hipblasHandle_t handle, double* a, double* b, double* c, double* s)
try
{
    return hipblasConvertStatus(cublasDrotg((cublasHandle_t)handle, a, b, c, s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCrotg_64(
    hipblasHandle_t handle, hipblasComplex* a, hipblasComplex* b, float* c, hipblasComplex* s)
try
{
    return hipblasConvertStatus(
        cublasCrotg((cublasHandle_t)handle, (cuComplex*)a, (cuComplex*)b, c, (cuComplex*)s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZrotg_64(hipblasHandle_t       handle,
                                hipblasDoubleComplex* a,
                                hipblasDoubleComplex* b,
                                double*               c,
                                hipblasDoubleComplex* s)
try
{
    return hipblasConvertStatus(cublasZrotg(
        (cublasHandle_t)handle, (cuDoubleComplex*)a, (cuDoubleComplex*)b, c, (cuDoubleComplex*)s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCrotg_v2_64(
    hipblasHandle_t handle, hipComplex* a, hipComplex* b, float* c, hipComplex* s)
try
{
    return hipblasConvertStatus(
        cublasCrotg((cublasHandle_t)handle, (cuComplex*)a, (cuComplex*)b, c, (cuComplex*)s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZrotg_v2_64(hipblasHandle_t   handle,
                                   hipDoubleComplex* a,
                                   hipDoubleComplex* b,
                                   double*           c,
                                   hipDoubleComplex* s)
try
{
    return hipblasConvertStatus(cublasZrotg(
        (cublasHandle_t)handle, (cuDoubleComplex*)a, (cuDoubleComplex*)b, c, (cuDoubleComplex*)s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rotg_batchced
hipblasStatus_t hipblasSrotgBatched(hipblasHandle_t handle,
                                    float* const    a[],
                                    float* const    b[],
                                    float* const    c[],
                                    float* const    s[],
                                    int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDrotgBatched(hipblasHandle_t handle,
                                    double* const   a[],
                                    double* const   b[],
                                    double* const   c[],
                                    double* const   s[],
                                    int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCrotgBatched(hipblasHandle_t       handle,
                                    hipblasComplex* const a[],
                                    hipblasComplex* const b[],
                                    float* const          c[],
                                    hipblasComplex* const s[],
                                    int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZrotgBatched(hipblasHandle_t             handle,
                                    hipblasDoubleComplex* const a[],
                                    hipblasDoubleComplex* const b[],
                                    double* const               c[],
                                    hipblasDoubleComplex* const s[],
                                    int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCrotgBatched_v2(hipblasHandle_t   handle,
                                       hipComplex* const a[],
                                       hipComplex* const b[],
                                       float* const      c[],
                                       hipComplex* const s[],
                                       int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZrotgBatched_v2(hipblasHandle_t         handle,
                                       hipDoubleComplex* const a[],
                                       hipDoubleComplex* const b[],
                                       double* const           c[],
                                       hipDoubleComplex* const s[],
                                       int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// rotg_batchced_64
hipblasStatus_t hipblasSrotgBatched_64(hipblasHandle_t handle,
                                       float* const    a[],
                                       float* const    b[],
                                       float* const    c[],
                                       float* const    s[],
                                       int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDrotgBatched_64(hipblasHandle_t handle,
                                       double* const   a[],
                                       double* const   b[],
                                       double* const   c[],
                                       double* const   s[],
                                       int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCrotgBatched_64(hipblasHandle_t       handle,
                                       hipblasComplex* const a[],
                                       hipblasComplex* const b[],
                                       float* const          c[],
                                       hipblasComplex* const s[],
                                       int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZrotgBatched_64(hipblasHandle_t             handle,
                                       hipblasDoubleComplex* const a[],
                                       hipblasDoubleComplex* const b[],
                                       double* const               c[],
                                       hipblasDoubleComplex* const s[],
                                       int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCrotgBatched_v2_64(hipblasHandle_t   handle,
                                          hipComplex* const a[],
                                          hipComplex* const b[],
                                          float* const      c[],
                                          hipComplex* const s[],
                                          int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZrotgBatched_v2_64(hipblasHandle_t         handle,
                                          hipDoubleComplex* const a[],
                                          hipDoubleComplex* const b[],
                                          double* const           c[],
                                          hipDoubleComplex* const s[],
                                          int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// rotg_strided_batched
hipblasStatus_t hipblasSrotgStridedBatched(hipblasHandle_t handle,
                                           float*          a,
                                           hipblasStride   stride_a,
                                           float*          b,
                                           hipblasStride   stride_b,
                                           float*          c,
                                           hipblasStride   stride_c,
                                           float*          s,
                                           hipblasStride   stride_s,
                                           int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDrotgStridedBatched(hipblasHandle_t handle,
                                           double*         a,
                                           hipblasStride   stride_a,
                                           double*         b,
                                           hipblasStride   stride_b,
                                           double*         c,
                                           hipblasStride   stride_c,
                                           double*         s,
                                           hipblasStride   stride_s,
                                           int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCrotgStridedBatched(hipblasHandle_t handle,
                                           hipblasComplex* a,
                                           hipblasStride   stride_a,
                                           hipblasComplex* b,
                                           hipblasStride   stride_b,
                                           float*          c,
                                           hipblasStride   stride_c,
                                           hipblasComplex* s,
                                           hipblasStride   stride_s,
                                           int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZrotgStridedBatched(hipblasHandle_t       handle,
                                           hipblasDoubleComplex* a,
                                           hipblasStride         stride_a,
                                           hipblasDoubleComplex* b,
                                           hipblasStride         stride_b,
                                           double*               c,
                                           hipblasStride         stride_c,
                                           hipblasDoubleComplex* s,
                                           hipblasStride         stride_s,
                                           int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCrotgStridedBatched_v2(hipblasHandle_t handle,
                                              hipComplex*     a,
                                              hipblasStride   stride_a,
                                              hipComplex*     b,
                                              hipblasStride   stride_b,
                                              float*          c,
                                              hipblasStride   stride_c,
                                              hipComplex*     s,
                                              hipblasStride   stride_s,
                                              int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZrotgStridedBatched_v2(hipblasHandle_t   handle,
                                              hipDoubleComplex* a,
                                              hipblasStride     stride_a,
                                              hipDoubleComplex* b,
                                              hipblasStride     stride_b,
                                              double*           c,
                                              hipblasStride     stride_c,
                                              hipDoubleComplex* s,
                                              hipblasStride     stride_s,
                                              int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// rotg_strided_batched_64
hipblasStatus_t hipblasSrotgStridedBatched_64(hipblasHandle_t handle,
                                              float*          a,
                                              hipblasStride   stride_a,
                                              float*          b,
                                              hipblasStride   stride_b,
                                              float*          c,
                                              hipblasStride   stride_c,
                                              float*          s,
                                              hipblasStride   stride_s,
                                              int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDrotgStridedBatched_64(hipblasHandle_t handle,
                                              double*         a,
                                              hipblasStride   stride_a,
                                              double*         b,
                                              hipblasStride   stride_b,
                                              double*         c,
                                              hipblasStride   stride_c,
                                              double*         s,
                                              hipblasStride   stride_s,
                                              int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCrotgStridedBatched_64(hipblasHandle_t handle,
                                              hipblasComplex* a,
                                              hipblasStride   stride_a,
                                              hipblasComplex* b,
                                              hipblasStride   stride_b,
                                              float*          c,
                                              hipblasStride   stride_c,
                                              hipblasComplex* s,
                                              hipblasStride   stride_s,
                                              int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZrotgStridedBatched_64(hipblasHandle_t       handle,
                                              hipblasDoubleComplex* a,
                                              hipblasStride         stride_a,
                                              hipblasDoubleComplex* b,
                                              hipblasStride         stride_b,
                                              double*               c,
                                              hipblasStride         stride_c,
                                              hipblasDoubleComplex* s,
                                              hipblasStride         stride_s,
                                              int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCrotgStridedBatched_v2_64(hipblasHandle_t handle,
                                                 hipComplex*     a,
                                                 hipblasStride   stride_a,
                                                 hipComplex*     b,
                                                 hipblasStride   stride_b,
                                                 float*          c,
                                                 hipblasStride   stride_c,
                                                 hipComplex*     s,
                                                 hipblasStride   stride_s,
                                                 int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZrotgStridedBatched_v2_64(hipblasHandle_t   handle,
                                                 hipDoubleComplex* a,
                                                 hipblasStride     stride_a,
                                                 hipDoubleComplex* b,
                                                 hipblasStride     stride_b,
                                                 double*           c,
                                                 hipblasStride     stride_c,
                                                 hipDoubleComplex* s,
                                                 hipblasStride     stride_s,
                                                 int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// rotm
hipblasStatus_t hipblasSrotm(
    hipblasHandle_t handle, int n, float* x, int incx, float* y, int incy, const float* param)
try
{
    return hipblasConvertStatus(cublasSrotm((cublasHandle_t)handle, n, x, incx, y, incy, param));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrotm(
    hipblasHandle_t handle, int n, double* x, int incx, double* y, int incy, const double* param)
try
{
    return hipblasConvertStatus(cublasDrotm((cublasHandle_t)handle, n, x, incx, y, incy, param));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rotm_64
hipblasStatus_t hipblasSrotm_64(hipblasHandle_t handle,
                                int64_t         n,
                                float*          x,
                                int64_t         incx,
                                float*          y,
                                int64_t         incy,
                                const float*    param)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasSrotm_64((cublasHandle_t)handle, n, x, incx, y, incy, param));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrotm_64(hipblasHandle_t handle,
                                int64_t         n,
                                double*         x,
                                int64_t         incx,
                                double*         y,
                                int64_t         incy,
                                const double*   param)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasDrotm_64((cublasHandle_t)handle, n, x, incx, y, incy, param));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rotm_batched
hipblasStatus_t hipblasSrotmBatched(hipblasHandle_t    handle,
                                    int                n,
                                    float* const       x[],
                                    int                incx,
                                    float* const       y[],
                                    int                incy,
                                    const float* const param[],
                                    int                batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDrotmBatched(hipblasHandle_t     handle,
                                    int                 n,
                                    double* const       x[],
                                    int                 incx,
                                    double* const       y[],
                                    int                 incy,
                                    const double* const param[],
                                    int                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// rotm_batched_64
hipblasStatus_t hipblasSrotmBatched_64(hipblasHandle_t    handle,
                                       int64_t            n,
                                       float* const       x[],
                                       int64_t            incx,
                                       float* const       y[],
                                       int64_t            incy,
                                       const float* const param[],
                                       int64_t            batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDrotmBatched_64(hipblasHandle_t     handle,
                                       int64_t             n,
                                       double* const       x[],
                                       int64_t             incx,
                                       double* const       y[],
                                       int64_t             incy,
                                       const double* const param[],
                                       int64_t             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// rotm_strided_batched
hipblasStatus_t hipblasSrotmStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           float*          x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           float*          y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           const float*    param,
                                           hipblasStride   strideParam,
                                           int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDrotmStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           double*         x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           double*         y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           const double*   param,
                                           hipblasStride   strideParam,
                                           int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// rotm_strided_batched_64
hipblasStatus_t hipblasSrotmStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              float*          x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              float*          y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              const float*    param,
                                              hipblasStride   strideParam,
                                              int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDrotmStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              double*         x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              double*         y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              const double*   param,
                                              hipblasStride   strideParam,
                                              int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// rotmg
hipblasStatus_t hipblasSrotmg(
    hipblasHandle_t handle, float* d1, float* d2, float* x1, const float* y1, float* param)
try
{
    return hipblasConvertStatus(cublasSrotmg((cublasHandle_t)handle, d1, d2, x1, y1, param));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrotmg(
    hipblasHandle_t handle, double* d1, double* d2, double* x1, const double* y1, double* param)
try
{
    return hipblasConvertStatus(cublasDrotmg((cublasHandle_t)handle, d1, d2, x1, y1, param));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rotmg_64
hipblasStatus_t hipblasSrotmg_64(
    hipblasHandle_t handle, float* d1, float* d2, float* x1, const float* y1, float* param)
try
{
    // _64 interface is same as regular interface
    return hipblasConvertStatus(cublasSrotmg((cublasHandle_t)handle, d1, d2, x1, y1, param));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrotmg_64(
    hipblasHandle_t handle, double* d1, double* d2, double* x1, const double* y1, double* param)
try
{
    return hipblasConvertStatus(cublasDrotmg((cublasHandle_t)handle, d1, d2, x1, y1, param));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rotmg_batched
hipblasStatus_t hipblasSrotmgBatched(hipblasHandle_t    handle,
                                     float* const       d1[],
                                     float* const       d2[],
                                     float* const       x1[],
                                     const float* const y1[],
                                     float* const       param[],
                                     int                batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDrotmgBatched(hipblasHandle_t     handle,
                                     double* const       d1[],
                                     double* const       d2[],
                                     double* const       x1[],
                                     const double* const y1[],
                                     double* const       param[],
                                     int                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// rotmg_batched_64
hipblasStatus_t hipblasSrotmgBatched_64(hipblasHandle_t    handle,
                                        float* const       d1[],
                                        float* const       d2[],
                                        float* const       x1[],
                                        const float* const y1[],
                                        float* const       param[],
                                        int64_t            batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDrotmgBatched_64(hipblasHandle_t     handle,
                                        double* const       d1[],
                                        double* const       d2[],
                                        double* const       x1[],
                                        const double* const y1[],
                                        double* const       param[],
                                        int64_t             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// rotmg_strided_batched
hipblasStatus_t hipblasSrotmgStridedBatched(hipblasHandle_t handle,
                                            float*          d1,
                                            hipblasStride   stride_d1,
                                            float*          d2,
                                            hipblasStride   stride_d2,
                                            float*          x1,
                                            hipblasStride   stride_x1,
                                            const float*    y1,
                                            hipblasStride   stride_y1,
                                            float*          param,
                                            hipblasStride   strideParam,
                                            int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDrotmgStridedBatched(hipblasHandle_t handle,
                                            double*         d1,
                                            hipblasStride   stride_d1,
                                            double*         d2,
                                            hipblasStride   stride_d2,
                                            double*         x1,
                                            hipblasStride   stride_x1,
                                            const double*   y1,
                                            hipblasStride   stride_y1,
                                            double*         param,
                                            hipblasStride   strideParam,
                                            int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// rotmg_strided_batched_64
hipblasStatus_t hipblasSrotmgStridedBatched_64(hipblasHandle_t handle,
                                               float*          d1,
                                               hipblasStride   stride_d1,
                                               float*          d2,
                                               hipblasStride   stride_d2,
                                               float*          x1,
                                               hipblasStride   stride_x1,
                                               const float*    y1,
                                               hipblasStride   stride_y1,
                                               float*          param,
                                               hipblasStride   strideParam,
                                               int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDrotmgStridedBatched_64(hipblasHandle_t handle,
                                               double*         d1,
                                               hipblasStride   stride_d1,
                                               double*         d2,
                                               hipblasStride   stride_d2,
                                               double*         x1,
                                               hipblasStride   stride_x1,
                                               const double*   y1,
                                               hipblasStride   stride_y1,
                                               double*         param,
                                               hipblasStride   strideParam,
                                               int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// scal
hipblasStatus_t hipblasSscal(hipblasHandle_t handle, int n, const float* alpha, float* x, int incx)
try
{
    return hipblasConvertStatus(cublasSscal((cublasHandle_t)handle, n, alpha, x, incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasDscal(hipblasHandle_t handle, int n, const double* alpha, double* x, int incx)
try
{
    return hipblasConvertStatus(cublasDscal((cublasHandle_t)handle, n, alpha, x, incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCscal(
    hipblasHandle_t handle, int n, const hipblasComplex* alpha, hipblasComplex* x, int incx)
try
{
    return hipblasConvertStatus(
        cublasCscal((cublasHandle_t)handle, n, (cuComplex*)alpha, (cuComplex*)x, incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasCsscal(hipblasHandle_t handle, int n, const float* alpha, hipblasComplex* x, int incx)
try
{
    return hipblasConvertStatus(
        cublasCsscal((cublasHandle_t)handle, n, alpha, (cuComplex*)x, incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZscal(hipblasHandle_t             handle,
                             int                         n,
                             const hipblasDoubleComplex* alpha,
                             hipblasDoubleComplex*       x,
                             int                         incx)
try
{
    return hipblasConvertStatus(
        cublasZscal((cublasHandle_t)handle, n, (cuDoubleComplex*)alpha, (cuDoubleComplex*)x, incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdscal(
    hipblasHandle_t handle, int n, const double* alpha, hipblasDoubleComplex* x, int incx)
try
{
    return hipblasConvertStatus(
        cublasZdscal((cublasHandle_t)handle, n, alpha, (cuDoubleComplex*)x, incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasCscal_v2(hipblasHandle_t handle, int n, const hipComplex* alpha, hipComplex* x, int incx)
try
{
    return hipblasConvertStatus(
        cublasCscal((cublasHandle_t)handle, n, (cuComplex*)alpha, (cuComplex*)x, incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasCsscal_v2(hipblasHandle_t handle, int n, const float* alpha, hipComplex* x, int incx)
try
{
    return hipblasConvertStatus(
        cublasCsscal((cublasHandle_t)handle, n, alpha, (cuComplex*)x, incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZscal_v2(
    hipblasHandle_t handle, int n, const hipDoubleComplex* alpha, hipDoubleComplex* x, int incx)
try
{
    return hipblasConvertStatus(
        cublasZscal((cublasHandle_t)handle, n, (cuDoubleComplex*)alpha, (cuDoubleComplex*)x, incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdscal_v2(
    hipblasHandle_t handle, int n, const double* alpha, hipDoubleComplex* x, int incx)
try
{
    return hipblasConvertStatus(
        cublasZdscal((cublasHandle_t)handle, n, alpha, (cuDoubleComplex*)x, incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// scal_64
hipblasStatus_t
    hipblasSscal_64(hipblasHandle_t handle, int64_t n, const float* alpha, float* x, int64_t incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasSscal_64((cublasHandle_t)handle, n, alpha, x, incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasDscal_64(hipblasHandle_t handle, int64_t n, const double* alpha, double* x, int64_t incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasDscal_64((cublasHandle_t)handle, n, alpha, x, incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCscal_64(
    hipblasHandle_t handle, int64_t n, const hipblasComplex* alpha, hipblasComplex* x, int64_t incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasCscal_64((cublasHandle_t)handle, n, (cuComplex*)alpha, (cuComplex*)x, incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsscal_64(
    hipblasHandle_t handle, int64_t n, const float* alpha, hipblasComplex* x, int64_t incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasCsscal_64((cublasHandle_t)handle, n, alpha, (cuComplex*)x, incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZscal_64(hipblasHandle_t             handle,
                                int64_t                     n,
                                const hipblasDoubleComplex* alpha,
                                hipblasDoubleComplex*       x,
                                int64_t                     incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZscal_64(
        (cublasHandle_t)handle, n, (cuDoubleComplex*)alpha, (cuDoubleComplex*)x, incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdscal_64(
    hipblasHandle_t handle, int64_t n, const double* alpha, hipblasDoubleComplex* x, int64_t incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasZdscal_64((cublasHandle_t)handle, n, alpha, (cuDoubleComplex*)x, incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCscal_v2_64(
    hipblasHandle_t handle, int64_t n, const hipComplex* alpha, hipComplex* x, int64_t incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasCscal_64((cublasHandle_t)handle, n, (cuComplex*)alpha, (cuComplex*)x, incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsscal_v2_64(
    hipblasHandle_t handle, int64_t n, const float* alpha, hipComplex* x, int64_t incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasCsscal_64((cublasHandle_t)handle, n, alpha, (cuComplex*)x, incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZscal_v2_64(hipblasHandle_t         handle,
                                   int64_t                 n,
                                   const hipDoubleComplex* alpha,
                                   hipDoubleComplex*       x,
                                   int64_t                 incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZscal_64(
        (cublasHandle_t)handle, n, (cuDoubleComplex*)alpha, (cuDoubleComplex*)x, incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdscal_v2_64(
    hipblasHandle_t handle, int64_t n, const double* alpha, hipDoubleComplex* x, int64_t incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasZdscal_64((cublasHandle_t)handle, n, alpha, (cuDoubleComplex*)x, incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// scal_batched
hipblasStatus_t hipblasSscalBatched(
    hipblasHandle_t handle, int n, const float* alpha, float* const x[], int incx, int batchCount)
try
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
    // TODO warn user that function was demoted to ignore batch
    // return hipblasConvertStatus(cublasSscal((cublasHandle_t)handle, n, alpha, x, incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}
hipblasStatus_t hipblasDscalBatched(
    hipblasHandle_t handle, int n, const double* alpha, double* const x[], int incx, int batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCscalBatched(hipblasHandle_t       handle,
                                    int                   n,
                                    const hipblasComplex* alpha,
                                    hipblasComplex* const x[],
                                    int                   incx,
                                    int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZscalBatched(hipblasHandle_t             handle,
                                    int                         n,
                                    const hipblasDoubleComplex* alpha,
                                    hipblasDoubleComplex* const x[],
                                    int                         incx,
                                    int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsscalBatched(hipblasHandle_t       handle,
                                     int                   n,
                                     const float*          alpha,
                                     hipblasComplex* const x[],
                                     int                   incx,
                                     int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdscalBatched(hipblasHandle_t             handle,
                                     int                         n,
                                     const double*               alpha,
                                     hipblasDoubleComplex* const x[],
                                     int                         incx,
                                     int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCscalBatched_v2(hipblasHandle_t   handle,
                                       int               n,
                                       const hipComplex* alpha,
                                       hipComplex* const x[],
                                       int               incx,
                                       int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZscalBatched_v2(hipblasHandle_t         handle,
                                       int                     n,
                                       const hipDoubleComplex* alpha,
                                       hipDoubleComplex* const x[],
                                       int                     incx,
                                       int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsscalBatched_v2(hipblasHandle_t   handle,
                                        int               n,
                                        const float*      alpha,
                                        hipComplex* const x[],
                                        int               incx,
                                        int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdscalBatched_v2(hipblasHandle_t         handle,
                                        int                     n,
                                        const double*           alpha,
                                        hipDoubleComplex* const x[],
                                        int                     incx,
                                        int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// scal_batched_64
hipblasStatus_t hipblasSscalBatched_64(hipblasHandle_t handle,
                                       int64_t         n,
                                       const float*    alpha,
                                       float* const    x[],
                                       int64_t         incx,
                                       int64_t         batchCount)
try
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}
catch(...)
{
    return hipblas_exception_to_status();
}
hipblasStatus_t hipblasDscalBatched_64(hipblasHandle_t handle,
                                       int64_t         n,
                                       const double*   alpha,
                                       double* const   x[],
                                       int64_t         incx,
                                       int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCscalBatched_64(hipblasHandle_t       handle,
                                       int64_t               n,
                                       const hipblasComplex* alpha,
                                       hipblasComplex* const x[],
                                       int64_t               incx,
                                       int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZscalBatched_64(hipblasHandle_t             handle,
                                       int64_t                     n,
                                       const hipblasDoubleComplex* alpha,
                                       hipblasDoubleComplex* const x[],
                                       int64_t                     incx,
                                       int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsscalBatched_64(hipblasHandle_t       handle,
                                        int64_t               n,
                                        const float*          alpha,
                                        hipblasComplex* const x[],
                                        int64_t               incx,
                                        int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdscalBatched_64(hipblasHandle_t             handle,
                                        int64_t                     n,
                                        const double*               alpha,
                                        hipblasDoubleComplex* const x[],
                                        int64_t                     incx,
                                        int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCscalBatched_v2_64(hipblasHandle_t   handle,
                                          int64_t           n,
                                          const hipComplex* alpha,
                                          hipComplex* const x[],
                                          int64_t           incx,
                                          int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZscalBatched_v2_64(hipblasHandle_t         handle,
                                          int64_t                 n,
                                          const hipDoubleComplex* alpha,
                                          hipDoubleComplex* const x[],
                                          int64_t                 incx,
                                          int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsscalBatched_v2_64(hipblasHandle_t   handle,
                                           int64_t           n,
                                           const float*      alpha,
                                           hipComplex* const x[],
                                           int64_t           incx,
                                           int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdscalBatched_v2_64(hipblasHandle_t         handle,
                                           int64_t                 n,
                                           const double*           alpha,
                                           hipDoubleComplex* const x[],
                                           int64_t                 incx,
                                           int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// scal_strided_batched
hipblasStatus_t hipblasSscalStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const float*    alpha,
                                           float*          x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDscalStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const double*   alpha,
                                           double*         x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCscalStridedBatched(hipblasHandle_t       handle,
                                           int                   n,
                                           const hipblasComplex* alpha,
                                           hipblasComplex*       x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZscalStridedBatched(hipblasHandle_t             handle,
                                           int                         n,
                                           const hipblasDoubleComplex* alpha,
                                           hipblasDoubleComplex*       x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsscalStridedBatched(hipblasHandle_t handle,
                                            int             n,
                                            const float*    alpha,
                                            hipblasComplex* x,
                                            int             incx,
                                            hipblasStride   stridex,
                                            int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdscalStridedBatched(hipblasHandle_t       handle,
                                            int                   n,
                                            const double*         alpha,
                                            hipblasDoubleComplex* x,
                                            int                   incx,
                                            hipblasStride         stridex,
                                            int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCscalStridedBatched_v2(hipblasHandle_t   handle,
                                              int               n,
                                              const hipComplex* alpha,
                                              hipComplex*       x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZscalStridedBatched_v2(hipblasHandle_t         handle,
                                              int                     n,
                                              const hipDoubleComplex* alpha,
                                              hipDoubleComplex*       x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsscalStridedBatched_v2(hipblasHandle_t handle,
                                               int             n,
                                               const float*    alpha,
                                               hipComplex*     x,
                                               int             incx,
                                               hipblasStride   stridex,
                                               int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdscalStridedBatched_v2(hipblasHandle_t   handle,
                                               int               n,
                                               const double*     alpha,
                                               hipDoubleComplex* x,
                                               int               incx,
                                               hipblasStride     stridex,
                                               int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// scal_strided_batched_64
hipblasStatus_t hipblasSscalStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const float*    alpha,
                                              float*          x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDscalStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const double*   alpha,
                                              double*         x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCscalStridedBatched_64(hipblasHandle_t       handle,
                                              int64_t               n,
                                              const hipblasComplex* alpha,
                                              hipblasComplex*       x,
                                              int64_t               incx,
                                              hipblasStride         stridex,
                                              int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZscalStridedBatched_64(hipblasHandle_t             handle,
                                              int64_t                     n,
                                              const hipblasDoubleComplex* alpha,
                                              hipblasDoubleComplex*       x,
                                              int64_t                     incx,
                                              hipblasStride               stridex,
                                              int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsscalStridedBatched_64(hipblasHandle_t handle,
                                               int64_t         n,
                                               const float*    alpha,
                                               hipblasComplex* x,
                                               int64_t         incx,
                                               hipblasStride   stridex,
                                               int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdscalStridedBatched_64(hipblasHandle_t       handle,
                                               int64_t               n,
                                               const double*         alpha,
                                               hipblasDoubleComplex* x,
                                               int64_t               incx,
                                               hipblasStride         stridex,
                                               int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCscalStridedBatched_v2_64(hipblasHandle_t   handle,
                                                 int64_t           n,
                                                 const hipComplex* alpha,
                                                 hipComplex*       x,
                                                 int64_t           incx,
                                                 hipblasStride     stridex,
                                                 int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZscalStridedBatched_v2_64(hipblasHandle_t         handle,
                                                 int64_t                 n,
                                                 const hipDoubleComplex* alpha,
                                                 hipDoubleComplex*       x,
                                                 int64_t                 incx,
                                                 hipblasStride           stridex,
                                                 int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsscalStridedBatched_v2_64(hipblasHandle_t handle,
                                                  int64_t         n,
                                                  const float*    alpha,
                                                  hipComplex*     x,
                                                  int64_t         incx,
                                                  hipblasStride   stridex,
                                                  int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZdscalStridedBatched_v2_64(hipblasHandle_t   handle,
                                                  int64_t           n,
                                                  const double*     alpha,
                                                  hipDoubleComplex* x,
                                                  int64_t           incx,
                                                  hipblasStride     stridex,
                                                  int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// swap
hipblasStatus_t hipblasSswap(hipblasHandle_t handle, int n, float* x, int incx, float* y, int incy)
try
{
    return hipblasConvertStatus(cublasSswap((cublasHandle_t)handle, n, x, incx, y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasDswap(hipblasHandle_t handle, int n, double* x, int incx, double* y, int incy)
try
{
    return hipblasConvertStatus(cublasDswap((cublasHandle_t)handle, n, x, incx, y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCswap(
    hipblasHandle_t handle, int n, hipblasComplex* x, int incx, hipblasComplex* y, int incy)
try
{
    return hipblasConvertStatus(
        cublasCswap((cublasHandle_t)handle, n, (cuComplex*)x, incx, (cuComplex*)y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZswap(hipblasHandle_t       handle,
                             int                   n,
                             hipblasDoubleComplex* x,
                             int                   incx,
                             hipblasDoubleComplex* y,
                             int                   incy)
try
{
    return hipblasConvertStatus(cublasZswap(
        (cublasHandle_t)handle, n, (cuDoubleComplex*)x, incx, (cuDoubleComplex*)y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasCswap_v2(hipblasHandle_t handle, int n, hipComplex* x, int incx, hipComplex* y, int incy)
try
{
    return hipblasConvertStatus(
        cublasCswap((cublasHandle_t)handle, n, (cuComplex*)x, incx, (cuComplex*)y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZswap_v2(
    hipblasHandle_t handle, int n, hipDoubleComplex* x, int incx, hipDoubleComplex* y, int incy)
try
{
    return hipblasConvertStatus(cublasZswap(
        (cublasHandle_t)handle, n, (cuDoubleComplex*)x, incx, (cuDoubleComplex*)y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// swap_64
hipblasStatus_t hipblasSswap_64(
    hipblasHandle_t handle, int64_t n, float* x, int64_t incx, float* y, int64_t incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasSswap_64((cublasHandle_t)handle, n, x, incx, y, incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDswap_64(
    hipblasHandle_t handle, int64_t n, double* x, int64_t incx, double* y, int64_t incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasDswap_64((cublasHandle_t)handle, n, x, incx, y, incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCswap_64(hipblasHandle_t handle,
                                int64_t         n,
                                hipblasComplex* x,
                                int64_t         incx,
                                hipblasComplex* y,
                                int64_t         incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasCswap_64((cublasHandle_t)handle, n, (cuComplex*)x, incx, (cuComplex*)y, incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZswap_64(hipblasHandle_t       handle,
                                int64_t               n,
                                hipblasDoubleComplex* x,
                                int64_t               incx,
                                hipblasDoubleComplex* y,
                                int64_t               incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZswap_64(
        (cublasHandle_t)handle, n, (cuDoubleComplex*)x, incx, (cuDoubleComplex*)y, incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCswap_v2_64(
    hipblasHandle_t handle, int64_t n, hipComplex* x, int64_t incx, hipComplex* y, int64_t incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasCswap_64((cublasHandle_t)handle, n, (cuComplex*)x, incx, (cuComplex*)y, incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZswap_v2_64(hipblasHandle_t   handle,
                                   int64_t           n,
                                   hipDoubleComplex* x,
                                   int64_t           incx,
                                   hipDoubleComplex* y,
                                   int64_t           incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZswap_64(
        (cublasHandle_t)handle, n, (cuDoubleComplex*)x, incx, (cuDoubleComplex*)y, incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// swap_batched
hipblasStatus_t hipblasSswapBatched(hipblasHandle_t handle,
                                    int             n,
                                    float* const    x[],
                                    int             incx,
                                    float* const    y[],
                                    int             incy,
                                    int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDswapBatched(hipblasHandle_t handle,
                                    int             n,
                                    double* const   x[],
                                    int             incx,
                                    double* const   y[],
                                    int             incy,
                                    int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCswapBatched(hipblasHandle_t       handle,
                                    int                   n,
                                    hipblasComplex* const x[],
                                    int                   incx,
                                    hipblasComplex* const y[],
                                    int                   incy,
                                    int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZswapBatched(hipblasHandle_t             handle,
                                    int                         n,
                                    hipblasDoubleComplex* const x[],
                                    int                         incx,
                                    hipblasDoubleComplex* const y[],
                                    int                         incy,
                                    int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCswapBatched_v2(hipblasHandle_t   handle,
                                       int               n,
                                       hipComplex* const x[],
                                       int               incx,
                                       hipComplex* const y[],
                                       int               incy,
                                       int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZswapBatched_v2(hipblasHandle_t         handle,
                                       int                     n,
                                       hipDoubleComplex* const x[],
                                       int                     incx,
                                       hipDoubleComplex* const y[],
                                       int                     incy,
                                       int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// swap_batched_64
hipblasStatus_t hipblasSswapBatched_64(hipblasHandle_t handle,
                                       int64_t         n,
                                       float* const    x[],
                                       int64_t         incx,
                                       float* const    y[],
                                       int64_t         incy,
                                       int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDswapBatched_64(hipblasHandle_t handle,
                                       int64_t         n,
                                       double* const   x[],
                                       int64_t         incx,
                                       double* const   y[],
                                       int64_t         incy,
                                       int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCswapBatched_64(hipblasHandle_t       handle,
                                       int64_t               n,
                                       hipblasComplex* const x[],
                                       int64_t               incx,
                                       hipblasComplex* const y[],
                                       int64_t               incy,
                                       int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZswapBatched_64(hipblasHandle_t             handle,
                                       int64_t                     n,
                                       hipblasDoubleComplex* const x[],
                                       int64_t                     incx,
                                       hipblasDoubleComplex* const y[],
                                       int64_t                     incy,
                                       int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCswapBatched_v2_64(hipblasHandle_t   handle,
                                          int64_t           n,
                                          hipComplex* const x[],
                                          int64_t           incx,
                                          hipComplex* const y[],
                                          int64_t           incy,
                                          int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZswapBatched_v2_64(hipblasHandle_t         handle,
                                          int64_t                 n,
                                          hipDoubleComplex* const x[],
                                          int64_t                 incx,
                                          hipDoubleComplex* const y[],
                                          int64_t                 incy,
                                          int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// swap_strided_batched
hipblasStatus_t hipblasSswapStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           float*          x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           float*          y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDswapStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           double*         x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           double*         y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCswapStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           hipblasComplex* x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           hipblasComplex* y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZswapStridedBatched(hipblasHandle_t       handle,
                                           int                   n,
                                           hipblasDoubleComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           hipblasDoubleComplex* y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCswapStridedBatched_v2(hipblasHandle_t handle,
                                              int             n,
                                              hipComplex*     x,
                                              int             incx,
                                              hipblasStride   stridex,
                                              hipComplex*     y,
                                              int             incy,
                                              hipblasStride   stridey,
                                              int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZswapStridedBatched_v2(hipblasHandle_t   handle,
                                              int               n,
                                              hipDoubleComplex* x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              hipDoubleComplex* y,
                                              int               incy,
                                              hipblasStride     stridey,
                                              int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// swap_strided_batched_64
hipblasStatus_t hipblasSswapStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              float*          x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              float*          y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDswapStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              double*         x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              double*         y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCswapStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              hipblasComplex* x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              hipblasComplex* y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZswapStridedBatched_64(hipblasHandle_t       handle,
                                              int64_t               n,
                                              hipblasDoubleComplex* x,
                                              int64_t               incx,
                                              hipblasStride         stridex,
                                              hipblasDoubleComplex* y,
                                              int64_t               incy,
                                              hipblasStride         stridey,
                                              int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCswapStridedBatched_v2_64(hipblasHandle_t handle,
                                                 int64_t         n,
                                                 hipComplex*     x,
                                                 int64_t         incx,
                                                 hipblasStride   stridex,
                                                 hipComplex*     y,
                                                 int64_t         incy,
                                                 hipblasStride   stridey,
                                                 int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZswapStridedBatched_v2_64(hipblasHandle_t   handle,
                                                 int64_t           n,
                                                 hipDoubleComplex* x,
                                                 int64_t           incx,
                                                 hipblasStride     stridex,
                                                 hipDoubleComplex* y,
                                                 int64_t           incy,
                                                 hipblasStride     stridey,
                                                 int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// gbmv
hipblasStatus_t hipblasSgbmv(hipblasHandle_t    handle,
                             hipblasOperation_t trans,
                             int                m,
                             int                n,
                             int                kl,
                             int                ku,
                             const float*       alpha,
                             const float*       A,
                             int                lda,
                             const float*       x,
                             int                incx,
                             const float*       beta,
                             float*             y,
                             int                incy)
try
{
    return hipblasConvertStatus(cublasSgbmv((cublasHandle_t)handle,
                                            hipblasConvertOperation(trans),
                                            m,
                                            n,
                                            kl,
                                            ku,
                                            alpha,
                                            A,
                                            lda,
                                            x,
                                            incx,
                                            beta,
                                            y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDgbmv(hipblasHandle_t    handle,
                             hipblasOperation_t trans,
                             int                m,
                             int                n,
                             int                kl,
                             int                ku,
                             const double*      alpha,
                             const double*      A,
                             int                lda,
                             const double*      x,
                             int                incx,
                             const double*      beta,
                             double*            y,
                             int                incy)
try
{
    return hipblasConvertStatus(cublasDgbmv((cublasHandle_t)handle,
                                            hipblasConvertOperation(trans),
                                            m,
                                            n,
                                            kl,
                                            ku,
                                            alpha,
                                            A,
                                            lda,
                                            x,
                                            incx,
                                            beta,
                                            y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgbmv(hipblasHandle_t       handle,
                             hipblasOperation_t    trans,
                             int                   m,
                             int                   n,
                             int                   kl,
                             int                   ku,
                             const hipblasComplex* alpha,
                             const hipblasComplex* A,
                             int                   lda,
                             const hipblasComplex* x,
                             int                   incx,
                             const hipblasComplex* beta,
                             hipblasComplex*       y,
                             int                   incy)
try
{
    return hipblasConvertStatus(cublasCgbmv((cublasHandle_t)handle,
                                            hipblasConvertOperation(trans),
                                            m,
                                            n,
                                            kl,
                                            ku,
                                            (cuComplex*)alpha,
                                            (cuComplex*)A,
                                            lda,
                                            (cuComplex*)x,
                                            incx,
                                            (cuComplex*)beta,
                                            (cuComplex*)y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgbmv(hipblasHandle_t             handle,
                             hipblasOperation_t          trans,
                             int                         m,
                             int                         n,
                             int                         kl,
                             int                         ku,
                             const hipblasDoubleComplex* alpha,
                             const hipblasDoubleComplex* A,
                             int                         lda,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             const hipblasDoubleComplex* beta,
                             hipblasDoubleComplex*       y,
                             int                         incy)
try
{
    return hipblasConvertStatus(cublasZgbmv((cublasHandle_t)handle,
                                            hipblasConvertOperation(trans),
                                            m,
                                            n,
                                            kl,
                                            ku,
                                            (cuDoubleComplex*)alpha,
                                            (cuDoubleComplex*)A,
                                            lda,
                                            (cuDoubleComplex*)x,
                                            incx,
                                            (cuDoubleComplex*)beta,
                                            (cuDoubleComplex*)y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgbmv_v2(hipblasHandle_t    handle,
                                hipblasOperation_t trans,
                                int                m,
                                int                n,
                                int                kl,
                                int                ku,
                                const hipComplex*  alpha,
                                const hipComplex*  A,
                                int                lda,
                                const hipComplex*  x,
                                int                incx,
                                const hipComplex*  beta,
                                hipComplex*        y,
                                int                incy)
try
{
    return hipblasConvertStatus(cublasCgbmv((cublasHandle_t)handle,
                                            hipblasConvertOperation(trans),
                                            m,
                                            n,
                                            kl,
                                            ku,
                                            (cuComplex*)alpha,
                                            (cuComplex*)A,
                                            lda,
                                            (cuComplex*)x,
                                            incx,
                                            (cuComplex*)beta,
                                            (cuComplex*)y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgbmv_v2(hipblasHandle_t         handle,
                                hipblasOperation_t      trans,
                                int                     m,
                                int                     n,
                                int                     kl,
                                int                     ku,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* A,
                                int                     lda,
                                const hipDoubleComplex* x,
                                int                     incx,
                                const hipDoubleComplex* beta,
                                hipDoubleComplex*       y,
                                int                     incy)
try
{
    return hipblasConvertStatus(cublasZgbmv((cublasHandle_t)handle,
                                            hipblasConvertOperation(trans),
                                            m,
                                            n,
                                            kl,
                                            ku,
                                            (cuDoubleComplex*)alpha,
                                            (cuDoubleComplex*)A,
                                            lda,
                                            (cuDoubleComplex*)x,
                                            incx,
                                            (cuDoubleComplex*)beta,
                                            (cuDoubleComplex*)y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// gbmv_64
hipblasStatus_t hipblasSgbmv_64(hipblasHandle_t    handle,
                                hipblasOperation_t trans,
                                int64_t            m,
                                int64_t            n,
                                int64_t            kl,
                                int64_t            ku,
                                const float*       alpha,
                                const float*       A,
                                int64_t            lda,
                                const float*       x,
                                int64_t            incx,
                                const float*       beta,
                                float*             y,
                                int64_t            incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasSgbmv_64((cublasHandle_t)handle,
                                               hipblasConvertOperation(trans),
                                               m,
                                               n,
                                               kl,
                                               ku,
                                               alpha,
                                               A,
                                               lda,
                                               x,
                                               incx,
                                               beta,
                                               y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDgbmv_64(hipblasHandle_t    handle,
                                hipblasOperation_t trans,
                                int64_t            m,
                                int64_t            n,
                                int64_t            kl,
                                int64_t            ku,
                                const double*      alpha,
                                const double*      A,
                                int64_t            lda,
                                const double*      x,
                                int64_t            incx,
                                const double*      beta,
                                double*            y,
                                int64_t            incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasDgbmv_64((cublasHandle_t)handle,
                                               hipblasConvertOperation(trans),
                                               m,
                                               n,
                                               kl,
                                               ku,
                                               alpha,
                                               A,
                                               lda,
                                               x,
                                               incx,
                                               beta,
                                               y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgbmv_64(hipblasHandle_t       handle,
                                hipblasOperation_t    trans,
                                int64_t               m,
                                int64_t               n,
                                int64_t               kl,
                                int64_t               ku,
                                const hipblasComplex* alpha,
                                const hipblasComplex* A,
                                int64_t               lda,
                                const hipblasComplex* x,
                                int64_t               incx,
                                const hipblasComplex* beta,
                                hipblasComplex*       y,
                                int64_t               incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCgbmv_64((cublasHandle_t)handle,
                                               hipblasConvertOperation(trans),
                                               m,
                                               n,
                                               kl,
                                               ku,
                                               (cuComplex*)alpha,
                                               (cuComplex*)A,
                                               lda,
                                               (cuComplex*)x,
                                               incx,
                                               (cuComplex*)beta,
                                               (cuComplex*)y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgbmv_64(hipblasHandle_t             handle,
                                hipblasOperation_t          trans,
                                int64_t                     m,
                                int64_t                     n,
                                int64_t                     kl,
                                int64_t                     ku,
                                const hipblasDoubleComplex* alpha,
                                const hipblasDoubleComplex* A,
                                int64_t                     lda,
                                const hipblasDoubleComplex* x,
                                int64_t                     incx,
                                const hipblasDoubleComplex* beta,
                                hipblasDoubleComplex*       y,
                                int64_t                     incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZgbmv_64((cublasHandle_t)handle,
                                               hipblasConvertOperation(trans),
                                               m,
                                               n,
                                               kl,
                                               ku,
                                               (cuDoubleComplex*)alpha,
                                               (cuDoubleComplex*)A,
                                               lda,
                                               (cuDoubleComplex*)x,
                                               incx,
                                               (cuDoubleComplex*)beta,
                                               (cuDoubleComplex*)y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgbmv_v2_64(hipblasHandle_t    handle,
                                   hipblasOperation_t trans,
                                   int64_t            m,
                                   int64_t            n,
                                   int64_t            kl,
                                   int64_t            ku,
                                   const hipComplex*  alpha,
                                   const hipComplex*  A,
                                   int64_t            lda,
                                   const hipComplex*  x,
                                   int64_t            incx,
                                   const hipComplex*  beta,
                                   hipComplex*        y,
                                   int64_t            incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCgbmv_64((cublasHandle_t)handle,
                                               hipblasConvertOperation(trans),
                                               m,
                                               n,
                                               kl,
                                               ku,
                                               (cuComplex*)alpha,
                                               (cuComplex*)A,
                                               lda,
                                               (cuComplex*)x,
                                               incx,
                                               (cuComplex*)beta,
                                               (cuComplex*)y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgbmv_v2_64(hipblasHandle_t         handle,
                                   hipblasOperation_t      trans,
                                   int64_t                 m,
                                   int64_t                 n,
                                   int64_t                 kl,
                                   int64_t                 ku,
                                   const hipDoubleComplex* alpha,
                                   const hipDoubleComplex* A,
                                   int64_t                 lda,
                                   const hipDoubleComplex* x,
                                   int64_t                 incx,
                                   const hipDoubleComplex* beta,
                                   hipDoubleComplex*       y,
                                   int64_t                 incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZgbmv_64((cublasHandle_t)handle,
                                               hipblasConvertOperation(trans),
                                               m,
                                               n,
                                               kl,
                                               ku,
                                               (cuDoubleComplex*)alpha,
                                               (cuDoubleComplex*)A,
                                               lda,
                                               (cuDoubleComplex*)x,
                                               incx,
                                               (cuDoubleComplex*)beta,
                                               (cuDoubleComplex*)y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// gbmv_batched
hipblasStatus_t hipblasSgbmvBatched(hipblasHandle_t    handle,
                                    hipblasOperation_t trans,
                                    int                m,
                                    int                n,
                                    int                kl,
                                    int                ku,
                                    const float*       alpha,
                                    const float* const A[],
                                    int                lda,
                                    const float* const x[],
                                    int                incx,
                                    const float*       beta,
                                    float* const       y[],
                                    int                incy,
                                    int                batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDgbmvBatched(hipblasHandle_t     handle,
                                    hipblasOperation_t  trans,
                                    int                 m,
                                    int                 n,
                                    int                 kl,
                                    int                 ku,
                                    const double*       alpha,
                                    const double* const A[],
                                    int                 lda,
                                    const double* const x[],
                                    int                 incx,
                                    const double*       beta,
                                    double* const       y[],
                                    int                 incy,
                                    int                 batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCgbmvBatched(hipblasHandle_t             handle,
                                    hipblasOperation_t          trans,
                                    int                         m,
                                    int                         n,
                                    int                         kl,
                                    int                         ku,
                                    const hipblasComplex*       alpha,
                                    const hipblasComplex* const A[],
                                    int                         lda,
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    const hipblasComplex*       beta,
                                    hipblasComplex* const       y[],
                                    int                         incy,
                                    int                         batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZgbmvBatched(hipblasHandle_t                   handle,
                                    hipblasOperation_t                trans,
                                    int                               m,
                                    int                               n,
                                    int                               kl,
                                    int                               ku,
                                    const hipblasDoubleComplex*       alpha,
                                    const hipblasDoubleComplex* const A[],
                                    int                               lda,
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    const hipblasDoubleComplex*       beta,
                                    hipblasDoubleComplex* const       y[],
                                    int                               incy,
                                    int                               batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCgbmvBatched_v2(hipblasHandle_t         handle,
                                       hipblasOperation_t      trans,
                                       int                     m,
                                       int                     n,
                                       int                     kl,
                                       int                     ku,
                                       const hipComplex*       alpha,
                                       const hipComplex* const A[],
                                       int                     lda,
                                       const hipComplex* const x[],
                                       int                     incx,
                                       const hipComplex*       beta,
                                       hipComplex* const       y[],
                                       int                     incy,
                                       int                     batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZgbmvBatched_v2(hipblasHandle_t               handle,
                                       hipblasOperation_t            trans,
                                       int                           m,
                                       int                           n,
                                       int                           kl,
                                       int                           ku,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const A[],
                                       int                           lda,
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       const hipDoubleComplex*       beta,
                                       hipDoubleComplex* const       y[],
                                       int                           incy,
                                       int                           batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// gbmv_batched_64
hipblasStatus_t hipblasSgbmvBatched_64(hipblasHandle_t    handle,
                                       hipblasOperation_t trans,
                                       int64_t            m,
                                       int64_t            n,
                                       int64_t            kl,
                                       int64_t            ku,
                                       const float*       alpha,
                                       const float* const A[],
                                       int64_t            lda,
                                       const float* const x[],
                                       int64_t            incx,
                                       const float*       beta,
                                       float* const       y[],
                                       int64_t            incy,
                                       int64_t            batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDgbmvBatched_64(hipblasHandle_t     handle,
                                       hipblasOperation_t  trans,
                                       int64_t             m,
                                       int64_t             n,
                                       int64_t             kl,
                                       int64_t             ku,
                                       const double*       alpha,
                                       const double* const A[],
                                       int64_t             lda,
                                       const double* const x[],
                                       int64_t             incx,
                                       const double*       beta,
                                       double* const       y[],
                                       int64_t             incy,
                                       int64_t             batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCgbmvBatched_64(hipblasHandle_t             handle,
                                       hipblasOperation_t          trans,
                                       int64_t                     m,
                                       int64_t                     n,
                                       int64_t                     kl,
                                       int64_t                     ku,
                                       const hipblasComplex*       alpha,
                                       const hipblasComplex* const A[],
                                       int64_t                     lda,
                                       const hipblasComplex* const x[],
                                       int64_t                     incx,
                                       const hipblasComplex*       beta,
                                       hipblasComplex* const       y[],
                                       int64_t                     incy,
                                       int64_t                     batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZgbmvBatched_64(hipblasHandle_t                   handle,
                                       hipblasOperation_t                trans,
                                       int64_t                           m,
                                       int64_t                           n,
                                       int64_t                           kl,
                                       int64_t                           ku,
                                       const hipblasDoubleComplex*       alpha,
                                       const hipblasDoubleComplex* const A[],
                                       int64_t                           lda,
                                       const hipblasDoubleComplex* const x[],
                                       int64_t                           incx,
                                       const hipblasDoubleComplex*       beta,
                                       hipblasDoubleComplex* const       y[],
                                       int64_t                           incy,
                                       int64_t                           batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCgbmvBatched_v2_64(hipblasHandle_t         handle,
                                          hipblasOperation_t      trans,
                                          int64_t                 m,
                                          int64_t                 n,
                                          int64_t                 kl,
                                          int64_t                 ku,
                                          const hipComplex*       alpha,
                                          const hipComplex* const A[],
                                          int64_t                 lda,
                                          const hipComplex* const x[],
                                          int64_t                 incx,
                                          const hipComplex*       beta,
                                          hipComplex* const       y[],
                                          int64_t                 incy,
                                          int64_t                 batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZgbmvBatched_v2_64(hipblasHandle_t               handle,
                                          hipblasOperation_t            trans,
                                          int64_t                       m,
                                          int64_t                       n,
                                          int64_t                       kl,
                                          int64_t                       ku,
                                          const hipDoubleComplex*       alpha,
                                          const hipDoubleComplex* const A[],
                                          int64_t                       lda,
                                          const hipDoubleComplex* const x[],
                                          int64_t                       incx,
                                          const hipDoubleComplex*       beta,
                                          hipDoubleComplex* const       y[],
                                          int64_t                       incy,
                                          int64_t                       batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// gbmv_strided_batched
hipblasStatus_t hipblasSgbmvStridedBatched(hipblasHandle_t    handle,
                                           hipblasOperation_t trans,
                                           int                m,
                                           int                n,
                                           int                kl,
                                           int                ku,
                                           const float*       alpha,
                                           const float*       A,
                                           int                lda,
                                           hipblasStride      stride_a,
                                           const float*       x,
                                           int                incx,
                                           hipblasStride      stride_x,
                                           const float*       beta,
                                           float*             y,
                                           int                incy,
                                           hipblasStride      stride_y,
                                           int                batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDgbmvStridedBatched(hipblasHandle_t    handle,
                                           hipblasOperation_t trans,
                                           int                m,
                                           int                n,
                                           int                kl,
                                           int                ku,
                                           const double*      alpha,
                                           const double*      A,
                                           int                lda,
                                           hipblasStride      stride_a,
                                           const double*      x,
                                           int                incx,
                                           hipblasStride      stride_x,
                                           const double*      beta,
                                           double*            y,
                                           int                incy,
                                           hipblasStride      stride_y,
                                           int                batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCgbmvStridedBatched(hipblasHandle_t       handle,
                                           hipblasOperation_t    trans,
                                           int                   m,
                                           int                   n,
                                           int                   kl,
                                           int                   ku,
                                           const hipblasComplex* alpha,
                                           const hipblasComplex* A,
                                           int                   lda,
                                           hipblasStride         stride_a,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stride_x,
                                           const hipblasComplex* beta,
                                           hipblasComplex*       y,
                                           int                   incy,
                                           hipblasStride         stride_y,
                                           int                   batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZgbmvStridedBatched(hipblasHandle_t             handle,
                                           hipblasOperation_t          trans,
                                           int                         m,
                                           int                         n,
                                           int                         kl,
                                           int                         ku,
                                           const hipblasDoubleComplex* alpha,
                                           const hipblasDoubleComplex* A,
                                           int                         lda,
                                           hipblasStride               stride_a,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stride_x,
                                           const hipblasDoubleComplex* beta,
                                           hipblasDoubleComplex*       y,
                                           int                         incy,
                                           hipblasStride               stride_y,
                                           int                         batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCgbmvStridedBatched_v2(hipblasHandle_t    handle,
                                              hipblasOperation_t trans,
                                              int                m,
                                              int                n,
                                              int                kl,
                                              int                ku,
                                              const hipComplex*  alpha,
                                              const hipComplex*  A,
                                              int                lda,
                                              hipblasStride      stride_a,
                                              const hipComplex*  x,
                                              int                incx,
                                              hipblasStride      stride_x,
                                              const hipComplex*  beta,
                                              hipComplex*        y,
                                              int                incy,
                                              hipblasStride      stride_y,
                                              int                batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZgbmvStridedBatched_v2(hipblasHandle_t         handle,
                                              hipblasOperation_t      trans,
                                              int                     m,
                                              int                     n,
                                              int                     kl,
                                              int                     ku,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* A,
                                              int                     lda,
                                              hipblasStride           stride_a,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stride_x,
                                              const hipDoubleComplex* beta,
                                              hipDoubleComplex*       y,
                                              int                     incy,
                                              hipblasStride           stride_y,
                                              int                     batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// gbmv_strided_batched_64
hipblasStatus_t hipblasSgbmvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasOperation_t trans,
                                              int64_t            m,
                                              int64_t            n,
                                              int64_t            kl,
                                              int64_t            ku,
                                              const float*       alpha,
                                              const float*       A,
                                              int64_t            lda,
                                              hipblasStride      stride_a,
                                              const float*       x,
                                              int64_t            incx,
                                              hipblasStride      stride_x,
                                              const float*       beta,
                                              float*             y,
                                              int64_t            incy,
                                              hipblasStride      stride_y,
                                              int64_t            batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDgbmvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasOperation_t trans,
                                              int64_t            m,
                                              int64_t            n,
                                              int64_t            kl,
                                              int64_t            ku,
                                              const double*      alpha,
                                              const double*      A,
                                              int64_t            lda,
                                              hipblasStride      stride_a,
                                              const double*      x,
                                              int64_t            incx,
                                              hipblasStride      stride_x,
                                              const double*      beta,
                                              double*            y,
                                              int64_t            incy,
                                              hipblasStride      stride_y,
                                              int64_t            batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCgbmvStridedBatched_64(hipblasHandle_t       handle,
                                              hipblasOperation_t    trans,
                                              int64_t               m,
                                              int64_t               n,
                                              int64_t               kl,
                                              int64_t               ku,
                                              const hipblasComplex* alpha,
                                              const hipblasComplex* A,
                                              int64_t               lda,
                                              hipblasStride         stride_a,
                                              const hipblasComplex* x,
                                              int64_t               incx,
                                              hipblasStride         stride_x,
                                              const hipblasComplex* beta,
                                              hipblasComplex*       y,
                                              int64_t               incy,
                                              hipblasStride         stride_y,
                                              int64_t               batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZgbmvStridedBatched_64(hipblasHandle_t             handle,
                                              hipblasOperation_t          trans,
                                              int64_t                     m,
                                              int64_t                     n,
                                              int64_t                     kl,
                                              int64_t                     ku,
                                              const hipblasDoubleComplex* alpha,
                                              const hipblasDoubleComplex* A,
                                              int64_t                     lda,
                                              hipblasStride               stride_a,
                                              const hipblasDoubleComplex* x,
                                              int64_t                     incx,
                                              hipblasStride               stride_x,
                                              const hipblasDoubleComplex* beta,
                                              hipblasDoubleComplex*       y,
                                              int64_t                     incy,
                                              hipblasStride               stride_y,
                                              int64_t                     batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCgbmvStridedBatched_v2_64(hipblasHandle_t    handle,
                                                 hipblasOperation_t trans,
                                                 int64_t            m,
                                                 int64_t            n,
                                                 int64_t            kl,
                                                 int64_t            ku,
                                                 const hipComplex*  alpha,
                                                 const hipComplex*  A,
                                                 int64_t            lda,
                                                 hipblasStride      stride_a,
                                                 const hipComplex*  x,
                                                 int64_t            incx,
                                                 hipblasStride      stride_x,
                                                 const hipComplex*  beta,
                                                 hipComplex*        y,
                                                 int64_t            incy,
                                                 hipblasStride      stride_y,
                                                 int64_t            batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZgbmvStridedBatched_v2_64(hipblasHandle_t         handle,
                                                 hipblasOperation_t      trans,
                                                 int64_t                 m,
                                                 int64_t                 n,
                                                 int64_t                 kl,
                                                 int64_t                 ku,
                                                 const hipDoubleComplex* alpha,
                                                 const hipDoubleComplex* A,
                                                 int64_t                 lda,
                                                 hipblasStride           stride_a,
                                                 const hipDoubleComplex* x,
                                                 int64_t                 incx,
                                                 hipblasStride           stride_x,
                                                 const hipDoubleComplex* beta,
                                                 hipDoubleComplex*       y,
                                                 int64_t                 incy,
                                                 hipblasStride           stride_y,
                                                 int64_t                 batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// gemv
hipblasStatus_t hipblasSgemv(hipblasHandle_t    handle,
                             hipblasOperation_t trans,
                             int                m,
                             int                n,
                             const float*       alpha,
                             const float*       A,
                             int                lda,
                             const float*       x,
                             int                incx,
                             const float*       beta,
                             float*             y,
                             int                incy)
try
{
    return hipblasConvertStatus(cublasSgemv((cublasHandle_t)handle,
                                            hipblasConvertOperation(trans),
                                            m,
                                            n,
                                            alpha,
                                            A,
                                            lda,
                                            x,
                                            incx,
                                            beta,
                                            y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDgemv(hipblasHandle_t    handle,
                             hipblasOperation_t trans,
                             int                m,
                             int                n,
                             const double*      alpha,
                             const double*      A,
                             int                lda,
                             const double*      x,
                             int                incx,
                             const double*      beta,
                             double*            y,
                             int                incy)
try
{
    return hipblasConvertStatus(cublasDgemv((cublasHandle_t)handle,
                                            hipblasConvertOperation(trans),
                                            m,
                                            n,
                                            alpha,
                                            A,
                                            lda,
                                            x,
                                            incx,
                                            beta,
                                            y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgemv(hipblasHandle_t       handle,
                             hipblasOperation_t    trans,
                             int                   m,
                             int                   n,
                             const hipblasComplex* alpha,
                             const hipblasComplex* A,
                             int                   lda,
                             const hipblasComplex* x,
                             int                   incx,
                             const hipblasComplex* beta,
                             hipblasComplex*       y,
                             int                   incy)
try
{
    return hipblasConvertStatus(cublasCgemv((cublasHandle_t)handle,
                                            hipblasConvertOperation(trans),
                                            m,
                                            n,
                                            (cuComplex*)alpha,
                                            (cuComplex*)A,
                                            lda,
                                            (cuComplex*)x,
                                            incx,
                                            (cuComplex*)beta,
                                            (cuComplex*)y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgemv(hipblasHandle_t             handle,
                             hipblasOperation_t          trans,
                             int                         m,
                             int                         n,
                             const hipblasDoubleComplex* alpha,
                             const hipblasDoubleComplex* A,
                             int                         lda,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             const hipblasDoubleComplex* beta,
                             hipblasDoubleComplex*       y,
                             int                         incy)
try
{
    return hipblasConvertStatus(cublasZgemv((cublasHandle_t)handle,
                                            hipblasConvertOperation(trans),
                                            m,
                                            n,
                                            (cuDoubleComplex*)alpha,
                                            (cuDoubleComplex*)A,
                                            lda,
                                            (cuDoubleComplex*)x,
                                            incx,
                                            (cuDoubleComplex*)beta,
                                            (cuDoubleComplex*)y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgemv_v2(hipblasHandle_t    handle,
                                hipblasOperation_t trans,
                                int                m,
                                int                n,
                                const hipComplex*  alpha,
                                const hipComplex*  A,
                                int                lda,
                                const hipComplex*  x,
                                int                incx,
                                const hipComplex*  beta,
                                hipComplex*        y,
                                int                incy)
try
{
    return hipblasConvertStatus(cublasCgemv((cublasHandle_t)handle,
                                            hipblasConvertOperation(trans),
                                            m,
                                            n,
                                            (cuComplex*)alpha,
                                            (cuComplex*)A,
                                            lda,
                                            (cuComplex*)x,
                                            incx,
                                            (cuComplex*)beta,
                                            (cuComplex*)y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgemv_v2(hipblasHandle_t         handle,
                                hipblasOperation_t      trans,
                                int                     m,
                                int                     n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* A,
                                int                     lda,
                                const hipDoubleComplex* x,
                                int                     incx,
                                const hipDoubleComplex* beta,
                                hipDoubleComplex*       y,
                                int                     incy)
try
{
    return hipblasConvertStatus(cublasZgemv((cublasHandle_t)handle,
                                            hipblasConvertOperation(trans),
                                            m,
                                            n,
                                            (cuDoubleComplex*)alpha,
                                            (cuDoubleComplex*)A,
                                            lda,
                                            (cuDoubleComplex*)x,
                                            incx,
                                            (cuDoubleComplex*)beta,
                                            (cuDoubleComplex*)y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// gemv_64
hipblasStatus_t hipblasSgemv_64(hipblasHandle_t    handle,
                                hipblasOperation_t trans,
                                int64_t            m,
                                int64_t            n,
                                const float*       alpha,
                                const float*       A,
                                int64_t            lda,
                                const float*       x,
                                int64_t            incx,
                                const float*       beta,
                                float*             y,
                                int64_t            incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasSgemv_64((cublasHandle_t)handle,
                                               hipblasConvertOperation(trans),
                                               m,
                                               n,
                                               alpha,
                                               A,
                                               lda,
                                               x,
                                               incx,
                                               beta,
                                               y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDgemv_64(hipblasHandle_t    handle,
                                hipblasOperation_t trans,
                                int64_t            m,
                                int64_t            n,
                                const double*      alpha,
                                const double*      A,
                                int64_t            lda,
                                const double*      x,
                                int64_t            incx,
                                const double*      beta,
                                double*            y,
                                int64_t            incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasDgemv_64((cublasHandle_t)handle,
                                               hipblasConvertOperation(trans),
                                               m,
                                               n,
                                               alpha,
                                               A,
                                               lda,
                                               x,
                                               incx,
                                               beta,
                                               y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgemv_64(hipblasHandle_t       handle,
                                hipblasOperation_t    trans,
                                int64_t               m,
                                int64_t               n,
                                const hipblasComplex* alpha,
                                const hipblasComplex* A,
                                int64_t               lda,
                                const hipblasComplex* x,
                                int64_t               incx,
                                const hipblasComplex* beta,
                                hipblasComplex*       y,
                                int64_t               incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCgemv_64((cublasHandle_t)handle,
                                               hipblasConvertOperation(trans),
                                               m,
                                               n,
                                               (cuComplex*)alpha,
                                               (cuComplex*)A,
                                               lda,
                                               (cuComplex*)x,
                                               incx,
                                               (cuComplex*)beta,
                                               (cuComplex*)y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgemv_64(hipblasHandle_t             handle,
                                hipblasOperation_t          trans,
                                int64_t                     m,
                                int64_t                     n,
                                const hipblasDoubleComplex* alpha,
                                const hipblasDoubleComplex* A,
                                int64_t                     lda,
                                const hipblasDoubleComplex* x,
                                int64_t                     incx,
                                const hipblasDoubleComplex* beta,
                                hipblasDoubleComplex*       y,
                                int64_t                     incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZgemv_64((cublasHandle_t)handle,
                                               hipblasConvertOperation(trans),
                                               m,
                                               n,
                                               (cuDoubleComplex*)alpha,
                                               (cuDoubleComplex*)A,
                                               lda,
                                               (cuDoubleComplex*)x,
                                               incx,
                                               (cuDoubleComplex*)beta,
                                               (cuDoubleComplex*)y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgemv_v2_64(hipblasHandle_t    handle,
                                   hipblasOperation_t trans,
                                   int64_t            m,
                                   int64_t            n,
                                   const hipComplex*  alpha,
                                   const hipComplex*  A,
                                   int64_t            lda,
                                   const hipComplex*  x,
                                   int64_t            incx,
                                   const hipComplex*  beta,
                                   hipComplex*        y,
                                   int64_t            incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCgemv_64((cublasHandle_t)handle,
                                               hipblasConvertOperation(trans),
                                               m,
                                               n,
                                               (cuComplex*)alpha,
                                               (cuComplex*)A,
                                               lda,
                                               (cuComplex*)x,
                                               incx,
                                               (cuComplex*)beta,
                                               (cuComplex*)y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgemv_v2_64(hipblasHandle_t         handle,
                                   hipblasOperation_t      trans,
                                   int64_t                 m,
                                   int64_t                 n,
                                   const hipDoubleComplex* alpha,
                                   const hipDoubleComplex* A,
                                   int64_t                 lda,
                                   const hipDoubleComplex* x,
                                   int64_t                 incx,
                                   const hipDoubleComplex* beta,
                                   hipDoubleComplex*       y,
                                   int64_t                 incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZgemv_64((cublasHandle_t)handle,
                                               hipblasConvertOperation(trans),
                                               m,
                                               n,
                                               (cuDoubleComplex*)alpha,
                                               (cuDoubleComplex*)A,
                                               lda,
                                               (cuDoubleComplex*)x,
                                               incx,
                                               (cuDoubleComplex*)beta,
                                               (cuDoubleComplex*)y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// gemv_batched
hipblasStatus_t hipblasSgemvBatched(hipblasHandle_t    handle,
                                    hipblasOperation_t trans,
                                    int                m,
                                    int                n,
                                    const float*       alpha,
                                    const float* const A[],
                                    int                lda,
                                    const float* const x[],
                                    int                incx,
                                    const float*       beta,
                                    float* const       y[],
                                    int                incy,
                                    int                batchCount)
try
{
    // at least cublas 11.7.0
#if CUBLAS_VERSION >= 110700
    return hipblasConvertStatus(cublasSgemvBatched((cublasHandle_t)handle,
                                                   hipblasConvertOperation(trans),
                                                   m,
                                                   n,
                                                   alpha,
                                                   A,
                                                   lda,
                                                   x,
                                                   incx,
                                                   beta,
                                                   y,
                                                   incy,
                                                   batchCount));

#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDgemvBatched(hipblasHandle_t     handle,
                                    hipblasOperation_t  trans,
                                    int                 m,
                                    int                 n,
                                    const double*       alpha,
                                    const double* const A[],
                                    int                 lda,
                                    const double* const x[],
                                    int                 incx,
                                    const double*       beta,
                                    double* const       y[],
                                    int                 incy,
                                    int                 batchCount)
try
{
#if CUBLAS_VERSION >= 110700
    return hipblasConvertStatus(cublasDgemvBatched((cublasHandle_t)handle,
                                                   hipblasConvertOperation(trans),
                                                   m,
                                                   n,
                                                   alpha,
                                                   A,
                                                   lda,
                                                   x,
                                                   incx,
                                                   beta,
                                                   y,
                                                   incy,
                                                   batchCount));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgemvBatched(hipblasHandle_t             handle,
                                    hipblasOperation_t          trans,
                                    int                         m,
                                    int                         n,
                                    const hipblasComplex*       alpha,
                                    const hipblasComplex* const A[],
                                    int                         lda,
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    const hipblasComplex*       beta,
                                    hipblasComplex* const       y[],
                                    int                         incy,
                                    int                         batchCount)
try
{
#if CUBLAS_VERSION >= 110700
    return hipblasConvertStatus(cublasCgemvBatched((cublasHandle_t)handle,
                                                   hipblasConvertOperation(trans),
                                                   m,
                                                   n,
                                                   (const cuComplex*)alpha,
                                                   (const cuComplex**)A,
                                                   lda,
                                                   (const cuComplex**)x,
                                                   incx,
                                                   (const cuComplex*)beta,
                                                   (cuComplex**)y,
                                                   incy,
                                                   batchCount));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgemvBatched(hipblasHandle_t                   handle,
                                    hipblasOperation_t                trans,
                                    int                               m,
                                    int                               n,
                                    const hipblasDoubleComplex*       alpha,
                                    const hipblasDoubleComplex* const A[],
                                    int                               lda,
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    const hipblasDoubleComplex*       beta,
                                    hipblasDoubleComplex* const       y[],
                                    int                               incy,
                                    int                               batchCount)
try
{
#if CUBLAS_VERSION >= 110700
    return hipblasConvertStatus(cublasZgemvBatched((cublasHandle_t)handle,
                                                   hipblasConvertOperation(trans),
                                                   m,
                                                   n,
                                                   (const cuDoubleComplex*)alpha,
                                                   (const cuDoubleComplex**)A,
                                                   lda,
                                                   (const cuDoubleComplex**)x,
                                                   incx,
                                                   (const cuDoubleComplex*)beta,
                                                   (cuDoubleComplex**)y,
                                                   incy,
                                                   batchCount));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgemvBatched_v2(hipblasHandle_t         handle,
                                       hipblasOperation_t      trans,
                                       int                     m,
                                       int                     n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const A[],
                                       int                     lda,
                                       const hipComplex* const x[],
                                       int                     incx,
                                       const hipComplex*       beta,
                                       hipComplex* const       y[],
                                       int                     incy,
                                       int                     batchCount)
try
{
#if CUBLAS_VERSION >= 110700
    return hipblasConvertStatus(cublasCgemvBatched((cublasHandle_t)handle,
                                                   hipblasConvertOperation(trans),
                                                   m,
                                                   n,
                                                   (const cuComplex*)alpha,
                                                   (const cuComplex**)A,
                                                   lda,
                                                   (const cuComplex**)x,
                                                   incx,
                                                   (const cuComplex*)beta,
                                                   (cuComplex**)y,
                                                   incy,
                                                   batchCount));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgemvBatched_v2(hipblasHandle_t               handle,
                                       hipblasOperation_t            trans,
                                       int                           m,
                                       int                           n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const A[],
                                       int                           lda,
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       const hipDoubleComplex*       beta,
                                       hipDoubleComplex* const       y[],
                                       int                           incy,
                                       int                           batchCount)
try
{
#if CUBLAS_VERSION >= 110700
    return hipblasConvertStatus(cublasZgemvBatched((cublasHandle_t)handle,
                                                   hipblasConvertOperation(trans),
                                                   m,
                                                   n,
                                                   (const cuDoubleComplex*)alpha,
                                                   (const cuDoubleComplex**)A,
                                                   lda,
                                                   (const cuDoubleComplex**)x,
                                                   incx,
                                                   (const cuDoubleComplex*)beta,
                                                   (cuDoubleComplex**)y,
                                                   incy,
                                                   batchCount));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// gemv_batched_64
hipblasStatus_t hipblasSgemvBatched_64(hipblasHandle_t    handle,
                                       hipblasOperation_t trans,
                                       int64_t            m,
                                       int64_t            n,
                                       const float*       alpha,
                                       const float* const A[],
                                       int64_t            lda,
                                       const float* const x[],
                                       int64_t            incx,
                                       const float*       beta,
                                       float* const       y[],
                                       int64_t            incy,
                                       int64_t            batchCount)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasSgemvBatched_64((cublasHandle_t)handle,
                                                      hipblasConvertOperation(trans),
                                                      m,
                                                      n,
                                                      alpha,
                                                      A,
                                                      lda,
                                                      x,
                                                      incx,
                                                      beta,
                                                      y,
                                                      incy,
                                                      batchCount));

#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDgemvBatched_64(hipblasHandle_t     handle,
                                       hipblasOperation_t  trans,
                                       int64_t             m,
                                       int64_t             n,
                                       const double*       alpha,
                                       const double* const A[],
                                       int64_t             lda,
                                       const double* const x[],
                                       int64_t             incx,
                                       const double*       beta,
                                       double* const       y[],
                                       int64_t             incy,
                                       int64_t             batchCount)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasDgemvBatched_64((cublasHandle_t)handle,
                                                      hipblasConvertOperation(trans),
                                                      m,
                                                      n,
                                                      alpha,
                                                      A,
                                                      lda,
                                                      x,
                                                      incx,
                                                      beta,
                                                      y,
                                                      incy,
                                                      batchCount));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgemvBatched_64(hipblasHandle_t             handle,
                                       hipblasOperation_t          trans,
                                       int64_t                     m,
                                       int64_t                     n,
                                       const hipblasComplex*       alpha,
                                       const hipblasComplex* const A[],
                                       int64_t                     lda,
                                       const hipblasComplex* const x[],
                                       int64_t                     incx,
                                       const hipblasComplex*       beta,
                                       hipblasComplex* const       y[],
                                       int64_t                     incy,
                                       int64_t                     batchCount)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCgemvBatched_64((cublasHandle_t)handle,
                                                      hipblasConvertOperation(trans),
                                                      m,
                                                      n,
                                                      (const cuComplex*)alpha,
                                                      (const cuComplex**)A,
                                                      lda,
                                                      (const cuComplex**)x,
                                                      incx,
                                                      (const cuComplex*)beta,
                                                      (cuComplex**)y,
                                                      incy,
                                                      batchCount));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgemvBatched_64(hipblasHandle_t                   handle,
                                       hipblasOperation_t                trans,
                                       int64_t                           m,
                                       int64_t                           n,
                                       const hipblasDoubleComplex*       alpha,
                                       const hipblasDoubleComplex* const A[],
                                       int64_t                           lda,
                                       const hipblasDoubleComplex* const x[],
                                       int64_t                           incx,
                                       const hipblasDoubleComplex*       beta,
                                       hipblasDoubleComplex* const       y[],
                                       int64_t                           incy,
                                       int64_t                           batchCount)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZgemvBatched_64((cublasHandle_t)handle,
                                                      hipblasConvertOperation(trans),
                                                      m,
                                                      n,
                                                      (const cuDoubleComplex*)alpha,
                                                      (const cuDoubleComplex**)A,
                                                      lda,
                                                      (const cuDoubleComplex**)x,
                                                      incx,
                                                      (const cuDoubleComplex*)beta,
                                                      (cuDoubleComplex**)y,
                                                      incy,
                                                      batchCount));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgemvBatched_v2_64(hipblasHandle_t         handle,
                                          hipblasOperation_t      trans,
                                          int64_t                 m,
                                          int64_t                 n,
                                          const hipComplex*       alpha,
                                          const hipComplex* const A[],
                                          int64_t                 lda,
                                          const hipComplex* const x[],
                                          int64_t                 incx,
                                          const hipComplex*       beta,
                                          hipComplex* const       y[],
                                          int64_t                 incy,
                                          int64_t                 batchCount)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCgemvBatched_64((cublasHandle_t)handle,
                                                      hipblasConvertOperation(trans),
                                                      m,
                                                      n,
                                                      (const cuComplex*)alpha,
                                                      (const cuComplex**)A,
                                                      lda,
                                                      (const cuComplex**)x,
                                                      incx,
                                                      (const cuComplex*)beta,
                                                      (cuComplex**)y,
                                                      incy,
                                                      batchCount));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgemvBatched_v2_64(hipblasHandle_t               handle,
                                          hipblasOperation_t            trans,
                                          int64_t                       m,
                                          int64_t                       n,
                                          const hipDoubleComplex*       alpha,
                                          const hipDoubleComplex* const A[],
                                          int64_t                       lda,
                                          const hipDoubleComplex* const x[],
                                          int64_t                       incx,
                                          const hipDoubleComplex*       beta,
                                          hipDoubleComplex* const       y[],
                                          int64_t                       incy,
                                          int64_t                       batchCount)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZgemvBatched_64((cublasHandle_t)handle,
                                                      hipblasConvertOperation(trans),
                                                      m,
                                                      n,
                                                      (const cuDoubleComplex*)alpha,
                                                      (const cuDoubleComplex**)A,
                                                      lda,
                                                      (const cuDoubleComplex**)x,
                                                      incx,
                                                      (const cuDoubleComplex*)beta,
                                                      (cuDoubleComplex**)y,
                                                      incy,
                                                      batchCount));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// gemv_strided_batched
hipblasStatus_t hipblasSgemvStridedBatched(hipblasHandle_t    handle,
                                           hipblasOperation_t trans,
                                           int                m,
                                           int                n,
                                           const float*       alpha,
                                           const float*       A,
                                           int                lda,
                                           hipblasStride      strideA,
                                           const float*       x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           const float*       beta,
                                           float*             y,
                                           int                incy,
                                           hipblasStride      stridey,
                                           int                batchCount)
try
{
#if CUBLAS_VERSION >= 110700
    return hipblasConvertStatus(cublasSgemvStridedBatched((cublasHandle_t)handle,
                                                          hipblasConvertOperation(trans),
                                                          m,
                                                          n,
                                                          alpha,
                                                          A,
                                                          lda,
                                                          strideA,
                                                          x,
                                                          incx,
                                                          stridex,
                                                          beta,
                                                          y,
                                                          incy,
                                                          stridey,
                                                          batchCount));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDgemvStridedBatched(hipblasHandle_t    handle,
                                           hipblasOperation_t trans,
                                           int                m,
                                           int                n,
                                           const double*      alpha,
                                           const double*      A,
                                           int                lda,
                                           hipblasStride      strideA,
                                           const double*      x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           const double*      beta,
                                           double*            y,
                                           int                incy,
                                           hipblasStride      stridey,
                                           int                batchCount)
try
{
#if CUBLAS_VERSION >= 110700
    return hipblasConvertStatus(cublasDgemvStridedBatched((cublasHandle_t)handle,
                                                          hipblasConvertOperation(trans),
                                                          m,
                                                          n,
                                                          alpha,
                                                          A,
                                                          lda,
                                                          strideA,
                                                          x,
                                                          incx,
                                                          stridex,
                                                          beta,
                                                          y,
                                                          incy,
                                                          stridey,
                                                          batchCount));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgemvStridedBatched(hipblasHandle_t       handle,
                                           hipblasOperation_t    trans,
                                           int                   m,
                                           int                   n,
                                           const hipblasComplex* alpha,
                                           const hipblasComplex* A,
                                           int                   lda,
                                           hipblasStride         strideA,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           const hipblasComplex* beta,
                                           hipblasComplex*       y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           int                   batchCount)
try
{
#if CUBLAS_VERSION >= 110700
    return hipblasConvertStatus(cublasCgemvStridedBatched((cublasHandle_t)handle,
                                                          hipblasConvertOperation(trans),
                                                          m,
                                                          n,
                                                          (const cuComplex*)alpha,
                                                          (const cuComplex*)A,
                                                          lda,
                                                          strideA,
                                                          (const cuComplex*)x,
                                                          incx,
                                                          stridex,
                                                          (const cuComplex*)beta,
                                                          (cuComplex*)y,
                                                          incy,
                                                          stridey,
                                                          batchCount));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgemvStridedBatched(hipblasHandle_t             handle,
                                           hipblasOperation_t          trans,
                                           int                         m,
                                           int                         n,
                                           const hipblasDoubleComplex* alpha,
                                           const hipblasDoubleComplex* A,
                                           int                         lda,
                                           hipblasStride               strideA,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           const hipblasDoubleComplex* beta,
                                           hipblasDoubleComplex*       y,
                                           int                         incy,
                                           hipblasStride               stridey,
                                           int                         batchCount)
try
{
#if CUBLAS_VERSION >= 110700
    return hipblasConvertStatus(cublasZgemvStridedBatched((cublasHandle_t)handle,
                                                          hipblasConvertOperation(trans),
                                                          m,
                                                          n,
                                                          (const cuDoubleComplex*)alpha,
                                                          (const cuDoubleComplex*)A,
                                                          lda,
                                                          strideA,
                                                          (const cuDoubleComplex*)x,
                                                          incx,
                                                          stridex,
                                                          (const cuDoubleComplex*)beta,
                                                          (cuDoubleComplex*)y,
                                                          incy,
                                                          stridey,
                                                          batchCount));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgemvStridedBatched_v2(hipblasHandle_t    handle,
                                              hipblasOperation_t trans,
                                              int                m,
                                              int                n,
                                              const hipComplex*  alpha,
                                              const hipComplex*  A,
                                              int                lda,
                                              hipblasStride      strideA,
                                              const hipComplex*  x,
                                              int                incx,
                                              hipblasStride      stridex,
                                              const hipComplex*  beta,
                                              hipComplex*        y,
                                              int                incy,
                                              hipblasStride      stridey,
                                              int                batchCount)
try
{
#if CUBLAS_VERSION >= 110700
    return hipblasConvertStatus(cublasCgemvStridedBatched((cublasHandle_t)handle,
                                                          hipblasConvertOperation(trans),
                                                          m,
                                                          n,
                                                          (const cuComplex*)alpha,
                                                          (const cuComplex*)A,
                                                          lda,
                                                          strideA,
                                                          (const cuComplex*)x,
                                                          incx,
                                                          stridex,
                                                          (const cuComplex*)beta,
                                                          (cuComplex*)y,
                                                          incy,
                                                          stridey,
                                                          batchCount));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgemvStridedBatched_v2(hipblasHandle_t         handle,
                                              hipblasOperation_t      trans,
                                              int                     m,
                                              int                     n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* A,
                                              int                     lda,
                                              hipblasStride           strideA,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* beta,
                                              hipDoubleComplex*       y,
                                              int                     incy,
                                              hipblasStride           stridey,
                                              int                     batchCount)
try
{
#if CUBLAS_VERSION >= 110700
    return hipblasConvertStatus(cublasZgemvStridedBatched((cublasHandle_t)handle,
                                                          hipblasConvertOperation(trans),
                                                          m,
                                                          n,
                                                          (const cuDoubleComplex*)alpha,
                                                          (const cuDoubleComplex*)A,
                                                          lda,
                                                          strideA,
                                                          (const cuDoubleComplex*)x,
                                                          incx,
                                                          stridex,
                                                          (const cuDoubleComplex*)beta,
                                                          (cuDoubleComplex*)y,
                                                          incy,
                                                          stridey,
                                                          batchCount));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// gemv_strided_batched_64
hipblasStatus_t hipblasSgemvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasOperation_t trans,
                                              int64_t            m,
                                              int64_t            n,
                                              const float*       alpha,
                                              const float*       A,
                                              int64_t            lda,
                                              hipblasStride      strideA,
                                              const float*       x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              const float*       beta,
                                              float*             y,
                                              int64_t            incy,
                                              hipblasStride      stridey,
                                              int64_t            batchCount)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasSgemvStridedBatched_64((cublasHandle_t)handle,
                                                             hipblasConvertOperation(trans),
                                                             m,
                                                             n,
                                                             alpha,
                                                             A,
                                                             lda,
                                                             strideA,
                                                             x,
                                                             incx,
                                                             stridex,
                                                             beta,
                                                             y,
                                                             incy,
                                                             stridey,
                                                             batchCount));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDgemvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasOperation_t trans,
                                              int64_t            m,
                                              int64_t            n,
                                              const double*      alpha,
                                              const double*      A,
                                              int64_t            lda,
                                              hipblasStride      strideA,
                                              const double*      x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              const double*      beta,
                                              double*            y,
                                              int64_t            incy,
                                              hipblasStride      stridey,
                                              int64_t            batchCount)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasDgemvStridedBatched_64((cublasHandle_t)handle,
                                                             hipblasConvertOperation(trans),
                                                             m,
                                                             n,
                                                             alpha,
                                                             A,
                                                             lda,
                                                             strideA,
                                                             x,
                                                             incx,
                                                             stridex,
                                                             beta,
                                                             y,
                                                             incy,
                                                             stridey,
                                                             batchCount));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgemvStridedBatched_64(hipblasHandle_t       handle,
                                              hipblasOperation_t    trans,
                                              int64_t               m,
                                              int64_t               n,
                                              const hipblasComplex* alpha,
                                              const hipblasComplex* A,
                                              int64_t               lda,
                                              hipblasStride         strideA,
                                              const hipblasComplex* x,
                                              int64_t               incx,
                                              hipblasStride         stridex,
                                              const hipblasComplex* beta,
                                              hipblasComplex*       y,
                                              int64_t               incy,
                                              hipblasStride         stridey,
                                              int64_t               batchCount)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCgemvStridedBatched_64((cublasHandle_t)handle,
                                                             hipblasConvertOperation(trans),
                                                             m,
                                                             n,
                                                             (const cuComplex*)alpha,
                                                             (const cuComplex*)A,
                                                             lda,
                                                             strideA,
                                                             (const cuComplex*)x,
                                                             incx,
                                                             stridex,
                                                             (const cuComplex*)beta,
                                                             (cuComplex*)y,
                                                             incy,
                                                             stridey,
                                                             batchCount));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgemvStridedBatched_64(hipblasHandle_t             handle,
                                              hipblasOperation_t          trans,
                                              int64_t                     m,
                                              int64_t                     n,
                                              const hipblasDoubleComplex* alpha,
                                              const hipblasDoubleComplex* A,
                                              int64_t                     lda,
                                              hipblasStride               strideA,
                                              const hipblasDoubleComplex* x,
                                              int64_t                     incx,
                                              hipblasStride               stridex,
                                              const hipblasDoubleComplex* beta,
                                              hipblasDoubleComplex*       y,
                                              int64_t                     incy,
                                              hipblasStride               stridey,
                                              int64_t                     batchCount)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZgemvStridedBatched_64((cublasHandle_t)handle,
                                                             hipblasConvertOperation(trans),
                                                             m,
                                                             n,
                                                             (const cuDoubleComplex*)alpha,
                                                             (const cuDoubleComplex*)A,
                                                             lda,
                                                             strideA,
                                                             (const cuDoubleComplex*)x,
                                                             incx,
                                                             stridex,
                                                             (const cuDoubleComplex*)beta,
                                                             (cuDoubleComplex*)y,
                                                             incy,
                                                             stridey,
                                                             batchCount));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgemvStridedBatched_v2_64(hipblasHandle_t    handle,
                                                 hipblasOperation_t trans,
                                                 int64_t            m,
                                                 int64_t            n,
                                                 const hipComplex*  alpha,
                                                 const hipComplex*  A,
                                                 int64_t            lda,
                                                 hipblasStride      strideA,
                                                 const hipComplex*  x,
                                                 int64_t            incx,
                                                 hipblasStride      stridex,
                                                 const hipComplex*  beta,
                                                 hipComplex*        y,
                                                 int64_t            incy,
                                                 hipblasStride      stridey,
                                                 int64_t            batchCount)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCgemvStridedBatched_64((cublasHandle_t)handle,
                                                             hipblasConvertOperation(trans),
                                                             m,
                                                             n,
                                                             (const cuComplex*)alpha,
                                                             (const cuComplex*)A,
                                                             lda,
                                                             strideA,
                                                             (const cuComplex*)x,
                                                             incx,
                                                             stridex,
                                                             (const cuComplex*)beta,
                                                             (cuComplex*)y,
                                                             incy,
                                                             stridey,
                                                             batchCount));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgemvStridedBatched_v2_64(hipblasHandle_t         handle,
                                                 hipblasOperation_t      trans,
                                                 int64_t                 m,
                                                 int64_t                 n,
                                                 const hipDoubleComplex* alpha,
                                                 const hipDoubleComplex* A,
                                                 int64_t                 lda,
                                                 hipblasStride           strideA,
                                                 const hipDoubleComplex* x,
                                                 int64_t                 incx,
                                                 hipblasStride           stridex,
                                                 const hipDoubleComplex* beta,
                                                 hipDoubleComplex*       y,
                                                 int64_t                 incy,
                                                 hipblasStride           stridey,
                                                 int64_t                 batchCount)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZgemvStridedBatched_64((cublasHandle_t)handle,
                                                             hipblasConvertOperation(trans),
                                                             m,
                                                             n,
                                                             (const cuDoubleComplex*)alpha,
                                                             (const cuDoubleComplex*)A,
                                                             lda,
                                                             strideA,
                                                             (const cuDoubleComplex*)x,
                                                             incx,
                                                             stridex,
                                                             (const cuDoubleComplex*)beta,
                                                             (cuDoubleComplex*)y,
                                                             incy,
                                                             stridey,
                                                             batchCount));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// ger
hipblasStatus_t hipblasSger(hipblasHandle_t handle,
                            int             m,
                            int             n,
                            const float*    alpha,
                            const float*    x,
                            int             incx,
                            const float*    y,
                            int             incy,
                            float*          A,
                            int             lda)
try
{
    return hipblasConvertStatus(
        cublasSger((cublasHandle_t)handle, m, n, alpha, x, incx, y, incy, A, lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDger(hipblasHandle_t handle,
                            int             m,
                            int             n,
                            const double*   alpha,
                            const double*   x,
                            int             incx,
                            const double*   y,
                            int             incy,
                            double*         A,
                            int             lda)
try
{
    return hipblasConvertStatus(
        cublasDger((cublasHandle_t)handle, m, n, alpha, x, incx, y, incy, A, lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgeru(hipblasHandle_t       handle,
                             int                   m,
                             int                   n,
                             const hipblasComplex* alpha,
                             const hipblasComplex* x,
                             int                   incx,
                             const hipblasComplex* y,
                             int                   incy,
                             hipblasComplex*       A,
                             int                   lda)
try
{
    return hipblasConvertStatus(cublasCgeru((cublasHandle_t)handle,
                                            m,
                                            n,
                                            (cuComplex*)alpha,
                                            (cuComplex*)x,
                                            incx,
                                            (cuComplex*)y,
                                            incy,
                                            (cuComplex*)A,
                                            lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgerc(hipblasHandle_t       handle,
                             int                   m,
                             int                   n,
                             const hipblasComplex* alpha,
                             const hipblasComplex* x,
                             int                   incx,
                             const hipblasComplex* y,
                             int                   incy,
                             hipblasComplex*       A,
                             int                   lda)
try
{
    return hipblasConvertStatus(cublasCgerc((cublasHandle_t)handle,
                                            m,
                                            n,
                                            (cuComplex*)alpha,
                                            (cuComplex*)x,
                                            incx,
                                            (cuComplex*)y,
                                            incy,
                                            (cuComplex*)A,
                                            lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgeru(hipblasHandle_t             handle,
                             int                         m,
                             int                         n,
                             const hipblasDoubleComplex* alpha,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             const hipblasDoubleComplex* y,
                             int                         incy,
                             hipblasDoubleComplex*       A,
                             int                         lda)
try
{
    return hipblasConvertStatus(cublasZgeru((cublasHandle_t)handle,
                                            m,
                                            n,
                                            (cuDoubleComplex*)alpha,
                                            (cuDoubleComplex*)x,
                                            incx,
                                            (cuDoubleComplex*)y,
                                            incy,
                                            (cuDoubleComplex*)A,
                                            lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgerc(hipblasHandle_t             handle,
                             int                         m,
                             int                         n,
                             const hipblasDoubleComplex* alpha,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             const hipblasDoubleComplex* y,
                             int                         incy,
                             hipblasDoubleComplex*       A,
                             int                         lda)
try
{
    return hipblasConvertStatus(cublasZgerc((cublasHandle_t)handle,
                                            m,
                                            n,
                                            (cuDoubleComplex*)alpha,
                                            (cuDoubleComplex*)x,
                                            incx,
                                            (cuDoubleComplex*)y,
                                            incy,
                                            (cuDoubleComplex*)A,
                                            lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgeru_v2(hipblasHandle_t   handle,
                                int               m,
                                int               n,
                                const hipComplex* alpha,
                                const hipComplex* x,
                                int               incx,
                                const hipComplex* y,
                                int               incy,
                                hipComplex*       A,
                                int               lda)
try
{
    return hipblasConvertStatus(cublasCgeru((cublasHandle_t)handle,
                                            m,
                                            n,
                                            (cuComplex*)alpha,
                                            (cuComplex*)x,
                                            incx,
                                            (cuComplex*)y,
                                            incy,
                                            (cuComplex*)A,
                                            lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgerc_v2(hipblasHandle_t   handle,
                                int               m,
                                int               n,
                                const hipComplex* alpha,
                                const hipComplex* x,
                                int               incx,
                                const hipComplex* y,
                                int               incy,
                                hipComplex*       A,
                                int               lda)
try
{
    return hipblasConvertStatus(cublasCgerc((cublasHandle_t)handle,
                                            m,
                                            n,
                                            (cuComplex*)alpha,
                                            (cuComplex*)x,
                                            incx,
                                            (cuComplex*)y,
                                            incy,
                                            (cuComplex*)A,
                                            lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgeru_v2(hipblasHandle_t         handle,
                                int                     m,
                                int                     n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* x,
                                int                     incx,
                                const hipDoubleComplex* y,
                                int                     incy,
                                hipDoubleComplex*       A,
                                int                     lda)
try
{
    return hipblasConvertStatus(cublasZgeru((cublasHandle_t)handle,
                                            m,
                                            n,
                                            (cuDoubleComplex*)alpha,
                                            (cuDoubleComplex*)x,
                                            incx,
                                            (cuDoubleComplex*)y,
                                            incy,
                                            (cuDoubleComplex*)A,
                                            lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgerc_v2(hipblasHandle_t         handle,
                                int                     m,
                                int                     n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* x,
                                int                     incx,
                                const hipDoubleComplex* y,
                                int                     incy,
                                hipDoubleComplex*       A,
                                int                     lda)
try
{
    return hipblasConvertStatus(cublasZgerc((cublasHandle_t)handle,
                                            m,
                                            n,
                                            (cuDoubleComplex*)alpha,
                                            (cuDoubleComplex*)x,
                                            incx,
                                            (cuDoubleComplex*)y,
                                            incy,
                                            (cuDoubleComplex*)A,
                                            lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// ger_64
hipblasStatus_t hipblasSger_64(hipblasHandle_t handle,
                               int64_t         m,
                               int64_t         n,
                               const float*    alpha,
                               const float*    x,
                               int64_t         incx,
                               const float*    y,
                               int64_t         incy,
                               float*          A,
                               int64_t         lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasSger_64((cublasHandle_t)handle, m, n, alpha, x, incx, y, incy, A, lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDger_64(hipblasHandle_t handle,
                               int64_t         m,
                               int64_t         n,
                               const double*   alpha,
                               const double*   x,
                               int64_t         incx,
                               const double*   y,
                               int64_t         incy,
                               double*         A,
                               int64_t         lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasDger_64((cublasHandle_t)handle, m, n, alpha, x, incx, y, incy, A, lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgeru_64(hipblasHandle_t       handle,
                                int64_t               m,
                                int64_t               n,
                                const hipblasComplex* alpha,
                                const hipblasComplex* x,
                                int64_t               incx,
                                const hipblasComplex* y,
                                int64_t               incy,
                                hipblasComplex*       A,
                                int64_t               lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCgeru_64((cublasHandle_t)handle,
                                               m,
                                               n,
                                               (cuComplex*)alpha,
                                               (cuComplex*)x,
                                               incx,
                                               (cuComplex*)y,
                                               incy,
                                               (cuComplex*)A,
                                               lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgerc_64(hipblasHandle_t       handle,
                                int64_t               m,
                                int64_t               n,
                                const hipblasComplex* alpha,
                                const hipblasComplex* x,
                                int64_t               incx,
                                const hipblasComplex* y,
                                int64_t               incy,
                                hipblasComplex*       A,
                                int64_t               lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCgerc((cublasHandle_t)handle,
                                            m,
                                            n,
                                            (cuComplex*)alpha,
                                            (cuComplex*)x,
                                            incx,
                                            (cuComplex*)y,
                                            incy,
                                            (cuComplex*)A,
                                            lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgeru_64(hipblasHandle_t             handle,
                                int64_t                     m,
                                int64_t                     n,
                                const hipblasDoubleComplex* alpha,
                                const hipblasDoubleComplex* x,
                                int64_t                     incx,
                                const hipblasDoubleComplex* y,
                                int64_t                     incy,
                                hipblasDoubleComplex*       A,
                                int64_t                     lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZgeru_64((cublasHandle_t)handle,
                                               m,
                                               n,
                                               (cuDoubleComplex*)alpha,
                                               (cuDoubleComplex*)x,
                                               incx,
                                               (cuDoubleComplex*)y,
                                               incy,
                                               (cuDoubleComplex*)A,
                                               lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgerc_64(hipblasHandle_t             handle,
                                int64_t                     m,
                                int64_t                     n,
                                const hipblasDoubleComplex* alpha,
                                const hipblasDoubleComplex* x,
                                int64_t                     incx,
                                const hipblasDoubleComplex* y,
                                int64_t                     incy,
                                hipblasDoubleComplex*       A,
                                int64_t                     lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZgerc_64((cublasHandle_t)handle,
                                               m,
                                               n,
                                               (cuDoubleComplex*)alpha,
                                               (cuDoubleComplex*)x,
                                               incx,
                                               (cuDoubleComplex*)y,
                                               incy,
                                               (cuDoubleComplex*)A,
                                               lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgeru_v2_64(hipblasHandle_t   handle,
                                   int64_t           m,
                                   int64_t           n,
                                   const hipComplex* alpha,
                                   const hipComplex* x,
                                   int64_t           incx,
                                   const hipComplex* y,
                                   int64_t           incy,
                                   hipComplex*       A,
                                   int64_t           lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCgeru_64((cublasHandle_t)handle,
                                               m,
                                               n,
                                               (cuComplex*)alpha,
                                               (cuComplex*)x,
                                               incx,
                                               (cuComplex*)y,
                                               incy,
                                               (cuComplex*)A,
                                               lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgerc_v2_64(hipblasHandle_t   handle,
                                   int64_t           m,
                                   int64_t           n,
                                   const hipComplex* alpha,
                                   const hipComplex* x,
                                   int64_t           incx,
                                   const hipComplex* y,
                                   int64_t           incy,
                                   hipComplex*       A,
                                   int64_t           lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCgerc_64((cublasHandle_t)handle,
                                               m,
                                               n,
                                               (cuComplex*)alpha,
                                               (cuComplex*)x,
                                               incx,
                                               (cuComplex*)y,
                                               incy,
                                               (cuComplex*)A,
                                               lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgeru_v2_64(hipblasHandle_t         handle,
                                   int64_t                 m,
                                   int64_t                 n,
                                   const hipDoubleComplex* alpha,
                                   const hipDoubleComplex* x,
                                   int64_t                 incx,
                                   const hipDoubleComplex* y,
                                   int64_t                 incy,
                                   hipDoubleComplex*       A,
                                   int64_t                 lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZgeru_64((cublasHandle_t)handle,
                                               m,
                                               n,
                                               (cuDoubleComplex*)alpha,
                                               (cuDoubleComplex*)x,
                                               incx,
                                               (cuDoubleComplex*)y,
                                               incy,
                                               (cuDoubleComplex*)A,
                                               lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgerc_v2_64(hipblasHandle_t         handle,
                                   int64_t                 m,
                                   int64_t                 n,
                                   const hipDoubleComplex* alpha,
                                   const hipDoubleComplex* x,
                                   int64_t                 incx,
                                   const hipDoubleComplex* y,
                                   int64_t                 incy,
                                   hipDoubleComplex*       A,
                                   int64_t                 lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZgerc_64((cublasHandle_t)handle,
                                               m,
                                               n,
                                               (cuDoubleComplex*)alpha,
                                               (cuDoubleComplex*)x,
                                               incx,
                                               (cuDoubleComplex*)y,
                                               incy,
                                               (cuDoubleComplex*)A,
                                               lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// ger_batched
hipblasStatus_t hipblasSgerBatched(hipblasHandle_t    handle,
                                   int                m,
                                   int                n,
                                   const float*       alpha,
                                   const float* const x[],
                                   int                incx,
                                   const float* const y[],
                                   int                incy,
                                   float* const       A[],
                                   int                lda,
                                   int                batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDgerBatched(hipblasHandle_t     handle,
                                   int                 m,
                                   int                 n,
                                   const double*       alpha,
                                   const double* const x[],
                                   int                 incx,
                                   const double* const y[],
                                   int                 incy,
                                   double* const       A[],
                                   int                 lda,
                                   int                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCgeruBatched(hipblasHandle_t             handle,
                                    int                         m,
                                    int                         n,
                                    const hipblasComplex*       alpha,
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    const hipblasComplex* const y[],
                                    int                         incy,
                                    hipblasComplex* const       A[],
                                    int                         lda,
                                    int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCgercBatched(hipblasHandle_t             handle,
                                    int                         m,
                                    int                         n,
                                    const hipblasComplex*       alpha,
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    const hipblasComplex* const y[],
                                    int                         incy,
                                    hipblasComplex* const       A[],
                                    int                         lda,
                                    int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZgeruBatched(hipblasHandle_t                   handle,
                                    int                               m,
                                    int                               n,
                                    const hipblasDoubleComplex*       alpha,
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    const hipblasDoubleComplex* const y[],
                                    int                               incy,
                                    hipblasDoubleComplex* const       A[],
                                    int                               lda,
                                    int                               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZgercBatched(hipblasHandle_t                   handle,
                                    int                               m,
                                    int                               n,
                                    const hipblasDoubleComplex*       alpha,
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    const hipblasDoubleComplex* const y[],
                                    int                               incy,
                                    hipblasDoubleComplex* const       A[],
                                    int                               lda,
                                    int                               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCgeruBatched_v2(hipblasHandle_t         handle,
                                       int                     m,
                                       int                     n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const x[],
                                       int                     incx,
                                       const hipComplex* const y[],
                                       int                     incy,
                                       hipComplex* const       A[],
                                       int                     lda,
                                       int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCgercBatched_v2(hipblasHandle_t         handle,
                                       int                     m,
                                       int                     n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const x[],
                                       int                     incx,
                                       const hipComplex* const y[],
                                       int                     incy,
                                       hipComplex* const       A[],
                                       int                     lda,
                                       int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZgeruBatched_v2(hipblasHandle_t               handle,
                                       int                           m,
                                       int                           n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       const hipDoubleComplex* const y[],
                                       int                           incy,
                                       hipDoubleComplex* const       A[],
                                       int                           lda,
                                       int                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZgercBatched_v2(hipblasHandle_t               handle,
                                       int                           m,
                                       int                           n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       const hipDoubleComplex* const y[],
                                       int                           incy,
                                       hipDoubleComplex* const       A[],
                                       int                           lda,
                                       int                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// ger_batched_64
hipblasStatus_t hipblasSgerBatched_64(hipblasHandle_t    handle,
                                      int64_t            m,
                                      int64_t            n,
                                      const float*       alpha,
                                      const float* const x[],
                                      int64_t            incx,
                                      const float* const y[],
                                      int64_t            incy,
                                      float* const       A[],
                                      int64_t            lda,
                                      int64_t            batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDgerBatched_64(hipblasHandle_t     handle,
                                      int64_t             m,
                                      int64_t             n,
                                      const double*       alpha,
                                      const double* const x[],
                                      int64_t             incx,
                                      const double* const y[],
                                      int64_t             incy,
                                      double* const       A[],
                                      int64_t             lda,
                                      int64_t             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCgeruBatched_64(hipblasHandle_t             handle,
                                       int64_t                     m,
                                       int64_t                     n,
                                       const hipblasComplex*       alpha,
                                       const hipblasComplex* const x[],
                                       int64_t                     incx,
                                       const hipblasComplex* const y[],
                                       int64_t                     incy,
                                       hipblasComplex* const       A[],
                                       int64_t                     lda,
                                       int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCgercBatched_64(hipblasHandle_t             handle,
                                       int64_t                     m,
                                       int64_t                     n,
                                       const hipblasComplex*       alpha,
                                       const hipblasComplex* const x[],
                                       int64_t                     incx,
                                       const hipblasComplex* const y[],
                                       int64_t                     incy,
                                       hipblasComplex* const       A[],
                                       int64_t                     lda,
                                       int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZgeruBatched_64(hipblasHandle_t                   handle,
                                       int64_t                           m,
                                       int64_t                           n,
                                       const hipblasDoubleComplex*       alpha,
                                       const hipblasDoubleComplex* const x[],
                                       int64_t                           incx,
                                       const hipblasDoubleComplex* const y[],
                                       int64_t                           incy,
                                       hipblasDoubleComplex* const       A[],
                                       int64_t                           lda,
                                       int64_t                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZgercBatched_64(hipblasHandle_t                   handle,
                                       int64_t                           m,
                                       int64_t                           n,
                                       const hipblasDoubleComplex*       alpha,
                                       const hipblasDoubleComplex* const x[],
                                       int64_t                           incx,
                                       const hipblasDoubleComplex* const y[],
                                       int64_t                           incy,
                                       hipblasDoubleComplex* const       A[],
                                       int64_t                           lda,
                                       int64_t                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCgeruBatched_v2_64(hipblasHandle_t         handle,
                                          int64_t                 m,
                                          int64_t                 n,
                                          const hipComplex*       alpha,
                                          const hipComplex* const x[],
                                          int64_t                 incx,
                                          const hipComplex* const y[],
                                          int64_t                 incy,
                                          hipComplex* const       A[],
                                          int64_t                 lda,
                                          int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCgercBatched_v2_64(hipblasHandle_t         handle,
                                          int64_t                 m,
                                          int64_t                 n,
                                          const hipComplex*       alpha,
                                          const hipComplex* const x[],
                                          int64_t                 incx,
                                          const hipComplex* const y[],
                                          int64_t                 incy,
                                          hipComplex* const       A[],
                                          int64_t                 lda,
                                          int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZgeruBatched_v2_64(hipblasHandle_t               handle,
                                          int64_t                       m,
                                          int64_t                       n,
                                          const hipDoubleComplex*       alpha,
                                          const hipDoubleComplex* const x[],
                                          int64_t                       incx,
                                          const hipDoubleComplex* const y[],
                                          int64_t                       incy,
                                          hipDoubleComplex* const       A[],
                                          int64_t                       lda,
                                          int64_t                       batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZgercBatched_v2_64(hipblasHandle_t               handle,
                                          int64_t                       m,
                                          int64_t                       n,
                                          const hipDoubleComplex*       alpha,
                                          const hipDoubleComplex* const x[],
                                          int64_t                       incx,
                                          const hipDoubleComplex* const y[],
                                          int64_t                       incy,
                                          hipDoubleComplex* const       A[],
                                          int64_t                       lda,
                                          int64_t                       batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// ger_strided_batched
hipblasStatus_t hipblasSgerStridedBatched(hipblasHandle_t handle,
                                          int             m,
                                          int             n,
                                          const float*    alpha,
                                          const float*    x,
                                          int             incx,
                                          hipblasStride   stridex,
                                          const float*    y,
                                          int             incy,
                                          hipblasStride   stridey,
                                          float*          A,
                                          int             lda,
                                          hipblasStride   strideA,
                                          int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDgerStridedBatched(hipblasHandle_t handle,
                                          int             m,
                                          int             n,
                                          const double*   alpha,
                                          const double*   x,
                                          int             incx,
                                          hipblasStride   stridex,
                                          const double*   y,
                                          int             incy,
                                          hipblasStride   stridey,
                                          double*         A,
                                          int             lda,
                                          hipblasStride   strideA,
                                          int             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCgeruStridedBatched(hipblasHandle_t       handle,
                                           int                   m,
                                           int                   n,
                                           const hipblasComplex* alpha,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           const hipblasComplex* y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           hipblasComplex*       A,
                                           int                   lda,
                                           hipblasStride         strideA,
                                           int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCgercStridedBatched(hipblasHandle_t       handle,
                                           int                   m,
                                           int                   n,
                                           const hipblasComplex* alpha,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           const hipblasComplex* y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           hipblasComplex*       A,
                                           int                   lda,
                                           hipblasStride         strideA,
                                           int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZgeruStridedBatched(hipblasHandle_t             handle,
                                           int                         m,
                                           int                         n,
                                           const hipblasDoubleComplex* alpha,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           const hipblasDoubleComplex* y,
                                           int                         incy,
                                           hipblasStride               stridey,
                                           hipblasDoubleComplex*       A,
                                           int                         lda,
                                           hipblasStride               strideA,
                                           int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZgercStridedBatched(hipblasHandle_t             handle,
                                           int                         m,
                                           int                         n,
                                           const hipblasDoubleComplex* alpha,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           const hipblasDoubleComplex* y,
                                           int                         incy,
                                           hipblasStride               stridey,
                                           hipblasDoubleComplex*       A,
                                           int                         lda,
                                           hipblasStride               strideA,
                                           int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCgeruStridedBatched_v2(hipblasHandle_t   handle,
                                              int               m,
                                              int               n,
                                              const hipComplex* alpha,
                                              const hipComplex* x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              const hipComplex* y,
                                              int               incy,
                                              hipblasStride     stridey,
                                              hipComplex*       A,
                                              int               lda,
                                              hipblasStride     strideA,
                                              int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCgercStridedBatched_v2(hipblasHandle_t   handle,
                                              int               m,
                                              int               n,
                                              const hipComplex* alpha,
                                              const hipComplex* x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              const hipComplex* y,
                                              int               incy,
                                              hipblasStride     stridey,
                                              hipComplex*       A,
                                              int               lda,
                                              hipblasStride     strideA,
                                              int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZgeruStridedBatched_v2(hipblasHandle_t         handle,
                                              int                     m,
                                              int                     n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* y,
                                              int                     incy,
                                              hipblasStride           stridey,
                                              hipDoubleComplex*       A,
                                              int                     lda,
                                              hipblasStride           strideA,
                                              int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZgercStridedBatched_v2(hipblasHandle_t         handle,
                                              int                     m,
                                              int                     n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* y,
                                              int                     incy,
                                              hipblasStride           stridey,
                                              hipDoubleComplex*       A,
                                              int                     lda,
                                              hipblasStride           strideA,
                                              int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// ger_strided_batched_64
hipblasStatus_t hipblasSgerStridedBatched_64(hipblasHandle_t handle,
                                             int64_t         m,
                                             int64_t         n,
                                             const float*    alpha,
                                             const float*    x,
                                             int64_t         incx,
                                             hipblasStride   stridex,
                                             const float*    y,
                                             int64_t         incy,
                                             hipblasStride   stridey,
                                             float*          A,
                                             int64_t         lda,
                                             hipblasStride   strideA,
                                             int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDgerStridedBatched_64(hipblasHandle_t handle,
                                             int64_t         m,
                                             int64_t         n,
                                             const double*   alpha,
                                             const double*   x,
                                             int64_t         incx,
                                             hipblasStride   stridex,
                                             const double*   y,
                                             int64_t         incy,
                                             hipblasStride   stridey,
                                             double*         A,
                                             int64_t         lda,
                                             hipblasStride   strideA,
                                             int64_t         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCgeruStridedBatched_64(hipblasHandle_t       handle,
                                              int64_t               m,
                                              int64_t               n,
                                              const hipblasComplex* alpha,
                                              const hipblasComplex* x,
                                              int64_t               incx,
                                              hipblasStride         stridex,
                                              const hipblasComplex* y,
                                              int64_t               incy,
                                              hipblasStride         stridey,
                                              hipblasComplex*       A,
                                              int64_t               lda,
                                              hipblasStride         strideA,
                                              int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCgercStridedBatched_64(hipblasHandle_t       handle,
                                              int64_t               m,
                                              int64_t               n,
                                              const hipblasComplex* alpha,
                                              const hipblasComplex* x,
                                              int64_t               incx,
                                              hipblasStride         stridex,
                                              const hipblasComplex* y,
                                              int64_t               incy,
                                              hipblasStride         stridey,
                                              hipblasComplex*       A,
                                              int64_t               lda,
                                              hipblasStride         strideA,
                                              int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZgeruStridedBatched_64(hipblasHandle_t             handle,
                                              int64_t                     m,
                                              int64_t                     n,
                                              const hipblasDoubleComplex* alpha,
                                              const hipblasDoubleComplex* x,
                                              int64_t                     incx,
                                              hipblasStride               stridex,
                                              const hipblasDoubleComplex* y,
                                              int64_t                     incy,
                                              hipblasStride               stridey,
                                              hipblasDoubleComplex*       A,
                                              int64_t                     lda,
                                              hipblasStride               strideA,
                                              int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZgercStridedBatched_64(hipblasHandle_t             handle,
                                              int64_t                     m,
                                              int64_t                     n,
                                              const hipblasDoubleComplex* alpha,
                                              const hipblasDoubleComplex* x,
                                              int64_t                     incx,
                                              hipblasStride               stridex,
                                              const hipblasDoubleComplex* y,
                                              int64_t                     incy,
                                              hipblasStride               stridey,
                                              hipblasDoubleComplex*       A,
                                              int64_t                     lda,
                                              hipblasStride               strideA,
                                              int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCgeruStridedBatched_v2_64(hipblasHandle_t   handle,
                                                 int64_t           m,
                                                 int64_t           n,
                                                 const hipComplex* alpha,
                                                 const hipComplex* x,
                                                 int64_t           incx,
                                                 hipblasStride     stridex,
                                                 const hipComplex* y,
                                                 int64_t           incy,
                                                 hipblasStride     stridey,
                                                 hipComplex*       A,
                                                 int64_t           lda,
                                                 hipblasStride     strideA,
                                                 int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCgercStridedBatched_v2_64(hipblasHandle_t   handle,
                                                 int64_t           m,
                                                 int64_t           n,
                                                 const hipComplex* alpha,
                                                 const hipComplex* x,
                                                 int64_t           incx,
                                                 hipblasStride     stridex,
                                                 const hipComplex* y,
                                                 int64_t           incy,
                                                 hipblasStride     stridey,
                                                 hipComplex*       A,
                                                 int64_t           lda,
                                                 hipblasStride     strideA,
                                                 int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZgeruStridedBatched_v2_64(hipblasHandle_t         handle,
                                                 int64_t                 m,
                                                 int64_t                 n,
                                                 const hipDoubleComplex* alpha,
                                                 const hipDoubleComplex* x,
                                                 int64_t                 incx,
                                                 hipblasStride           stridex,
                                                 const hipDoubleComplex* y,
                                                 int64_t                 incy,
                                                 hipblasStride           stridey,
                                                 hipDoubleComplex*       A,
                                                 int64_t                 lda,
                                                 hipblasStride           strideA,
                                                 int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZgercStridedBatched_v2_64(hipblasHandle_t         handle,
                                                 int64_t                 m,
                                                 int64_t                 n,
                                                 const hipDoubleComplex* alpha,
                                                 const hipDoubleComplex* x,
                                                 int64_t                 incx,
                                                 hipblasStride           stridex,
                                                 const hipDoubleComplex* y,
                                                 int64_t                 incy,
                                                 hipblasStride           stridey,
                                                 hipDoubleComplex*       A,
                                                 int64_t                 lda,
                                                 hipblasStride           strideA,
                                                 int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// hbmv
hipblasStatus_t hipblasChbmv(hipblasHandle_t       handle,
                             hipblasFillMode_t     uplo,
                             int                   n,
                             int                   k,
                             const hipblasComplex* alpha,
                             const hipblasComplex* A,
                             int                   lda,
                             const hipblasComplex* x,
                             int                   incx,
                             const hipblasComplex* beta,
                             hipblasComplex*       y,
                             int                   incy)
try
{
    return hipblasConvertStatus(cublasChbmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            k,
                                            (cuComplex*)alpha,
                                            (cuComplex*)A,
                                            lda,
                                            (cuComplex*)x,
                                            incx,
                                            (cuComplex*)beta,
                                            (cuComplex*)y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhbmv(hipblasHandle_t             handle,
                             hipblasFillMode_t           uplo,
                             int                         n,
                             int                         k,
                             const hipblasDoubleComplex* alpha,
                             const hipblasDoubleComplex* A,
                             int                         lda,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             const hipblasDoubleComplex* beta,
                             hipblasDoubleComplex*       y,
                             int                         incy)
try
{
    return hipblasConvertStatus(cublasZhbmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            k,
                                            (cuDoubleComplex*)alpha,
                                            (cuDoubleComplex*)A,
                                            lda,
                                            (cuDoubleComplex*)x,
                                            incx,
                                            (cuDoubleComplex*)beta,
                                            (cuDoubleComplex*)y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasChbmv_v2(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int               n,
                                int               k,
                                const hipComplex* alpha,
                                const hipComplex* A,
                                int               lda,
                                const hipComplex* x,
                                int               incx,
                                const hipComplex* beta,
                                hipComplex*       y,
                                int               incy)
try
{
    return hipblasConvertStatus(cublasChbmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            k,
                                            (cuComplex*)alpha,
                                            (cuComplex*)A,
                                            lda,
                                            (cuComplex*)x,
                                            incx,
                                            (cuComplex*)beta,
                                            (cuComplex*)y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhbmv_v2(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                int                     n,
                                int                     k,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* A,
                                int                     lda,
                                const hipDoubleComplex* x,
                                int                     incx,
                                const hipDoubleComplex* beta,
                                hipDoubleComplex*       y,
                                int                     incy)
try
{
    return hipblasConvertStatus(cublasZhbmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            k,
                                            (cuDoubleComplex*)alpha,
                                            (cuDoubleComplex*)A,
                                            lda,
                                            (cuDoubleComplex*)x,
                                            incx,
                                            (cuDoubleComplex*)beta,
                                            (cuDoubleComplex*)y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hbmv_64
hipblasStatus_t hipblasChbmv_64(hipblasHandle_t       handle,
                                hipblasFillMode_t     uplo,
                                int64_t               n,
                                int64_t               k,
                                const hipblasComplex* alpha,
                                const hipblasComplex* A,
                                int64_t               lda,
                                const hipblasComplex* x,
                                int64_t               incx,
                                const hipblasComplex* beta,
                                hipblasComplex*       y,
                                int64_t               incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasChbmv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               k,
                                               (cuComplex*)alpha,
                                               (cuComplex*)A,
                                               lda,
                                               (cuComplex*)x,
                                               incx,
                                               (cuComplex*)beta,
                                               (cuComplex*)y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhbmv_64(hipblasHandle_t             handle,
                                hipblasFillMode_t           uplo,
                                int64_t                     n,
                                int64_t                     k,
                                const hipblasDoubleComplex* alpha,
                                const hipblasDoubleComplex* A,
                                int64_t                     lda,
                                const hipblasDoubleComplex* x,
                                int64_t                     incx,
                                const hipblasDoubleComplex* beta,
                                hipblasDoubleComplex*       y,
                                int64_t                     incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZhbmv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               k,
                                               (cuDoubleComplex*)alpha,
                                               (cuDoubleComplex*)A,
                                               lda,
                                               (cuDoubleComplex*)x,
                                               incx,
                                               (cuDoubleComplex*)beta,
                                               (cuDoubleComplex*)y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasChbmv_v2_64(hipblasHandle_t   handle,
                                   hipblasFillMode_t uplo,
                                   int64_t           n,
                                   int64_t           k,
                                   const hipComplex* alpha,
                                   const hipComplex* A,
                                   int64_t           lda,
                                   const hipComplex* x,
                                   int64_t           incx,
                                   const hipComplex* beta,
                                   hipComplex*       y,
                                   int64_t           incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasChbmv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               k,
                                               (cuComplex*)alpha,
                                               (cuComplex*)A,
                                               lda,
                                               (cuComplex*)x,
                                               incx,
                                               (cuComplex*)beta,
                                               (cuComplex*)y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhbmv_v2_64(hipblasHandle_t         handle,
                                   hipblasFillMode_t       uplo,
                                   int64_t                 n,
                                   int64_t                 k,
                                   const hipDoubleComplex* alpha,
                                   const hipDoubleComplex* A,
                                   int64_t                 lda,
                                   const hipDoubleComplex* x,
                                   int64_t                 incx,
                                   const hipDoubleComplex* beta,
                                   hipDoubleComplex*       y,
                                   int64_t                 incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZhbmv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               k,
                                               (cuDoubleComplex*)alpha,
                                               (cuDoubleComplex*)A,
                                               lda,
                                               (cuDoubleComplex*)x,
                                               incx,
                                               (cuDoubleComplex*)beta,
                                               (cuDoubleComplex*)y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hbmv_batched
hipblasStatus_t hipblasChbmvBatched(hipblasHandle_t             handle,
                                    hipblasFillMode_t           uplo,
                                    int                         n,
                                    int                         k,
                                    const hipblasComplex*       alpha,
                                    const hipblasComplex* const A[],
                                    int                         lda,
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    const hipblasComplex*       beta,
                                    hipblasComplex* const       y[],
                                    int                         incy,
                                    int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhbmvBatched(hipblasHandle_t                   handle,
                                    hipblasFillMode_t                 uplo,
                                    int                               n,
                                    int                               k,
                                    const hipblasDoubleComplex*       alpha,
                                    const hipblasDoubleComplex* const A[],
                                    int                               lda,
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    const hipblasDoubleComplex*       beta,
                                    hipblasDoubleComplex* const       y[],
                                    int                               incy,
                                    int                               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasChbmvBatched_v2(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       int                     n,
                                       int                     k,
                                       const hipComplex*       alpha,
                                       const hipComplex* const A[],
                                       int                     lda,
                                       const hipComplex* const x[],
                                       int                     incx,
                                       const hipComplex*       beta,
                                       hipComplex* const       y[],
                                       int                     incy,
                                       int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhbmvBatched_v2(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       int                           n,
                                       int                           k,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const A[],
                                       int                           lda,
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       const hipDoubleComplex*       beta,
                                       hipDoubleComplex* const       y[],
                                       int                           incy,
                                       int                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// hbmv_batched_64
hipblasStatus_t hipblasChbmvBatched_64(hipblasHandle_t             handle,
                                       hipblasFillMode_t           uplo,
                                       int64_t                     n,
                                       int64_t                     k,
                                       const hipblasComplex*       alpha,
                                       const hipblasComplex* const A[],
                                       int64_t                     lda,
                                       const hipblasComplex* const x[],
                                       int64_t                     incx,
                                       const hipblasComplex*       beta,
                                       hipblasComplex* const       y[],
                                       int64_t                     incy,
                                       int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhbmvBatched_64(hipblasHandle_t                   handle,
                                       hipblasFillMode_t                 uplo,
                                       int64_t                           n,
                                       int64_t                           k,
                                       const hipblasDoubleComplex*       alpha,
                                       const hipblasDoubleComplex* const A[],
                                       int64_t                           lda,
                                       const hipblasDoubleComplex* const x[],
                                       int64_t                           incx,
                                       const hipblasDoubleComplex*       beta,
                                       hipblasDoubleComplex* const       y[],
                                       int64_t                           incy,
                                       int64_t                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasChbmvBatched_v2_64(hipblasHandle_t         handle,
                                          hipblasFillMode_t       uplo,
                                          int64_t                 n,
                                          int64_t                 k,
                                          const hipComplex*       alpha,
                                          const hipComplex* const A[],
                                          int64_t                 lda,
                                          const hipComplex* const x[],
                                          int64_t                 incx,
                                          const hipComplex*       beta,
                                          hipComplex* const       y[],
                                          int64_t                 incy,
                                          int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhbmvBatched_v2_64(hipblasHandle_t               handle,
                                          hipblasFillMode_t             uplo,
                                          int64_t                       n,
                                          int64_t                       k,
                                          const hipDoubleComplex*       alpha,
                                          const hipDoubleComplex* const A[],
                                          int64_t                       lda,
                                          const hipDoubleComplex* const x[],
                                          int64_t                       incx,
                                          const hipDoubleComplex*       beta,
                                          hipDoubleComplex* const       y[],
                                          int64_t                       incy,
                                          int64_t                       batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// hbmv_strided_batched
hipblasStatus_t hipblasChbmvStridedBatched(hipblasHandle_t       handle,
                                           hipblasFillMode_t     uplo,
                                           int                   n,
                                           int                   k,
                                           const hipblasComplex* alpha,
                                           const hipblasComplex* A,
                                           int                   lda,
                                           hipblasStride         strideA,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           const hipblasComplex* beta,
                                           hipblasComplex*       y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhbmvStridedBatched(hipblasHandle_t             handle,
                                           hipblasFillMode_t           uplo,
                                           int                         n,
                                           int                         k,
                                           const hipblasDoubleComplex* alpha,
                                           const hipblasDoubleComplex* A,
                                           int                         lda,
                                           hipblasStride               strideA,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           const hipblasDoubleComplex* beta,
                                           hipblasDoubleComplex*       y,
                                           int                         incy,
                                           hipblasStride               stridey,
                                           int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasChbmvStridedBatched_v2(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int               n,
                                              int               k,
                                              const hipComplex* alpha,
                                              const hipComplex* A,
                                              int               lda,
                                              hipblasStride     strideA,
                                              const hipComplex* x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              const hipComplex* beta,
                                              hipComplex*       y,
                                              int               incy,
                                              hipblasStride     stridey,
                                              int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhbmvStridedBatched_v2(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              int                     n,
                                              int                     k,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* A,
                                              int                     lda,
                                              hipblasStride           strideA,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* beta,
                                              hipDoubleComplex*       y,
                                              int                     incy,
                                              hipblasStride           stridey,
                                              int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// hbmv_strided_batched_64
hipblasStatus_t hipblasChbmvStridedBatched_64(hipblasHandle_t       handle,
                                              hipblasFillMode_t     uplo,
                                              int64_t               n,
                                              int64_t               k,
                                              const hipblasComplex* alpha,
                                              const hipblasComplex* A,
                                              int64_t               lda,
                                              hipblasStride         strideA,
                                              const hipblasComplex* x,
                                              int64_t               incx,
                                              hipblasStride         stridex,
                                              const hipblasComplex* beta,
                                              hipblasComplex*       y,
                                              int64_t               incy,
                                              hipblasStride         stridey,
                                              int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhbmvStridedBatched_64(hipblasHandle_t             handle,
                                              hipblasFillMode_t           uplo,
                                              int64_t                     n,
                                              int64_t                     k,
                                              const hipblasDoubleComplex* alpha,
                                              const hipblasDoubleComplex* A,
                                              int64_t                     lda,
                                              hipblasStride               strideA,
                                              const hipblasDoubleComplex* x,
                                              int64_t                     incx,
                                              hipblasStride               stridex,
                                              const hipblasDoubleComplex* beta,
                                              hipblasDoubleComplex*       y,
                                              int64_t                     incy,
                                              hipblasStride               stridey,
                                              int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasChbmvStridedBatched_v2_64(hipblasHandle_t   handle,
                                                 hipblasFillMode_t uplo,
                                                 int64_t           n,
                                                 int64_t           k,
                                                 const hipComplex* alpha,
                                                 const hipComplex* A,
                                                 int64_t           lda,
                                                 hipblasStride     strideA,
                                                 const hipComplex* x,
                                                 int64_t           incx,
                                                 hipblasStride     stridex,
                                                 const hipComplex* beta,
                                                 hipComplex*       y,
                                                 int64_t           incy,
                                                 hipblasStride     stridey,
                                                 int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhbmvStridedBatched_v2_64(hipblasHandle_t         handle,
                                                 hipblasFillMode_t       uplo,
                                                 int64_t                 n,
                                                 int64_t                 k,
                                                 const hipDoubleComplex* alpha,
                                                 const hipDoubleComplex* A,
                                                 int64_t                 lda,
                                                 hipblasStride           strideA,
                                                 const hipDoubleComplex* x,
                                                 int64_t                 incx,
                                                 hipblasStride           stridex,
                                                 const hipDoubleComplex* beta,
                                                 hipDoubleComplex*       y,
                                                 int64_t                 incy,
                                                 hipblasStride           stridey,
                                                 int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// hemv
hipblasStatus_t hipblasChemv(hipblasHandle_t       handle,
                             hipblasFillMode_t     uplo,
                             int                   n,
                             const hipblasComplex* alpha,
                             const hipblasComplex* A,
                             int                   lda,
                             const hipblasComplex* x,
                             int                   incx,
                             const hipblasComplex* beta,
                             hipblasComplex*       y,
                             int                   incy)
try
{
    return hipblasConvertStatus(cublasChemv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            (cuComplex*)alpha,
                                            (cuComplex*)A,
                                            lda,
                                            (cuComplex*)x,
                                            incx,
                                            (cuComplex*)beta,
                                            (cuComplex*)y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhemv(hipblasHandle_t             handle,
                             hipblasFillMode_t           uplo,
                             int                         n,
                             const hipblasDoubleComplex* alpha,
                             const hipblasDoubleComplex* A,
                             int                         lda,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             const hipblasDoubleComplex* beta,
                             hipblasDoubleComplex*       y,
                             int                         incy)
try
{
    return hipblasConvertStatus(cublasZhemv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            (cuDoubleComplex*)alpha,
                                            (cuDoubleComplex*)A,
                                            lda,
                                            (cuDoubleComplex*)x,
                                            incx,
                                            (cuDoubleComplex*)beta,
                                            (cuDoubleComplex*)y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasChemv_v2(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int               n,
                                const hipComplex* alpha,
                                const hipComplex* A,
                                int               lda,
                                const hipComplex* x,
                                int               incx,
                                const hipComplex* beta,
                                hipComplex*       y,
                                int               incy)
try
{
    return hipblasConvertStatus(cublasChemv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            (cuComplex*)alpha,
                                            (cuComplex*)A,
                                            lda,
                                            (cuComplex*)x,
                                            incx,
                                            (cuComplex*)beta,
                                            (cuComplex*)y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhemv_v2(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                int                     n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* A,
                                int                     lda,
                                const hipDoubleComplex* x,
                                int                     incx,
                                const hipDoubleComplex* beta,
                                hipDoubleComplex*       y,
                                int                     incy)
try
{
    return hipblasConvertStatus(cublasZhemv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            (cuDoubleComplex*)alpha,
                                            (cuDoubleComplex*)A,
                                            lda,
                                            (cuDoubleComplex*)x,
                                            incx,
                                            (cuDoubleComplex*)beta,
                                            (cuDoubleComplex*)y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hemv_64
hipblasStatus_t hipblasChemv_64(hipblasHandle_t       handle,
                                hipblasFillMode_t     uplo,
                                int64_t               n,
                                const hipblasComplex* alpha,
                                const hipblasComplex* A,
                                int64_t               lda,
                                const hipblasComplex* x,
                                int64_t               incx,
                                const hipblasComplex* beta,
                                hipblasComplex*       y,
                                int64_t               incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasChemv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               (cuComplex*)alpha,
                                               (cuComplex*)A,
                                               lda,
                                               (cuComplex*)x,
                                               incx,
                                               (cuComplex*)beta,
                                               (cuComplex*)y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhemv_64(hipblasHandle_t             handle,
                                hipblasFillMode_t           uplo,
                                int64_t                     n,
                                const hipblasDoubleComplex* alpha,
                                const hipblasDoubleComplex* A,
                                int64_t                     lda,
                                const hipblasDoubleComplex* x,
                                int64_t                     incx,
                                const hipblasDoubleComplex* beta,
                                hipblasDoubleComplex*       y,
                                int64_t                     incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZhemv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               (cuDoubleComplex*)alpha,
                                               (cuDoubleComplex*)A,
                                               lda,
                                               (cuDoubleComplex*)x,
                                               incx,
                                               (cuDoubleComplex*)beta,
                                               (cuDoubleComplex*)y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasChemv_v2_64(hipblasHandle_t   handle,
                                   hipblasFillMode_t uplo,
                                   int64_t           n,
                                   const hipComplex* alpha,
                                   const hipComplex* A,
                                   int64_t           lda,
                                   const hipComplex* x,
                                   int64_t           incx,
                                   const hipComplex* beta,
                                   hipComplex*       y,
                                   int64_t           incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasChemv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               (cuComplex*)alpha,
                                               (cuComplex*)A,
                                               lda,
                                               (cuComplex*)x,
                                               incx,
                                               (cuComplex*)beta,
                                               (cuComplex*)y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhemv_v2_64(hipblasHandle_t         handle,
                                   hipblasFillMode_t       uplo,
                                   int64_t                 n,
                                   const hipDoubleComplex* alpha,
                                   const hipDoubleComplex* A,
                                   int64_t                 lda,
                                   const hipDoubleComplex* x,
                                   int64_t                 incx,
                                   const hipDoubleComplex* beta,
                                   hipDoubleComplex*       y,
                                   int64_t                 incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZhemv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               (cuDoubleComplex*)alpha,
                                               (cuDoubleComplex*)A,
                                               lda,
                                               (cuDoubleComplex*)x,
                                               incx,
                                               (cuDoubleComplex*)beta,
                                               (cuDoubleComplex*)y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hemv_batched
hipblasStatus_t hipblasChemvBatched(hipblasHandle_t             handle,
                                    hipblasFillMode_t           uplo,
                                    int                         n,
                                    const hipblasComplex*       alpha,
                                    const hipblasComplex* const A[],
                                    int                         lda,
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    const hipblasComplex*       beta,
                                    hipblasComplex* const       y[],
                                    int                         incy,
                                    int                         batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhemvBatched(hipblasHandle_t                   handle,
                                    hipblasFillMode_t                 uplo,
                                    int                               n,
                                    const hipblasDoubleComplex*       alpha,
                                    const hipblasDoubleComplex* const A[],
                                    int                               lda,
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    const hipblasDoubleComplex*       beta,
                                    hipblasDoubleComplex* const       y[],
                                    int                               incy,
                                    int                               batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasChemvBatched_v2(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       int                     n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const A[],
                                       int                     lda,
                                       const hipComplex* const x[],
                                       int                     incx,
                                       const hipComplex*       beta,
                                       hipComplex* const       y[],
                                       int                     incy,
                                       int                     batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhemvBatched_v2(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       int                           n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const A[],
                                       int                           lda,
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       const hipDoubleComplex*       beta,
                                       hipDoubleComplex* const       y[],
                                       int                           incy,
                                       int                           batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// hemv_batched_64
hipblasStatus_t hipblasChemvBatched_64(hipblasHandle_t             handle,
                                       hipblasFillMode_t           uplo,
                                       int64_t                     n,
                                       const hipblasComplex*       alpha,
                                       const hipblasComplex* const A[],
                                       int64_t                     lda,
                                       const hipblasComplex* const x[],
                                       int64_t                     incx,
                                       const hipblasComplex*       beta,
                                       hipblasComplex* const       y[],
                                       int64_t                     incy,
                                       int64_t                     batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhemvBatched_64(hipblasHandle_t                   handle,
                                       hipblasFillMode_t                 uplo,
                                       int64_t                           n,
                                       const hipblasDoubleComplex*       alpha,
                                       const hipblasDoubleComplex* const A[],
                                       int64_t                           lda,
                                       const hipblasDoubleComplex* const x[],
                                       int64_t                           incx,
                                       const hipblasDoubleComplex*       beta,
                                       hipblasDoubleComplex* const       y[],
                                       int64_t                           incy,
                                       int64_t                           batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasChemvBatched_v2_64(hipblasHandle_t         handle,
                                          hipblasFillMode_t       uplo,
                                          int64_t                 n,
                                          const hipComplex*       alpha,
                                          const hipComplex* const A[],
                                          int64_t                 lda,
                                          const hipComplex* const x[],
                                          int64_t                 incx,
                                          const hipComplex*       beta,
                                          hipComplex* const       y[],
                                          int64_t                 incy,
                                          int64_t                 batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhemvBatched_v2_64(hipblasHandle_t               handle,
                                          hipblasFillMode_t             uplo,
                                          int64_t                       n,
                                          const hipDoubleComplex*       alpha,
                                          const hipDoubleComplex* const A[],
                                          int64_t                       lda,
                                          const hipDoubleComplex* const x[],
                                          int64_t                       incx,
                                          const hipDoubleComplex*       beta,
                                          hipDoubleComplex* const       y[],
                                          int64_t                       incy,
                                          int64_t                       batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// hemv_strided_batched
hipblasStatus_t hipblasChemvStridedBatched(hipblasHandle_t       handle,
                                           hipblasFillMode_t     uplo,
                                           int                   n,
                                           const hipblasComplex* alpha,
                                           const hipblasComplex* A,
                                           int                   lda,
                                           hipblasStride         stride_a,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stride_x,
                                           const hipblasComplex* beta,
                                           hipblasComplex*       y,
                                           int                   incy,
                                           hipblasStride         stride_y,
                                           int                   batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhemvStridedBatched(hipblasHandle_t             handle,
                                           hipblasFillMode_t           uplo,
                                           int                         n,
                                           const hipblasDoubleComplex* alpha,
                                           const hipblasDoubleComplex* A,
                                           int                         lda,
                                           hipblasStride               stride_a,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stride_x,
                                           const hipblasDoubleComplex* beta,
                                           hipblasDoubleComplex*       y,
                                           int                         incy,
                                           hipblasStride               stride_y,
                                           int                         batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasChemvStridedBatched_v2(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int               n,
                                              const hipComplex* alpha,
                                              const hipComplex* A,
                                              int               lda,
                                              hipblasStride     stride_a,
                                              const hipComplex* x,
                                              int               incx,
                                              hipblasStride     stride_x,
                                              const hipComplex* beta,
                                              hipComplex*       y,
                                              int               incy,
                                              hipblasStride     stride_y,
                                              int               batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhemvStridedBatched_v2(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              int                     n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* A,
                                              int                     lda,
                                              hipblasStride           stride_a,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stride_x,
                                              const hipDoubleComplex* beta,
                                              hipDoubleComplex*       y,
                                              int                     incy,
                                              hipblasStride           stride_y,
                                              int                     batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// hemv_strided_batched_64
hipblasStatus_t hipblasChemvStridedBatched_64(hipblasHandle_t       handle,
                                              hipblasFillMode_t     uplo,
                                              int64_t               n,
                                              const hipblasComplex* alpha,
                                              const hipblasComplex* A,
                                              int64_t               lda,
                                              hipblasStride         stride_a,
                                              const hipblasComplex* x,
                                              int64_t               incx,
                                              hipblasStride         stride_x,
                                              const hipblasComplex* beta,
                                              hipblasComplex*       y,
                                              int64_t               incy,
                                              hipblasStride         stride_y,
                                              int64_t               batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhemvStridedBatched_64(hipblasHandle_t             handle,
                                              hipblasFillMode_t           uplo,
                                              int64_t                     n,
                                              const hipblasDoubleComplex* alpha,
                                              const hipblasDoubleComplex* A,
                                              int64_t                     lda,
                                              hipblasStride               stride_a,
                                              const hipblasDoubleComplex* x,
                                              int64_t                     incx,
                                              hipblasStride               stride_x,
                                              const hipblasDoubleComplex* beta,
                                              hipblasDoubleComplex*       y,
                                              int64_t                     incy,
                                              hipblasStride               stride_y,
                                              int64_t                     batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasChemvStridedBatched_v2_64(hipblasHandle_t   handle,
                                                 hipblasFillMode_t uplo,
                                                 int64_t           n,
                                                 const hipComplex* alpha,
                                                 const hipComplex* A,
                                                 int64_t           lda,
                                                 hipblasStride     stride_a,
                                                 const hipComplex* x,
                                                 int64_t           incx,
                                                 hipblasStride     stride_x,
                                                 const hipComplex* beta,
                                                 hipComplex*       y,
                                                 int64_t           incy,
                                                 hipblasStride     stride_y,
                                                 int64_t           batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhemvStridedBatched_v2_64(hipblasHandle_t         handle,
                                                 hipblasFillMode_t       uplo,
                                                 int64_t                 n,
                                                 const hipDoubleComplex* alpha,
                                                 const hipDoubleComplex* A,
                                                 int64_t                 lda,
                                                 hipblasStride           stride_a,
                                                 const hipDoubleComplex* x,
                                                 int64_t                 incx,
                                                 hipblasStride           stride_x,
                                                 const hipDoubleComplex* beta,
                                                 hipDoubleComplex*       y,
                                                 int64_t                 incy,
                                                 hipblasStride           stride_y,
                                                 int64_t                 batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// her
hipblasStatus_t hipblasCher(hipblasHandle_t       handle,
                            hipblasFillMode_t     uplo,
                            int                   n,
                            const float*          alpha,
                            const hipblasComplex* x,
                            int                   incx,
                            hipblasComplex*       A,
                            int                   lda)
try
{
    return hipblasConvertStatus(cublasCher((cublasHandle_t)handle,
                                           hipblasConvertFill(uplo),
                                           n,
                                           alpha,
                                           (cuComplex*)x,
                                           incx,
                                           (cuComplex*)A,
                                           lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZher(hipblasHandle_t             handle,
                            hipblasFillMode_t           uplo,
                            int                         n,
                            const double*               alpha,
                            const hipblasDoubleComplex* x,
                            int                         incx,
                            hipblasDoubleComplex*       A,
                            int                         lda)
try
{
    return hipblasConvertStatus(cublasZher((cublasHandle_t)handle,
                                           hipblasConvertFill(uplo),
                                           n,
                                           alpha,
                                           (cuDoubleComplex*)x,
                                           incx,
                                           (cuDoubleComplex*)A,
                                           lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCher_v2(hipblasHandle_t   handle,
                               hipblasFillMode_t uplo,
                               int               n,
                               const float*      alpha,
                               const hipComplex* x,
                               int               incx,
                               hipComplex*       A,
                               int               lda)
try
{
    return hipblasConvertStatus(cublasCher((cublasHandle_t)handle,
                                           hipblasConvertFill(uplo),
                                           n,
                                           alpha,
                                           (cuComplex*)x,
                                           incx,
                                           (cuComplex*)A,
                                           lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZher_v2(hipblasHandle_t         handle,
                               hipblasFillMode_t       uplo,
                               int                     n,
                               const double*           alpha,
                               const hipDoubleComplex* x,
                               int                     incx,
                               hipDoubleComplex*       A,
                               int                     lda)
try
{
    return hipblasConvertStatus(cublasZher((cublasHandle_t)handle,
                                           hipblasConvertFill(uplo),
                                           n,
                                           alpha,
                                           (cuDoubleComplex*)x,
                                           incx,
                                           (cuDoubleComplex*)A,
                                           lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// her_64
hipblasStatus_t hipblasCher_64(hipblasHandle_t       handle,
                               hipblasFillMode_t     uplo,
                               int64_t               n,
                               const float*          alpha,
                               const hipblasComplex* x,
                               int64_t               incx,
                               hipblasComplex*       A,
                               int64_t               lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCher_64((cublasHandle_t)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              alpha,
                                              (cuComplex*)x,
                                              incx,
                                              (cuComplex*)A,
                                              lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZher_64(hipblasHandle_t             handle,
                               hipblasFillMode_t           uplo,
                               int64_t                     n,
                               const double*               alpha,
                               const hipblasDoubleComplex* x,
                               int64_t                     incx,
                               hipblasDoubleComplex*       A,
                               int64_t                     lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZher_64((cublasHandle_t)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              alpha,
                                              (cuDoubleComplex*)x,
                                              incx,
                                              (cuDoubleComplex*)A,
                                              lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCher_v2_64(hipblasHandle_t   handle,
                                  hipblasFillMode_t uplo,
                                  int64_t           n,
                                  const float*      alpha,
                                  const hipComplex* x,
                                  int64_t           incx,
                                  hipComplex*       A,
                                  int64_t           lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCher_64((cublasHandle_t)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              alpha,
                                              (cuComplex*)x,
                                              incx,
                                              (cuComplex*)A,
                                              lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZher_v2_64(hipblasHandle_t         handle,
                                  hipblasFillMode_t       uplo,
                                  int64_t                 n,
                                  const double*           alpha,
                                  const hipDoubleComplex* x,
                                  int64_t                 incx,
                                  hipDoubleComplex*       A,
                                  int64_t                 lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZher_64((cublasHandle_t)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              alpha,
                                              (cuDoubleComplex*)x,
                                              incx,
                                              (cuDoubleComplex*)A,
                                              lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// her_batched
hipblasStatus_t hipblasCherBatched(hipblasHandle_t             handle,
                                   hipblasFillMode_t           uplo,
                                   int                         n,
                                   const float*                alpha,
                                   const hipblasComplex* const x[],
                                   int                         incx,
                                   hipblasComplex* const       A[],
                                   int                         lda,
                                   int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZherBatched(hipblasHandle_t                   handle,
                                   hipblasFillMode_t                 uplo,
                                   int                               n,
                                   const double*                     alpha,
                                   const hipblasDoubleComplex* const x[],
                                   int                               incx,
                                   hipblasDoubleComplex* const       A[],
                                   int                               lda,
                                   int                               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCherBatched_v2(hipblasHandle_t         handle,
                                      hipblasFillMode_t       uplo,
                                      int                     n,
                                      const float*            alpha,
                                      const hipComplex* const x[],
                                      int                     incx,
                                      hipComplex* const       A[],
                                      int                     lda,
                                      int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZherBatched_v2(hipblasHandle_t               handle,
                                      hipblasFillMode_t             uplo,
                                      int                           n,
                                      const double*                 alpha,
                                      const hipDoubleComplex* const x[],
                                      int                           incx,
                                      hipDoubleComplex* const       A[],
                                      int                           lda,
                                      int                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// her_batched_64
hipblasStatus_t hipblasCherBatched_64(hipblasHandle_t             handle,
                                      hipblasFillMode_t           uplo,
                                      int64_t                     n,
                                      const float*                alpha,
                                      const hipblasComplex* const x[],
                                      int64_t                     incx,
                                      hipblasComplex* const       A[],
                                      int64_t                     lda,
                                      int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZherBatched_64(hipblasHandle_t                   handle,
                                      hipblasFillMode_t                 uplo,
                                      int64_t                           n,
                                      const double*                     alpha,
                                      const hipblasDoubleComplex* const x[],
                                      int64_t                           incx,
                                      hipblasDoubleComplex* const       A[],
                                      int64_t                           lda,
                                      int64_t                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCherBatched_v2_64(hipblasHandle_t         handle,
                                         hipblasFillMode_t       uplo,
                                         int64_t                 n,
                                         const float*            alpha,
                                         const hipComplex* const x[],
                                         int64_t                 incx,
                                         hipComplex* const       A[],
                                         int64_t                 lda,
                                         int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZherBatched_v2_64(hipblasHandle_t               handle,
                                         hipblasFillMode_t             uplo,
                                         int64_t                       n,
                                         const double*                 alpha,
                                         const hipDoubleComplex* const x[],
                                         int64_t                       incx,
                                         hipDoubleComplex* const       A[],
                                         int64_t                       lda,
                                         int64_t                       batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// her_strided_batched
hipblasStatus_t hipblasCherStridedBatched(hipblasHandle_t       handle,
                                          hipblasFillMode_t     uplo,
                                          int                   n,
                                          const float*          alpha,
                                          const hipblasComplex* x,
                                          int                   incx,
                                          hipblasStride         stridex,
                                          hipblasComplex*       A,
                                          int                   lda,
                                          hipblasStride         strideA,
                                          int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZherStridedBatched(hipblasHandle_t             handle,
                                          hipblasFillMode_t           uplo,
                                          int                         n,
                                          const double*               alpha,
                                          const hipblasDoubleComplex* x,
                                          int                         incx,
                                          hipblasStride               stridex,
                                          hipblasDoubleComplex*       A,
                                          int                         lda,
                                          hipblasStride               strideA,
                                          int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCherStridedBatched_v2(hipblasHandle_t   handle,
                                             hipblasFillMode_t uplo,
                                             int               n,
                                             const float*      alpha,
                                             const hipComplex* x,
                                             int               incx,
                                             hipblasStride     stridex,
                                             hipComplex*       A,
                                             int               lda,
                                             hipblasStride     strideA,
                                             int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZherStridedBatched_v2(hipblasHandle_t         handle,
                                             hipblasFillMode_t       uplo,
                                             int                     n,
                                             const double*           alpha,
                                             const hipDoubleComplex* x,
                                             int                     incx,
                                             hipblasStride           stridex,
                                             hipDoubleComplex*       A,
                                             int                     lda,
                                             hipblasStride           strideA,
                                             int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// her_strided_batched_64
hipblasStatus_t hipblasCherStridedBatched_64(hipblasHandle_t       handle,
                                             hipblasFillMode_t     uplo,
                                             int64_t               n,
                                             const float*          alpha,
                                             const hipblasComplex* x,
                                             int64_t               incx,
                                             hipblasStride         stridex,
                                             hipblasComplex*       A,
                                             int64_t               lda,
                                             hipblasStride         strideA,
                                             int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZherStridedBatched_64(hipblasHandle_t             handle,
                                             hipblasFillMode_t           uplo,
                                             int64_t                     n,
                                             const double*               alpha,
                                             const hipblasDoubleComplex* x,
                                             int64_t                     incx,
                                             hipblasStride               stridex,
                                             hipblasDoubleComplex*       A,
                                             int64_t                     lda,
                                             hipblasStride               strideA,
                                             int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCherStridedBatched_v2_64(hipblasHandle_t   handle,
                                                hipblasFillMode_t uplo,
                                                int64_t           n,
                                                const float*      alpha,
                                                const hipComplex* x,
                                                int64_t           incx,
                                                hipblasStride     stridex,
                                                hipComplex*       A,
                                                int64_t           lda,
                                                hipblasStride     strideA,
                                                int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZherStridedBatched_v2_64(hipblasHandle_t         handle,
                                                hipblasFillMode_t       uplo,
                                                int64_t                 n,
                                                const double*           alpha,
                                                const hipDoubleComplex* x,
                                                int64_t                 incx,
                                                hipblasStride           stridex,
                                                hipDoubleComplex*       A,
                                                int64_t                 lda,
                                                hipblasStride           strideA,
                                                int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// her2
hipblasStatus_t hipblasCher2(hipblasHandle_t       handle,
                             hipblasFillMode_t     uplo,
                             int                   n,
                             const hipblasComplex* alpha,
                             const hipblasComplex* x,
                             int                   incx,
                             const hipblasComplex* y,
                             int                   incy,
                             hipblasComplex*       A,
                             int                   lda)
try
{
    return hipblasConvertStatus(cublasCher2((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            (cuComplex*)alpha,
                                            (cuComplex*)x,
                                            incx,
                                            (cuComplex*)y,
                                            incy,
                                            (cuComplex*)A,
                                            lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZher2(hipblasHandle_t             handle,
                             hipblasFillMode_t           uplo,
                             int                         n,
                             const hipblasDoubleComplex* alpha,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             const hipblasDoubleComplex* y,
                             int                         incy,
                             hipblasDoubleComplex*       A,
                             int                         lda)
try
{
    return hipblasConvertStatus(cublasZher2((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            (cuDoubleComplex*)alpha,
                                            (cuDoubleComplex*)x,
                                            incx,
                                            (cuDoubleComplex*)y,
                                            incy,
                                            (cuDoubleComplex*)A,
                                            lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCher2_v2(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int               n,
                                const hipComplex* alpha,
                                const hipComplex* x,
                                int               incx,
                                const hipComplex* y,
                                int               incy,
                                hipComplex*       A,
                                int               lda)
try
{
    return hipblasConvertStatus(cublasCher2((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            (cuComplex*)alpha,
                                            (cuComplex*)x,
                                            incx,
                                            (cuComplex*)y,
                                            incy,
                                            (cuComplex*)A,
                                            lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZher2_v2(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                int                     n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* x,
                                int                     incx,
                                const hipDoubleComplex* y,
                                int                     incy,
                                hipDoubleComplex*       A,
                                int                     lda)
try
{
    return hipblasConvertStatus(cublasZher2((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            (cuDoubleComplex*)alpha,
                                            (cuDoubleComplex*)x,
                                            incx,
                                            (cuDoubleComplex*)y,
                                            incy,
                                            (cuDoubleComplex*)A,
                                            lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// her2_64
hipblasStatus_t hipblasCher2_64(hipblasHandle_t       handle,
                                hipblasFillMode_t     uplo,
                                int64_t               n,
                                const hipblasComplex* alpha,
                                const hipblasComplex* x,
                                int64_t               incx,
                                const hipblasComplex* y,
                                int64_t               incy,
                                hipblasComplex*       A,
                                int64_t               lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCher2_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               (cuComplex*)alpha,
                                               (cuComplex*)x,
                                               incx,
                                               (cuComplex*)y,
                                               incy,
                                               (cuComplex*)A,
                                               lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZher2_64(hipblasHandle_t             handle,
                                hipblasFillMode_t           uplo,
                                int64_t                     n,
                                const hipblasDoubleComplex* alpha,
                                const hipblasDoubleComplex* x,
                                int64_t                     incx,
                                const hipblasDoubleComplex* y,
                                int64_t                     incy,
                                hipblasDoubleComplex*       A,
                                int64_t                     lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZher2_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               (cuDoubleComplex*)alpha,
                                               (cuDoubleComplex*)x,
                                               incx,
                                               (cuDoubleComplex*)y,
                                               incy,
                                               (cuDoubleComplex*)A,
                                               lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCher2_v2_64(hipblasHandle_t   handle,
                                   hipblasFillMode_t uplo,
                                   int64_t           n,
                                   const hipComplex* alpha,
                                   const hipComplex* x,
                                   int64_t           incx,
                                   const hipComplex* y,
                                   int64_t           incy,
                                   hipComplex*       A,
                                   int64_t           lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCher2_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               (cuComplex*)alpha,
                                               (cuComplex*)x,
                                               incx,
                                               (cuComplex*)y,
                                               incy,
                                               (cuComplex*)A,
                                               lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZher2_v2_64(hipblasHandle_t         handle,
                                   hipblasFillMode_t       uplo,
                                   int64_t                 n,
                                   const hipDoubleComplex* alpha,
                                   const hipDoubleComplex* x,
                                   int64_t                 incx,
                                   const hipDoubleComplex* y,
                                   int64_t                 incy,
                                   hipDoubleComplex*       A,
                                   int64_t                 lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZher2_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               (cuDoubleComplex*)alpha,
                                               (cuDoubleComplex*)x,
                                               incx,
                                               (cuDoubleComplex*)y,
                                               incy,
                                               (cuDoubleComplex*)A,
                                               lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// her2_batched
hipblasStatus_t hipblasCher2Batched(hipblasHandle_t             handle,
                                    hipblasFillMode_t           uplo,
                                    int                         n,
                                    const hipblasComplex*       alpha,
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    const hipblasComplex* const y[],
                                    int                         incy,
                                    hipblasComplex* const       A[],
                                    int                         lda,
                                    int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZher2Batched(hipblasHandle_t                   handle,
                                    hipblasFillMode_t                 uplo,
                                    int                               n,
                                    const hipblasDoubleComplex*       alpha,
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    const hipblasDoubleComplex* const y[],
                                    int                               incy,
                                    hipblasDoubleComplex* const       A[],
                                    int                               lda,
                                    int                               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCher2Batched_v2(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       int                     n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const x[],
                                       int                     incx,
                                       const hipComplex* const y[],
                                       int                     incy,
                                       hipComplex* const       A[],
                                       int                     lda,
                                       int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZher2Batched_v2(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       int                           n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       const hipDoubleComplex* const y[],
                                       int                           incy,
                                       hipDoubleComplex* const       A[],
                                       int                           lda,
                                       int                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// her2_batched_64
hipblasStatus_t hipblasCher2Batched_64(hipblasHandle_t             handle,
                                       hipblasFillMode_t           uplo,
                                       int64_t                     n,
                                       const hipblasComplex*       alpha,
                                       const hipblasComplex* const x[],
                                       int64_t                     incx,
                                       const hipblasComplex* const y[],
                                       int64_t                     incy,
                                       hipblasComplex* const       A[],
                                       int64_t                     lda,
                                       int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZher2Batched_64(hipblasHandle_t                   handle,
                                       hipblasFillMode_t                 uplo,
                                       int64_t                           n,
                                       const hipblasDoubleComplex*       alpha,
                                       const hipblasDoubleComplex* const x[],
                                       int64_t                           incx,
                                       const hipblasDoubleComplex* const y[],
                                       int64_t                           incy,
                                       hipblasDoubleComplex* const       A[],
                                       int64_t                           lda,
                                       int64_t                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCher2Batched_v2_64(hipblasHandle_t         handle,
                                          hipblasFillMode_t       uplo,
                                          int64_t                 n,
                                          const hipComplex*       alpha,
                                          const hipComplex* const x[],
                                          int64_t                 incx,
                                          const hipComplex* const y[],
                                          int64_t                 incy,
                                          hipComplex* const       A[],
                                          int64_t                 lda,
                                          int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZher2Batched_v2_64(hipblasHandle_t               handle,
                                          hipblasFillMode_t             uplo,
                                          int64_t                       n,
                                          const hipDoubleComplex*       alpha,
                                          const hipDoubleComplex* const x[],
                                          int64_t                       incx,
                                          const hipDoubleComplex* const y[],
                                          int64_t                       incy,
                                          hipDoubleComplex* const       A[],
                                          int64_t                       lda,
                                          int64_t                       batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// her2_strided_batched
hipblasStatus_t hipblasCher2StridedBatched(hipblasHandle_t       handle,
                                           hipblasFillMode_t     uplo,
                                           int                   n,
                                           const hipblasComplex* alpha,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           const hipblasComplex* y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           hipblasComplex*       A,
                                           int                   lda,
                                           hipblasStride         strideA,
                                           int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZher2StridedBatched(hipblasHandle_t             handle,
                                           hipblasFillMode_t           uplo,
                                           int                         n,
                                           const hipblasDoubleComplex* alpha,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           const hipblasDoubleComplex* y,
                                           int                         incy,
                                           hipblasStride               stridey,
                                           hipblasDoubleComplex*       A,
                                           int                         lda,
                                           hipblasStride               strideA,
                                           int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCher2StridedBatched_v2(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int               n,
                                              const hipComplex* alpha,
                                              const hipComplex* x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              const hipComplex* y,
                                              int               incy,
                                              hipblasStride     stridey,
                                              hipComplex*       A,
                                              int               lda,
                                              hipblasStride     strideA,
                                              int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZher2StridedBatched_v2(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              int                     n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* y,
                                              int                     incy,
                                              hipblasStride           stridey,
                                              hipDoubleComplex*       A,
                                              int                     lda,
                                              hipblasStride           strideA,
                                              int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// her2_strided_batched_64
hipblasStatus_t hipblasCher2StridedBatched_64(hipblasHandle_t       handle,
                                              hipblasFillMode_t     uplo,
                                              int64_t               n,
                                              const hipblasComplex* alpha,
                                              const hipblasComplex* x,
                                              int64_t               incx,
                                              hipblasStride         stridex,
                                              const hipblasComplex* y,
                                              int64_t               incy,
                                              hipblasStride         stridey,
                                              hipblasComplex*       A,
                                              int64_t               lda,
                                              hipblasStride         strideA,
                                              int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZher2StridedBatched_64(hipblasHandle_t             handle,
                                              hipblasFillMode_t           uplo,
                                              int64_t                     n,
                                              const hipblasDoubleComplex* alpha,
                                              const hipblasDoubleComplex* x,
                                              int64_t                     incx,
                                              hipblasStride               stridex,
                                              const hipblasDoubleComplex* y,
                                              int64_t                     incy,
                                              hipblasStride               stridey,
                                              hipblasDoubleComplex*       A,
                                              int64_t                     lda,
                                              hipblasStride               strideA,
                                              int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCher2StridedBatched_v2_64(hipblasHandle_t   handle,
                                                 hipblasFillMode_t uplo,
                                                 int64_t           n,
                                                 const hipComplex* alpha,
                                                 const hipComplex* x,
                                                 int64_t           incx,
                                                 hipblasStride     stridex,
                                                 const hipComplex* y,
                                                 int64_t           incy,
                                                 hipblasStride     stridey,
                                                 hipComplex*       A,
                                                 int64_t           lda,
                                                 hipblasStride     strideA,
                                                 int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZher2StridedBatched_v2_64(hipblasHandle_t         handle,
                                                 hipblasFillMode_t       uplo,
                                                 int64_t                 n,
                                                 const hipDoubleComplex* alpha,
                                                 const hipDoubleComplex* x,
                                                 int64_t                 incx,
                                                 hipblasStride           stridex,
                                                 const hipDoubleComplex* y,
                                                 int64_t                 incy,
                                                 hipblasStride           stridey,
                                                 hipDoubleComplex*       A,
                                                 int64_t                 lda,
                                                 hipblasStride           strideA,
                                                 int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// hpmv
hipblasStatus_t hipblasChpmv(hipblasHandle_t       handle,
                             hipblasFillMode_t     uplo,
                             int                   n,
                             const hipblasComplex* alpha,
                             const hipblasComplex* AP,
                             const hipblasComplex* x,
                             int                   incx,
                             const hipblasComplex* beta,
                             hipblasComplex*       y,
                             int                   incy)
try
{
    return hipblasConvertStatus(cublasChpmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            (cuComplex*)alpha,
                                            (cuComplex*)AP,
                                            (cuComplex*)x,
                                            incx,
                                            (cuComplex*)beta,
                                            (cuComplex*)y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhpmv(hipblasHandle_t             handle,
                             hipblasFillMode_t           uplo,
                             int                         n,
                             const hipblasDoubleComplex* alpha,
                             const hipblasDoubleComplex* AP,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             const hipblasDoubleComplex* beta,
                             hipblasDoubleComplex*       y,
                             int                         incy)
try
{
    return hipblasConvertStatus(cublasZhpmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            (cuDoubleComplex*)alpha,
                                            (cuDoubleComplex*)AP,
                                            (cuDoubleComplex*)x,
                                            incx,
                                            (cuDoubleComplex*)beta,
                                            (cuDoubleComplex*)y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasChpmv_v2(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int               n,
                                const hipComplex* alpha,
                                const hipComplex* AP,
                                const hipComplex* x,
                                int               incx,
                                const hipComplex* beta,
                                hipComplex*       y,
                                int               incy)
try
{
    return hipblasConvertStatus(cublasChpmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            (cuComplex*)alpha,
                                            (cuComplex*)AP,
                                            (cuComplex*)x,
                                            incx,
                                            (cuComplex*)beta,
                                            (cuComplex*)y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhpmv_v2(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                int                     n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* AP,
                                const hipDoubleComplex* x,
                                int                     incx,
                                const hipDoubleComplex* beta,
                                hipDoubleComplex*       y,
                                int                     incy)
try
{
    return hipblasConvertStatus(cublasZhpmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            (cuDoubleComplex*)alpha,
                                            (cuDoubleComplex*)AP,
                                            (cuDoubleComplex*)x,
                                            incx,
                                            (cuDoubleComplex*)beta,
                                            (cuDoubleComplex*)y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hpmv_64
hipblasStatus_t hipblasChpmv_64(hipblasHandle_t       handle,
                                hipblasFillMode_t     uplo,
                                int64_t               n,
                                const hipblasComplex* alpha,
                                const hipblasComplex* AP,
                                const hipblasComplex* x,
                                int64_t               incx,
                                const hipblasComplex* beta,
                                hipblasComplex*       y,
                                int64_t               incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasChpmv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               (cuComplex*)alpha,
                                               (cuComplex*)AP,
                                               (cuComplex*)x,
                                               incx,
                                               (cuComplex*)beta,
                                               (cuComplex*)y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhpmv_64(hipblasHandle_t             handle,
                                hipblasFillMode_t           uplo,
                                int64_t                     n,
                                const hipblasDoubleComplex* alpha,
                                const hipblasDoubleComplex* AP,
                                const hipblasDoubleComplex* x,
                                int64_t                     incx,
                                const hipblasDoubleComplex* beta,
                                hipblasDoubleComplex*       y,
                                int64_t                     incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZhpmv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               (cuDoubleComplex*)alpha,
                                               (cuDoubleComplex*)AP,
                                               (cuDoubleComplex*)x,
                                               incx,
                                               (cuDoubleComplex*)beta,
                                               (cuDoubleComplex*)y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasChpmv_v2_64(hipblasHandle_t   handle,
                                   hipblasFillMode_t uplo,
                                   int64_t           n,
                                   const hipComplex* alpha,
                                   const hipComplex* AP,
                                   const hipComplex* x,
                                   int64_t           incx,
                                   const hipComplex* beta,
                                   hipComplex*       y,
                                   int64_t           incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasChpmv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               (cuComplex*)alpha,
                                               (cuComplex*)AP,
                                               (cuComplex*)x,
                                               incx,
                                               (cuComplex*)beta,
                                               (cuComplex*)y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhpmv_v2_64(hipblasHandle_t         handle,
                                   hipblasFillMode_t       uplo,
                                   int64_t                 n,
                                   const hipDoubleComplex* alpha,
                                   const hipDoubleComplex* AP,
                                   const hipDoubleComplex* x,
                                   int64_t                 incx,
                                   const hipDoubleComplex* beta,
                                   hipDoubleComplex*       y,
                                   int64_t                 incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZhpmv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               (cuDoubleComplex*)alpha,
                                               (cuDoubleComplex*)AP,
                                               (cuDoubleComplex*)x,
                                               incx,
                                               (cuDoubleComplex*)beta,
                                               (cuDoubleComplex*)y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hpmv_batched
hipblasStatus_t hipblasChpmvBatched(hipblasHandle_t             handle,
                                    hipblasFillMode_t           uplo,
                                    int                         n,
                                    const hipblasComplex*       alpha,
                                    const hipblasComplex* const AP[],
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    const hipblasComplex*       beta,
                                    hipblasComplex* const       y[],
                                    int                         incy,
                                    int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhpmvBatched(hipblasHandle_t                   handle,
                                    hipblasFillMode_t                 uplo,
                                    int                               n,
                                    const hipblasDoubleComplex*       alpha,
                                    const hipblasDoubleComplex* const AP[],
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    const hipblasDoubleComplex*       beta,
                                    hipblasDoubleComplex* const       y[],
                                    int                               incy,
                                    int                               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasChpmvBatched_v2(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       int                     n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const AP[],
                                       const hipComplex* const x[],
                                       int                     incx,
                                       const hipComplex*       beta,
                                       hipComplex* const       y[],
                                       int                     incy,
                                       int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhpmvBatched_v2(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       int                           n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const AP[],
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       const hipDoubleComplex*       beta,
                                       hipDoubleComplex* const       y[],
                                       int                           incy,
                                       int                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// hpmv_batched_64
hipblasStatus_t hipblasChpmvBatched_64(hipblasHandle_t             handle,
                                       hipblasFillMode_t           uplo,
                                       int64_t                     n,
                                       const hipblasComplex*       alpha,
                                       const hipblasComplex* const AP[],
                                       const hipblasComplex* const x[],
                                       int64_t                     incx,
                                       const hipblasComplex*       beta,
                                       hipblasComplex* const       y[],
                                       int64_t                     incy,
                                       int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhpmvBatched_64(hipblasHandle_t                   handle,
                                       hipblasFillMode_t                 uplo,
                                       int64_t                           n,
                                       const hipblasDoubleComplex*       alpha,
                                       const hipblasDoubleComplex* const AP[],
                                       const hipblasDoubleComplex* const x[],
                                       int64_t                           incx,
                                       const hipblasDoubleComplex*       beta,
                                       hipblasDoubleComplex* const       y[],
                                       int64_t                           incy,
                                       int64_t                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasChpmvBatched_v2_64(hipblasHandle_t         handle,
                                          hipblasFillMode_t       uplo,
                                          int64_t                 n,
                                          const hipComplex*       alpha,
                                          const hipComplex* const AP[],
                                          const hipComplex* const x[],
                                          int64_t                 incx,
                                          const hipComplex*       beta,
                                          hipComplex* const       y[],
                                          int64_t                 incy,
                                          int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhpmvBatched_v2_64(hipblasHandle_t               handle,
                                          hipblasFillMode_t             uplo,
                                          int64_t                       n,
                                          const hipDoubleComplex*       alpha,
                                          const hipDoubleComplex* const AP[],
                                          const hipDoubleComplex* const x[],
                                          int64_t                       incx,
                                          const hipDoubleComplex*       beta,
                                          hipDoubleComplex* const       y[],
                                          int64_t                       incy,
                                          int64_t                       batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// hpmv_strided_batched
hipblasStatus_t hipblasChpmvStridedBatched(hipblasHandle_t       handle,
                                           hipblasFillMode_t     uplo,
                                           int                   n,
                                           const hipblasComplex* alpha,
                                           const hipblasComplex* AP,
                                           hipblasStride         strideAP,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           const hipblasComplex* beta,
                                           hipblasComplex*       y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhpmvStridedBatched(hipblasHandle_t             handle,
                                           hipblasFillMode_t           uplo,
                                           int                         n,
                                           const hipblasDoubleComplex* alpha,
                                           const hipblasDoubleComplex* AP,
                                           hipblasStride               strideAP,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           const hipblasDoubleComplex* beta,
                                           hipblasDoubleComplex*       y,
                                           int                         incy,
                                           hipblasStride               stridey,
                                           int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasChpmvStridedBatched_v2(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int               n,
                                              const hipComplex* alpha,
                                              const hipComplex* AP,
                                              hipblasStride     strideAP,
                                              const hipComplex* x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              const hipComplex* beta,
                                              hipComplex*       y,
                                              int               incy,
                                              hipblasStride     stridey,
                                              int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhpmvStridedBatched_v2(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              int                     n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* AP,
                                              hipblasStride           strideAP,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* beta,
                                              hipDoubleComplex*       y,
                                              int                     incy,
                                              hipblasStride           stridey,
                                              int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// hpmv_strided_batched_64
hipblasStatus_t hipblasChpmvStridedBatched_64(hipblasHandle_t       handle,
                                              hipblasFillMode_t     uplo,
                                              int64_t               n,
                                              const hipblasComplex* alpha,
                                              const hipblasComplex* AP,
                                              hipblasStride         strideAP,
                                              const hipblasComplex* x,
                                              int64_t               incx,
                                              hipblasStride         stridex,
                                              const hipblasComplex* beta,
                                              hipblasComplex*       y,
                                              int64_t               incy,
                                              hipblasStride         stridey,
                                              int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhpmvStridedBatched_64(hipblasHandle_t             handle,
                                              hipblasFillMode_t           uplo,
                                              int64_t                     n,
                                              const hipblasDoubleComplex* alpha,
                                              const hipblasDoubleComplex* AP,
                                              hipblasStride               strideAP,
                                              const hipblasDoubleComplex* x,
                                              int64_t                     incx,
                                              hipblasStride               stridex,
                                              const hipblasDoubleComplex* beta,
                                              hipblasDoubleComplex*       y,
                                              int64_t                     incy,
                                              hipblasStride               stridey,
                                              int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasChpmvStridedBatched_v2_64(hipblasHandle_t   handle,
                                                 hipblasFillMode_t uplo,
                                                 int64_t           n,
                                                 const hipComplex* alpha,
                                                 const hipComplex* AP,
                                                 hipblasStride     strideAP,
                                                 const hipComplex* x,
                                                 int64_t           incx,
                                                 hipblasStride     stridex,
                                                 const hipComplex* beta,
                                                 hipComplex*       y,
                                                 int64_t           incy,
                                                 hipblasStride     stridey,
                                                 int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhpmvStridedBatched_v2_64(hipblasHandle_t         handle,
                                                 hipblasFillMode_t       uplo,
                                                 int64_t                 n,
                                                 const hipDoubleComplex* alpha,
                                                 const hipDoubleComplex* AP,
                                                 hipblasStride           strideAP,
                                                 const hipDoubleComplex* x,
                                                 int64_t                 incx,
                                                 hipblasStride           stridex,
                                                 const hipDoubleComplex* beta,
                                                 hipDoubleComplex*       y,
                                                 int64_t                 incy,
                                                 hipblasStride           stridey,
                                                 int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// hpr
hipblasStatus_t hipblasChpr(hipblasHandle_t       handle,
                            hipblasFillMode_t     uplo,
                            int                   n,
                            const float*          alpha,
                            const hipblasComplex* x,
                            int                   incx,
                            hipblasComplex*       AP)
try
{
    return hipblasConvertStatus(cublasChpr((cublasHandle_t)handle,
                                           hipblasConvertFill(uplo),
                                           n,
                                           alpha,
                                           (cuComplex*)x,
                                           incx,
                                           (cuComplex*)AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhpr(hipblasHandle_t             handle,
                            hipblasFillMode_t           uplo,
                            int                         n,
                            const double*               alpha,
                            const hipblasDoubleComplex* x,
                            int                         incx,
                            hipblasDoubleComplex*       AP)
try
{
    return hipblasConvertStatus(cublasZhpr((cublasHandle_t)handle,
                                           hipblasConvertFill(uplo),
                                           n,
                                           alpha,
                                           (cuDoubleComplex*)x,
                                           incx,
                                           (cuDoubleComplex*)AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasChpr_v2(hipblasHandle_t   handle,
                               hipblasFillMode_t uplo,
                               int               n,
                               const float*      alpha,
                               const hipComplex* x,
                               int               incx,
                               hipComplex*       AP)
try
{
    return hipblasConvertStatus(cublasChpr((cublasHandle_t)handle,
                                           hipblasConvertFill(uplo),
                                           n,
                                           alpha,
                                           (cuComplex*)x,
                                           incx,
                                           (cuComplex*)AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhpr_v2(hipblasHandle_t         handle,
                               hipblasFillMode_t       uplo,
                               int                     n,
                               const double*           alpha,
                               const hipDoubleComplex* x,
                               int                     incx,
                               hipDoubleComplex*       AP)
try
{
    return hipblasConvertStatus(cublasZhpr((cublasHandle_t)handle,
                                           hipblasConvertFill(uplo),
                                           n,
                                           alpha,
                                           (cuDoubleComplex*)x,
                                           incx,
                                           (cuDoubleComplex*)AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hpr_64
hipblasStatus_t hipblasChpr_64(hipblasHandle_t       handle,
                               hipblasFillMode_t     uplo,
                               int64_t               n,
                               const float*          alpha,
                               const hipblasComplex* x,
                               int64_t               incx,
                               hipblasComplex*       AP)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasChpr_64((cublasHandle_t)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              alpha,
                                              (cuComplex*)x,
                                              incx,
                                              (cuComplex*)AP));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhpr_64(hipblasHandle_t             handle,
                               hipblasFillMode_t           uplo,
                               int64_t                     n,
                               const double*               alpha,
                               const hipblasDoubleComplex* x,
                               int64_t                     incx,
                               hipblasDoubleComplex*       AP)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZhpr_64((cublasHandle_t)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              alpha,
                                              (cuDoubleComplex*)x,
                                              incx,
                                              (cuDoubleComplex*)AP));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasChpr_v2_64(hipblasHandle_t   handle,
                                  hipblasFillMode_t uplo,
                                  int64_t           n,
                                  const float*      alpha,
                                  const hipComplex* x,
                                  int64_t           incx,
                                  hipComplex*       AP)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasChpr_64((cublasHandle_t)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              alpha,
                                              (cuComplex*)x,
                                              incx,
                                              (cuComplex*)AP));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhpr_v2_64(hipblasHandle_t         handle,
                                  hipblasFillMode_t       uplo,
                                  int64_t                 n,
                                  const double*           alpha,
                                  const hipDoubleComplex* x,
                                  int64_t                 incx,
                                  hipDoubleComplex*       AP)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZhpr_64((cublasHandle_t)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              alpha,
                                              (cuDoubleComplex*)x,
                                              incx,
                                              (cuDoubleComplex*)AP));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hpr_batched
hipblasStatus_t hipblasChprBatched(hipblasHandle_t             handle,
                                   hipblasFillMode_t           uplo,
                                   int                         n,
                                   const float*                alpha,
                                   const hipblasComplex* const x[],
                                   int                         incx,
                                   hipblasComplex* const       AP[],
                                   int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhprBatched(hipblasHandle_t                   handle,
                                   hipblasFillMode_t                 uplo,
                                   int                               n,
                                   const double*                     alpha,
                                   const hipblasDoubleComplex* const x[],
                                   int                               incx,
                                   hipblasDoubleComplex* const       AP[],
                                   int                               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasChprBatched_v2(hipblasHandle_t         handle,
                                      hipblasFillMode_t       uplo,
                                      int                     n,
                                      const float*            alpha,
                                      const hipComplex* const x[],
                                      int                     incx,
                                      hipComplex* const       AP[],
                                      int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhprBatched_v2(hipblasHandle_t               handle,
                                      hipblasFillMode_t             uplo,
                                      int                           n,
                                      const double*                 alpha,
                                      const hipDoubleComplex* const x[],
                                      int                           incx,
                                      hipDoubleComplex* const       AP[],
                                      int                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// hpr_batched_64
hipblasStatus_t hipblasChprBatched_64(hipblasHandle_t             handle,
                                      hipblasFillMode_t           uplo,
                                      int64_t                     n,
                                      const float*                alpha,
                                      const hipblasComplex* const x[],
                                      int64_t                     incx,
                                      hipblasComplex* const       AP[],
                                      int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhprBatched_64(hipblasHandle_t                   handle,
                                      hipblasFillMode_t                 uplo,
                                      int64_t                           n,
                                      const double*                     alpha,
                                      const hipblasDoubleComplex* const x[],
                                      int64_t                           incx,
                                      hipblasDoubleComplex* const       AP[],
                                      int64_t                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasChprBatched_v2_64(hipblasHandle_t         handle,
                                         hipblasFillMode_t       uplo,
                                         int64_t                 n,
                                         const float*            alpha,
                                         const hipComplex* const x[],
                                         int64_t                 incx,
                                         hipComplex* const       AP[],
                                         int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhprBatched_v2_64(hipblasHandle_t               handle,
                                         hipblasFillMode_t             uplo,
                                         int64_t                       n,
                                         const double*                 alpha,
                                         const hipDoubleComplex* const x[],
                                         int64_t                       incx,
                                         hipDoubleComplex* const       AP[],
                                         int64_t                       batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// hpr_strided_batched
hipblasStatus_t hipblasChprStridedBatched(hipblasHandle_t       handle,
                                          hipblasFillMode_t     uplo,
                                          int                   n,
                                          const float*          alpha,
                                          const hipblasComplex* x,
                                          int                   incx,
                                          hipblasStride         stridex,
                                          hipblasComplex*       AP,
                                          hipblasStride         strideAP,
                                          int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhprStridedBatched(hipblasHandle_t             handle,
                                          hipblasFillMode_t           uplo,
                                          int                         n,
                                          const double*               alpha,
                                          const hipblasDoubleComplex* x,
                                          int                         incx,
                                          hipblasStride               stridex,
                                          hipblasDoubleComplex*       AP,
                                          hipblasStride               strideAP,
                                          int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasChprStridedBatched_v2(hipblasHandle_t   handle,
                                             hipblasFillMode_t uplo,
                                             int               n,
                                             const float*      alpha,
                                             const hipComplex* x,
                                             int               incx,
                                             hipblasStride     stridex,
                                             hipComplex*       AP,
                                             hipblasStride     strideAP,
                                             int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhprStridedBatched_v2(hipblasHandle_t         handle,
                                             hipblasFillMode_t       uplo,
                                             int                     n,
                                             const double*           alpha,
                                             const hipDoubleComplex* x,
                                             int                     incx,
                                             hipblasStride           stridex,
                                             hipDoubleComplex*       AP,
                                             hipblasStride           strideAP,
                                             int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// hpr_strided_batched_64
hipblasStatus_t hipblasChprStridedBatched_64(hipblasHandle_t       handle,
                                             hipblasFillMode_t     uplo,
                                             int64_t               n,
                                             const float*          alpha,
                                             const hipblasComplex* x,
                                             int64_t               incx,
                                             hipblasStride         stridex,
                                             hipblasComplex*       AP,
                                             hipblasStride         strideAP,
                                             int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhprStridedBatched_64(hipblasHandle_t             handle,
                                             hipblasFillMode_t           uplo,
                                             int64_t                     n,
                                             const double*               alpha,
                                             const hipblasDoubleComplex* x,
                                             int64_t                     incx,
                                             hipblasStride               stridex,
                                             hipblasDoubleComplex*       AP,
                                             hipblasStride               strideAP,
                                             int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasChprStridedBatched_v2_64(hipblasHandle_t   handle,
                                                hipblasFillMode_t uplo,
                                                int64_t           n,
                                                const float*      alpha,
                                                const hipComplex* x,
                                                int64_t           incx,
                                                hipblasStride     stridex,
                                                hipComplex*       AP,
                                                hipblasStride     strideAP,
                                                int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhprStridedBatched_v2_64(hipblasHandle_t         handle,
                                                hipblasFillMode_t       uplo,
                                                int64_t                 n,
                                                const double*           alpha,
                                                const hipDoubleComplex* x,
                                                int64_t                 incx,
                                                hipblasStride           stridex,
                                                hipDoubleComplex*       AP,
                                                hipblasStride           strideAP,
                                                int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// hpr2
hipblasStatus_t hipblasChpr2(hipblasHandle_t       handle,
                             hipblasFillMode_t     uplo,
                             int                   n,
                             const hipblasComplex* alpha,
                             const hipblasComplex* x,
                             int                   incx,
                             const hipblasComplex* y,
                             int                   incy,
                             hipblasComplex*       AP)
try
{
    return hipblasConvertStatus(cublasChpr2((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            (cuComplex*)alpha,
                                            (cuComplex*)x,
                                            incx,
                                            (cuComplex*)y,
                                            incy,
                                            (cuComplex*)AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhpr2(hipblasHandle_t             handle,
                             hipblasFillMode_t           uplo,
                             int                         n,
                             const hipblasDoubleComplex* alpha,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             const hipblasDoubleComplex* y,
                             int                         incy,
                             hipblasDoubleComplex*       AP)
try
{
    return hipblasConvertStatus(cublasZhpr2((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            (cuDoubleComplex*)alpha,
                                            (cuDoubleComplex*)x,
                                            incx,
                                            (cuDoubleComplex*)y,
                                            incy,
                                            (cuDoubleComplex*)AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasChpr2_v2(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int               n,
                                const hipComplex* alpha,
                                const hipComplex* x,
                                int               incx,
                                const hipComplex* y,
                                int               incy,
                                hipComplex*       AP)
try
{
    return hipblasConvertStatus(cublasChpr2((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            (cuComplex*)alpha,
                                            (cuComplex*)x,
                                            incx,
                                            (cuComplex*)y,
                                            incy,
                                            (cuComplex*)AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhpr2_v2(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                int                     n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* x,
                                int                     incx,
                                const hipDoubleComplex* y,
                                int                     incy,
                                hipDoubleComplex*       AP)
try
{
    return hipblasConvertStatus(cublasZhpr2((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            (cuDoubleComplex*)alpha,
                                            (cuDoubleComplex*)x,
                                            incx,
                                            (cuDoubleComplex*)y,
                                            incy,
                                            (cuDoubleComplex*)AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hpr2_64
hipblasStatus_t hipblasChpr2_64(hipblasHandle_t       handle,
                                hipblasFillMode_t     uplo,
                                int64_t               n,
                                const hipblasComplex* alpha,
                                const hipblasComplex* x,
                                int64_t               incx,
                                const hipblasComplex* y,
                                int64_t               incy,
                                hipblasComplex*       AP)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasChpr2_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               (cuComplex*)alpha,
                                               (cuComplex*)x,
                                               incx,
                                               (cuComplex*)y,
                                               incy,
                                               (cuComplex*)AP));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhpr2_64(hipblasHandle_t             handle,
                                hipblasFillMode_t           uplo,
                                int64_t                     n,
                                const hipblasDoubleComplex* alpha,
                                const hipblasDoubleComplex* x,
                                int64_t                     incx,
                                const hipblasDoubleComplex* y,
                                int64_t                     incy,
                                hipblasDoubleComplex*       AP)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZhpr2_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               (cuDoubleComplex*)alpha,
                                               (cuDoubleComplex*)x,
                                               incx,
                                               (cuDoubleComplex*)y,
                                               incy,
                                               (cuDoubleComplex*)AP));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasChpr2_v2_64(hipblasHandle_t   handle,
                                   hipblasFillMode_t uplo,
                                   int64_t           n,
                                   const hipComplex* alpha,
                                   const hipComplex* x,
                                   int64_t           incx,
                                   const hipComplex* y,
                                   int64_t           incy,
                                   hipComplex*       AP)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasChpr2_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               (cuComplex*)alpha,
                                               (cuComplex*)x,
                                               incx,
                                               (cuComplex*)y,
                                               incy,
                                               (cuComplex*)AP));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhpr2_v2_64(hipblasHandle_t         handle,
                                   hipblasFillMode_t       uplo,
                                   int64_t                 n,
                                   const hipDoubleComplex* alpha,
                                   const hipDoubleComplex* x,
                                   int64_t                 incx,
                                   const hipDoubleComplex* y,
                                   int64_t                 incy,
                                   hipDoubleComplex*       AP)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZhpr2_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               (cuDoubleComplex*)alpha,
                                               (cuDoubleComplex*)x,
                                               incx,
                                               (cuDoubleComplex*)y,
                                               incy,
                                               (cuDoubleComplex*)AP));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hpr2_batched
hipblasStatus_t hipblasChpr2Batched(hipblasHandle_t             handle,
                                    hipblasFillMode_t           uplo,
                                    int                         n,
                                    const hipblasComplex*       alpha,
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    const hipblasComplex* const yp[],
                                    int                         incy,
                                    hipblasComplex* const       AP[],
                                    int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhpr2Batched(hipblasHandle_t                   handle,
                                    hipblasFillMode_t                 uplo,
                                    int                               n,
                                    const hipblasDoubleComplex*       alpha,
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    const hipblasDoubleComplex* const yp[],
                                    int                               incy,
                                    hipblasDoubleComplex* const       AP[],
                                    int                               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasChpr2Batched_v2(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       int                     n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const x[],
                                       int                     incx,
                                       const hipComplex* const yp[],
                                       int                     incy,
                                       hipComplex* const       AP[],
                                       int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhpr2Batched_v2(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       int                           n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       const hipDoubleComplex* const yp[],
                                       int                           incy,
                                       hipDoubleComplex* const       AP[],
                                       int                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// hpr2_batched
hipblasStatus_t hipblasChpr2Batched_64(hipblasHandle_t             handle,
                                       hipblasFillMode_t           uplo,
                                       int64_t                     n,
                                       const hipblasComplex*       alpha,
                                       const hipblasComplex* const x[],
                                       int64_t                     incx,
                                       const hipblasComplex* const y[],
                                       int64_t                     incy,
                                       hipblasComplex* const       AP[],
                                       int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhpr2Batched_64(hipblasHandle_t                   handle,
                                       hipblasFillMode_t                 uplo,
                                       int64_t                           n,
                                       const hipblasDoubleComplex*       alpha,
                                       const hipblasDoubleComplex* const x[],
                                       int64_t                           incx,
                                       const hipblasDoubleComplex* const y[],
                                       int64_t                           incy,
                                       hipblasDoubleComplex* const       AP[],
                                       int64_t                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasChpr2Batched_v2_64(hipblasHandle_t         handle,
                                          hipblasFillMode_t       uplo,
                                          int64_t                 n,
                                          const hipComplex*       alpha,
                                          const hipComplex* const x[],
                                          int64_t                 incx,
                                          const hipComplex* const y[],
                                          int64_t                 incy,
                                          hipComplex* const       AP[],
                                          int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhpr2Batched_v2_64(hipblasHandle_t               handle,
                                          hipblasFillMode_t             uplo,
                                          int64_t                       n,
                                          const hipDoubleComplex*       alpha,
                                          const hipDoubleComplex* const x[],
                                          int64_t                       incx,
                                          const hipDoubleComplex* const y[],
                                          int64_t                       incy,
                                          hipDoubleComplex* const       AP[],
                                          int64_t                       batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// hpr2_strided_batched
hipblasStatus_t hipblasChpr2StridedBatched(hipblasHandle_t       handle,
                                           hipblasFillMode_t     uplo,
                                           int                   n,
                                           const hipblasComplex* alpha,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           const hipblasComplex* y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           hipblasComplex*       AP,
                                           hipblasStride         strideAP,
                                           int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhpr2StridedBatched(hipblasHandle_t             handle,
                                           hipblasFillMode_t           uplo,
                                           int                         n,
                                           const hipblasDoubleComplex* alpha,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           const hipblasDoubleComplex* y,
                                           int                         incy,
                                           hipblasStride               stridey,
                                           hipblasDoubleComplex*       AP,
                                           hipblasStride               strideAP,
                                           int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasChpr2StridedBatched_v2(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int               n,
                                              const hipComplex* alpha,
                                              const hipComplex* x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              const hipComplex* y,
                                              int               incy,
                                              hipblasStride     stridey,
                                              hipComplex*       AP,
                                              hipblasStride     strideAP,
                                              int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhpr2StridedBatched_v2(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              int                     n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* y,
                                              int                     incy,
                                              hipblasStride           stridey,
                                              hipDoubleComplex*       AP,
                                              hipblasStride           strideAP,
                                              int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// hpr2_strided_batched_64
hipblasStatus_t hipblasChpr2StridedBatched_64(hipblasHandle_t       handle,
                                              hipblasFillMode_t     uplo,
                                              int64_t               n,
                                              const hipblasComplex* alpha,
                                              const hipblasComplex* x,
                                              int64_t               incx,
                                              hipblasStride         stridex,
                                              const hipblasComplex* y,
                                              int64_t               incy,
                                              hipblasStride         stridey,
                                              hipblasComplex*       AP,
                                              hipblasStride         strideAP,
                                              int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhpr2StridedBatched_64(hipblasHandle_t             handle,
                                              hipblasFillMode_t           uplo,
                                              int64_t                     n,
                                              const hipblasDoubleComplex* alpha,
                                              const hipblasDoubleComplex* x,
                                              int64_t                     incx,
                                              hipblasStride               stridex,
                                              const hipblasDoubleComplex* y,
                                              int64_t                     incy,
                                              hipblasStride               stridey,
                                              hipblasDoubleComplex*       AP,
                                              hipblasStride               strideAP,
                                              int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasChpr2StridedBatched_v2_64(hipblasHandle_t   handle,
                                                 hipblasFillMode_t uplo,
                                                 int64_t           n,
                                                 const hipComplex* alpha,
                                                 const hipComplex* x,
                                                 int64_t           incx,
                                                 hipblasStride     stridex,
                                                 const hipComplex* y,
                                                 int64_t           incy,
                                                 hipblasStride     stridey,
                                                 hipComplex*       AP,
                                                 hipblasStride     strideAP,
                                                 int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZhpr2StridedBatched_v2_64(hipblasHandle_t         handle,
                                                 hipblasFillMode_t       uplo,
                                                 int64_t                 n,
                                                 const hipDoubleComplex* alpha,
                                                 const hipDoubleComplex* x,
                                                 int64_t                 incx,
                                                 hipblasStride           stridex,
                                                 const hipDoubleComplex* y,
                                                 int64_t                 incy,
                                                 hipblasStride           stridey,
                                                 hipDoubleComplex*       AP,
                                                 hipblasStride           strideAP,
                                                 int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// sbmv
hipblasStatus_t hipblasSsbmv(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             int               k,
                             const float*      alpha,
                             const float*      A,
                             int               lda,
                             const float*      x,
                             int               incx,
                             const float*      beta,
                             float*            y,
                             int               incy)
try
{
    return hipblasConvertStatus(cublasSsbmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            k,
                                            alpha,
                                            A,
                                            lda,
                                            x,
                                            incx,
                                            beta,
                                            y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsbmv(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             int               k,
                             const double*     alpha,
                             const double*     A,
                             int               lda,
                             const double*     x,
                             int               incx,
                             const double*     beta,
                             double*           y,
                             int               incy)
try
{
    return hipblasConvertStatus(cublasDsbmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            k,
                                            alpha,
                                            A,
                                            lda,
                                            x,
                                            incx,
                                            beta,
                                            y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// sbmv_64
hipblasStatus_t hipblasSsbmv_64(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int64_t           n,
                                int64_t           k,
                                const float*      alpha,
                                const float*      A,
                                int64_t           lda,
                                const float*      x,
                                int64_t           incx,
                                const float*      beta,
                                float*            y,
                                int64_t           incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasSsbmv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               k,
                                               alpha,
                                               A,
                                               lda,
                                               x,
                                               incx,
                                               beta,
                                               y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsbmv_64(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int64_t           n,
                                int64_t           k,
                                const double*     alpha,
                                const double*     A,
                                int64_t           lda,
                                const double*     x,
                                int64_t           incx,
                                const double*     beta,
                                double*           y,
                                int64_t           incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasDsbmv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               k,
                                               alpha,
                                               A,
                                               lda,
                                               x,
                                               incx,
                                               beta,
                                               y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// sbmv_batched
hipblasStatus_t hipblasSsbmvBatched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    int                n,
                                    int                k,
                                    const float*       alpha,
                                    const float* const A[],
                                    int                lda,
                                    const float* const x[],
                                    int                incx,
                                    const float*       beta,
                                    float* const       y[],
                                    int                incy,
                                    int                batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDsbmvBatched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    int                 n,
                                    int                 k,
                                    const double*       alpha,
                                    const double* const A[],
                                    int                 lda,
                                    const double* const x[],
                                    int                 incx,
                                    const double*       beta,
                                    double* const       y[],
                                    int                 incy,
                                    int                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// sbmv_batched_64
hipblasStatus_t hipblasSsbmvBatched_64(hipblasHandle_t    handle,
                                       hipblasFillMode_t  uplo,
                                       int64_t            n,
                                       int64_t            k,
                                       const float*       alpha,
                                       const float* const A[],
                                       int64_t            lda,
                                       const float* const x[],
                                       int64_t            incx,
                                       const float*       beta,
                                       float* const       y[],
                                       int64_t            incy,
                                       int64_t            batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDsbmvBatched_64(hipblasHandle_t     handle,
                                       hipblasFillMode_t   uplo,
                                       int64_t             n,
                                       int64_t             k,
                                       const double*       alpha,
                                       const double* const A[],
                                       int64_t             lda,
                                       const double* const x[],
                                       int64_t             incx,
                                       const double*       beta,
                                       double* const       y[],
                                       int64_t             incy,
                                       int64_t             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// sbmv_strided_batched
hipblasStatus_t hipblasSsbmvStridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           int               k,
                                           const float*      alpha,
                                           const float*      A,
                                           int               lda,
                                           hipblasStride     strideA,
                                           const float*      x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const float*      beta,
                                           float*            y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDsbmvStridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           int               k,
                                           const double*     alpha,
                                           const double*     A,
                                           int               lda,
                                           hipblasStride     strideA,
                                           const double*     x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const double*     beta,
                                           double*           y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// sbmv_strided_batched_64
hipblasStatus_t hipblasSsbmvStridedBatched_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              int64_t           k,
                                              const float*      alpha,
                                              const float*      A,
                                              int64_t           lda,
                                              hipblasStride     strideA,
                                              const float*      x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const float*      beta,
                                              float*            y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDsbmvStridedBatched_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              int64_t           k,
                                              const double*     alpha,
                                              const double*     A,
                                              int64_t           lda,
                                              hipblasStride     strideA,
                                              const double*     x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const double*     beta,
                                              double*           y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// spmv
hipblasStatus_t hipblasSspmv(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const float*      alpha,
                             const float*      AP,
                             const float*      x,
                             int               incx,
                             const float*      beta,
                             float*            y,
                             int               incy)
try
{
    return hipblasConvertStatus(cublasSspmv(
        (cublasHandle_t)handle, hipblasConvertFill(uplo), n, alpha, AP, x, incx, beta, y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDspmv(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const double*     alpha,
                             const double*     AP,
                             const double*     x,
                             int               incx,
                             const double*     beta,
                             double*           y,
                             int               incy)
try
{
    return hipblasConvertStatus(cublasDspmv(
        (cublasHandle_t)handle, hipblasConvertFill(uplo), n, alpha, AP, x, incx, beta, y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// spmv_64
hipblasStatus_t hipblasSspmv_64(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int64_t           n,
                                const float*      alpha,
                                const float*      AP,
                                const float*      x,
                                int64_t           incx,
                                const float*      beta,
                                float*            y,
                                int64_t           incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasSspmv_64(
        (cublasHandle_t)handle, hipblasConvertFill(uplo), n, alpha, AP, x, incx, beta, y, incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDspmv_64(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int64_t           n,
                                const double*     alpha,
                                const double*     AP,
                                const double*     x,
                                int64_t           incx,
                                const double*     beta,
                                double*           y,
                                int64_t           incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasDspmv_64(
        (cublasHandle_t)handle, hipblasConvertFill(uplo), n, alpha, AP, x, incx, beta, y, incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// spmv_batched
hipblasStatus_t hipblasSspmvBatched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    int                n,
                                    const float*       alpha,
                                    const float* const AP[],
                                    const float* const x[],
                                    int                incx,
                                    const float*       beta,
                                    float* const       y[],
                                    int                incy,
                                    int                batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDspmvBatched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    int                 n,
                                    const double*       alpha,
                                    const double* const AP[],
                                    const double* const x[],
                                    int                 incx,
                                    const double*       beta,
                                    double* const       y[],
                                    int                 incy,
                                    int                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// spmv_batched_64
hipblasStatus_t hipblasSspmvBatched_64(hipblasHandle_t    handle,
                                       hipblasFillMode_t  uplo,
                                       int64_t            n,
                                       const float*       alpha,
                                       const float* const AP[],
                                       const float* const x[],
                                       int64_t            incx,
                                       const float*       beta,
                                       float* const       y[],
                                       int64_t            incy,
                                       int64_t            batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDspmvBatched_64(hipblasHandle_t     handle,
                                       hipblasFillMode_t   uplo,
                                       int64_t             n,
                                       const double*       alpha,
                                       const double* const AP[],
                                       const double* const x[],
                                       int64_t             incx,
                                       const double*       beta,
                                       double* const       y[],
                                       int64_t             incy,
                                       int64_t             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// spmv_strided_batched
hipblasStatus_t hipblasSspmvStridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const float*      alpha,
                                           const float*      AP,
                                           hipblasStride     strideAP,
                                           const float*      x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const float*      beta,
                                           float*            y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDspmvStridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const double*     alpha,
                                           const double*     AP,
                                           hipblasStride     strideAP,
                                           const double*     x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const double*     beta,
                                           double*           y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// spmv_strided_batched_64
hipblasStatus_t hipblasSspmvStridedBatched_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              const float*      alpha,
                                              const float*      AP,
                                              hipblasStride     strideAP,
                                              const float*      x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const float*      beta,
                                              float*            y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDspmvStridedBatched_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              const double*     alpha,
                                              const double*     AP,
                                              hipblasStride     strideAP,
                                              const double*     x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const double*     beta,
                                              double*           y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// spr
hipblasStatus_t hipblasSspr(hipblasHandle_t   handle,
                            hipblasFillMode_t uplo,
                            int               n,
                            const float*      alpha,
                            const float*      x,
                            int               incx,
                            float*            AP)
try
{
    return hipblasConvertStatus(
        cublasSspr((cublasHandle_t)handle, hipblasConvertFill(uplo), n, alpha, x, incx, AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDspr(hipblasHandle_t   handle,
                            hipblasFillMode_t uplo,
                            int               n,
                            const double*     alpha,
                            const double*     x,
                            int               incx,
                            double*           AP)
try
{
    return hipblasConvertStatus(
        cublasDspr((cublasHandle_t)handle, hipblasConvertFill(uplo), n, alpha, x, incx, AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCspr(hipblasHandle_t       handle,
                            hipblasFillMode_t     uplo,
                            int                   n,
                            const hipblasComplex* alpha,
                            const hipblasComplex* x,
                            int                   incx,
                            hipblasComplex*       AP)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZspr(hipblasHandle_t             handle,
                            hipblasFillMode_t           uplo,
                            int                         n,
                            const hipblasDoubleComplex* alpha,
                            const hipblasDoubleComplex* x,
                            int                         incx,
                            hipblasDoubleComplex*       AP)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCspr_v2(hipblasHandle_t   handle,
                               hipblasFillMode_t uplo,
                               int               n,
                               const hipComplex* alpha,
                               const hipComplex* x,
                               int               incx,
                               hipComplex*       AP)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZspr_v2(hipblasHandle_t         handle,
                               hipblasFillMode_t       uplo,
                               int                     n,
                               const hipDoubleComplex* alpha,
                               const hipDoubleComplex* x,
                               int                     incx,
                               hipDoubleComplex*       AP)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// spr_64
hipblasStatus_t hipblasSspr_64(hipblasHandle_t   handle,
                               hipblasFillMode_t uplo,
                               int64_t           n,
                               const float*      alpha,
                               const float*      x,
                               int64_t           incx,
                               float*            AP)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasSspr_64((cublasHandle_t)handle, hipblasConvertFill(uplo), n, alpha, x, incx, AP));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDspr_64(hipblasHandle_t   handle,
                               hipblasFillMode_t uplo,
                               int64_t           n,
                               const double*     alpha,
                               const double*     x,
                               int64_t           incx,
                               double*           AP)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasDspr_64((cublasHandle_t)handle, hipblasConvertFill(uplo), n, alpha, x, incx, AP));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCspr_64(hipblasHandle_t       handle,
                               hipblasFillMode_t     uplo,
                               int64_t               n,
                               const hipblasComplex* alpha,
                               const hipblasComplex* x,
                               int64_t               incx,
                               hipblasComplex*       AP)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZspr_64(hipblasHandle_t             handle,
                               hipblasFillMode_t           uplo,
                               int64_t                     n,
                               const hipblasDoubleComplex* alpha,
                               const hipblasDoubleComplex* x,
                               int64_t                     incx,
                               hipblasDoubleComplex*       AP)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCspr_v2_64(hipblasHandle_t   handle,
                                  hipblasFillMode_t uplo,
                                  int64_t           n,
                                  const hipComplex* alpha,
                                  const hipComplex* x,
                                  int64_t           incx,
                                  hipComplex*       AP)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZspr_v2_64(hipblasHandle_t         handle,
                                  hipblasFillMode_t       uplo,
                                  int64_t                 n,
                                  const hipDoubleComplex* alpha,
                                  const hipDoubleComplex* x,
                                  int64_t                 incx,
                                  hipDoubleComplex*       AP)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// spr_batched
hipblasStatus_t hipblasSsprBatched(hipblasHandle_t    handle,
                                   hipblasFillMode_t  uplo,
                                   int                n,
                                   const float*       alpha,
                                   const float* const x[],
                                   int                incx,
                                   float* const       AP[],
                                   int                batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDsprBatched(hipblasHandle_t     handle,
                                   hipblasFillMode_t   uplo,
                                   int                 n,
                                   const double*       alpha,
                                   const double* const x[],
                                   int                 incx,
                                   double* const       AP[],
                                   int                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsprBatched(hipblasHandle_t             handle,
                                   hipblasFillMode_t           uplo,
                                   int                         n,
                                   const hipblasComplex*       alpha,
                                   const hipblasComplex* const x[],
                                   int                         incx,
                                   hipblasComplex* const       AP[],
                                   int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsprBatched(hipblasHandle_t                   handle,
                                   hipblasFillMode_t                 uplo,
                                   int                               n,
                                   const hipblasDoubleComplex*       alpha,
                                   const hipblasDoubleComplex* const x[],
                                   int                               incx,
                                   hipblasDoubleComplex* const       AP[],
                                   int                               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsprBatched_v2(hipblasHandle_t         handle,
                                      hipblasFillMode_t       uplo,
                                      int                     n,
                                      const hipComplex*       alpha,
                                      const hipComplex* const x[],
                                      int                     incx,
                                      hipComplex* const       AP[],
                                      int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsprBatched_v2(hipblasHandle_t               handle,
                                      hipblasFillMode_t             uplo,
                                      int                           n,
                                      const hipDoubleComplex*       alpha,
                                      const hipDoubleComplex* const x[],
                                      int                           incx,
                                      hipDoubleComplex* const       AP[],
                                      int                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// spr_batched_64
hipblasStatus_t hipblasSsprBatched_64(hipblasHandle_t    handle,
                                      hipblasFillMode_t  uplo,
                                      int64_t            n,
                                      const float*       alpha,
                                      const float* const x[],
                                      int64_t            incx,
                                      float* const       AP[],
                                      int64_t            batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDsprBatched_64(hipblasHandle_t     handle,
                                      hipblasFillMode_t   uplo,
                                      int64_t             n,
                                      const double*       alpha,
                                      const double* const x[],
                                      int64_t             incx,
                                      double* const       AP[],
                                      int64_t             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsprBatched_64(hipblasHandle_t             handle,
                                      hipblasFillMode_t           uplo,
                                      int64_t                     n,
                                      const hipblasComplex*       alpha,
                                      const hipblasComplex* const x[],
                                      int64_t                     incx,
                                      hipblasComplex* const       AP[],
                                      int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsprBatched_64(hipblasHandle_t                   handle,
                                      hipblasFillMode_t                 uplo,
                                      int64_t                           n,
                                      const hipblasDoubleComplex*       alpha,
                                      const hipblasDoubleComplex* const x[],
                                      int64_t                           incx,
                                      hipblasDoubleComplex* const       AP[],
                                      int64_t                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsprBatched_v2_64(hipblasHandle_t         handle,
                                         hipblasFillMode_t       uplo,
                                         int64_t                 n,
                                         const hipComplex*       alpha,
                                         const hipComplex* const x[],
                                         int64_t                 incx,
                                         hipComplex* const       AP[],
                                         int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsprBatched_v2_64(hipblasHandle_t               handle,
                                         hipblasFillMode_t             uplo,
                                         int64_t                       n,
                                         const hipDoubleComplex*       alpha,
                                         const hipDoubleComplex* const x[],
                                         int64_t                       incx,
                                         hipDoubleComplex* const       AP[],
                                         int64_t                       batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// spr_strided_batched
hipblasStatus_t hipblasSsprStridedBatched(hipblasHandle_t   handle,
                                          hipblasFillMode_t uplo,
                                          int               n,
                                          const float*      alpha,
                                          const float*      x,
                                          int               incx,
                                          hipblasStride     stridex,
                                          float*            AP,
                                          hipblasStride     strideAP,
                                          int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDsprStridedBatched(hipblasHandle_t   handle,
                                          hipblasFillMode_t uplo,
                                          int               n,
                                          const double*     alpha,
                                          const double*     x,
                                          int               incx,
                                          hipblasStride     stridex,
                                          double*           AP,
                                          hipblasStride     strideAP,
                                          int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsprStridedBatched(hipblasHandle_t       handle,
                                          hipblasFillMode_t     uplo,
                                          int                   n,
                                          const hipblasComplex* alpha,
                                          const hipblasComplex* x,
                                          int                   incx,
                                          hipblasStride         stridex,
                                          hipblasComplex*       AP,
                                          hipblasStride         strideAP,
                                          int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsprStridedBatched(hipblasHandle_t             handle,
                                          hipblasFillMode_t           uplo,
                                          int                         n,
                                          const hipblasDoubleComplex* alpha,
                                          const hipblasDoubleComplex* x,
                                          int                         incx,
                                          hipblasStride               stridex,
                                          hipblasDoubleComplex*       AP,
                                          hipblasStride               strideAP,
                                          int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsprStridedBatched_v2(hipblasHandle_t   handle,
                                             hipblasFillMode_t uplo,
                                             int               n,
                                             const hipComplex* alpha,
                                             const hipComplex* x,
                                             int               incx,
                                             hipblasStride     stridex,
                                             hipComplex*       AP,
                                             hipblasStride     strideAP,
                                             int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsprStridedBatched_v2(hipblasHandle_t         handle,
                                             hipblasFillMode_t       uplo,
                                             int                     n,
                                             const hipDoubleComplex* alpha,
                                             const hipDoubleComplex* x,
                                             int                     incx,
                                             hipblasStride           stridex,
                                             hipDoubleComplex*       AP,
                                             hipblasStride           strideAP,
                                             int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// spr_strided_batched_64
hipblasStatus_t hipblasSsprStridedBatched_64(hipblasHandle_t   handle,
                                             hipblasFillMode_t uplo,
                                             int64_t           n,
                                             const float*      alpha,
                                             const float*      x,
                                             int64_t           incx,
                                             hipblasStride     stridex,
                                             float*            AP,
                                             hipblasStride     strideAP,
                                             int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDsprStridedBatched_64(hipblasHandle_t   handle,
                                             hipblasFillMode_t uplo,
                                             int64_t           n,
                                             const double*     alpha,
                                             const double*     x,
                                             int64_t           incx,
                                             hipblasStride     stridex,
                                             double*           AP,
                                             hipblasStride     strideAP,
                                             int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsprStridedBatched_64(hipblasHandle_t       handle,
                                             hipblasFillMode_t     uplo,
                                             int64_t               n,
                                             const hipblasComplex* alpha,
                                             const hipblasComplex* x,
                                             int64_t               incx,
                                             hipblasStride         stridex,
                                             hipblasComplex*       AP,
                                             hipblasStride         strideAP,
                                             int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsprStridedBatched_64(hipblasHandle_t             handle,
                                             hipblasFillMode_t           uplo,
                                             int64_t                     n,
                                             const hipblasDoubleComplex* alpha,
                                             const hipblasDoubleComplex* x,
                                             int64_t                     incx,
                                             hipblasStride               stridex,
                                             hipblasDoubleComplex*       AP,
                                             hipblasStride               strideAP,
                                             int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsprStridedBatched_v2_64(hipblasHandle_t   handle,
                                                hipblasFillMode_t uplo,
                                                int64_t           n,
                                                const hipComplex* alpha,
                                                const hipComplex* x,
                                                int64_t           incx,
                                                hipblasStride     stridex,
                                                hipComplex*       AP,
                                                hipblasStride     strideAP,
                                                int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsprStridedBatched_v2_64(hipblasHandle_t         handle,
                                                hipblasFillMode_t       uplo,
                                                int64_t                 n,
                                                const hipDoubleComplex* alpha,
                                                const hipDoubleComplex* x,
                                                int64_t                 incx,
                                                hipblasStride           stridex,
                                                hipDoubleComplex*       AP,
                                                hipblasStride           strideAP,
                                                int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// spr2
hipblasStatus_t hipblasSspr2(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const float*      alpha,
                             const float*      x,
                             int               incx,
                             const float*      y,
                             int               incy,
                             float*            AP)
try
{
    return hipblasConvertStatus(cublasSspr2(
        (cublasHandle_t)handle, hipblasConvertFill(uplo), n, alpha, x, incx, y, incy, AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDspr2(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const double*     alpha,
                             const double*     x,
                             int               incx,
                             const double*     y,
                             int               incy,
                             double*           AP)
try
{
    return hipblasConvertStatus(cublasDspr2(
        (cublasHandle_t)handle, hipblasConvertFill(uplo), n, alpha, x, incx, y, incy, AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// spr2_64
hipblasStatus_t hipblasSspr2_64(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int64_t           n,
                                const float*      alpha,
                                const float*      x,
                                int64_t           incx,
                                const float*      y,
                                int64_t           incy,
                                float*            AP)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasSspr2_64(
        (cublasHandle_t)handle, hipblasConvertFill(uplo), n, alpha, x, incx, y, incy, AP));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDspr2_64(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int64_t           n,
                                const double*     alpha,
                                const double*     x,
                                int64_t           incx,
                                const double*     y,
                                int64_t           incy,
                                double*           AP)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasDspr2_64(
        (cublasHandle_t)handle, hipblasConvertFill(uplo), n, alpha, x, incx, y, incy, AP));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// spr2_batched
hipblasStatus_t hipblasSspr2Batched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    int                n,
                                    const float*       alpha,
                                    const float* const x[],
                                    int                incx,
                                    const float* const y[],
                                    int                incy,
                                    float* const       AP[],
                                    int                batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDspr2Batched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    int                 n,
                                    const double*       alpha,
                                    const double* const x[],
                                    int                 incx,
                                    const double* const y[],
                                    int                 incy,
                                    double* const       AP[],
                                    int                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// spr2_batched_64
hipblasStatus_t hipblasSspr2Batched_64(hipblasHandle_t    handle,
                                       hipblasFillMode_t  uplo,
                                       int64_t            n,
                                       const float*       alpha,
                                       const float* const x[],
                                       int64_t            incx,
                                       const float* const y[],
                                       int64_t            incy,
                                       float* const       AP[],
                                       int64_t            batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDspr2Batched_64(hipblasHandle_t     handle,
                                       hipblasFillMode_t   uplo,
                                       int64_t             n,
                                       const double*       alpha,
                                       const double* const x[],
                                       int64_t             incx,
                                       const double* const y[],
                                       int64_t             incy,
                                       double* const       AP[],
                                       int64_t             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// spr2_strided_batched
hipblasStatus_t hipblasSspr2StridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const float*      alpha,
                                           const float*      x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const float*      y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           float*            AP,
                                           hipblasStride     strideAP,
                                           int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDspr2StridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const double*     alpha,
                                           const double*     x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const double*     y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           double*           AP,
                                           hipblasStride     strideAP,
                                           int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// spr2_strided_batched_64
hipblasStatus_t hipblasSspr2StridedBatched_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              const float*      alpha,
                                              const float*      x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const float*      y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              float*            AP,
                                              hipblasStride     strideAP,
                                              int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDspr2StridedBatched_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              const double*     alpha,
                                              const double*     x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const double*     y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              double*           AP,
                                              hipblasStride     strideAP,
                                              int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// symv
hipblasStatus_t hipblasSsymv(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const float*      alpha,
                             const float*      A,
                             int               lda,
                             const float*      x,
                             int               incx,
                             const float*      beta,
                             float*            y,
                             int               incy)
try
{
    return hipblasConvertStatus(cublasSsymv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            alpha,
                                            A,
                                            lda,
                                            x,
                                            incx,
                                            beta,
                                            y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsymv(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const double*     alpha,
                             const double*     A,
                             int               lda,
                             const double*     x,
                             int               incx,
                             const double*     beta,
                             double*           y,
                             int               incy)
try
{
    return hipblasConvertStatus(cublasDsymv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            alpha,
                                            A,
                                            lda,
                                            x,
                                            incx,
                                            beta,
                                            y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsymv(hipblasHandle_t       handle,
                             hipblasFillMode_t     uplo,
                             int                   n,
                             const hipblasComplex* alpha,
                             const hipblasComplex* A,
                             int                   lda,
                             const hipblasComplex* x,
                             int                   incx,
                             const hipblasComplex* beta,
                             hipblasComplex*       y,
                             int                   incy)
try
{
    return hipblasConvertStatus(cublasCsymv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            (cuComplex*)alpha,
                                            (cuComplex*)A,
                                            lda,
                                            (cuComplex*)x,
                                            incx,
                                            (cuComplex*)beta,
                                            (cuComplex*)y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsymv(hipblasHandle_t             handle,
                             hipblasFillMode_t           uplo,
                             int                         n,
                             const hipblasDoubleComplex* alpha,
                             const hipblasDoubleComplex* A,
                             int                         lda,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             const hipblasDoubleComplex* beta,
                             hipblasDoubleComplex*       y,
                             int                         incy)
try
{
    return hipblasConvertStatus(cublasZsymv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            (cuDoubleComplex*)alpha,
                                            (cuDoubleComplex*)A,
                                            lda,
                                            (cuDoubleComplex*)x,
                                            incx,
                                            (cuDoubleComplex*)beta,
                                            (cuDoubleComplex*)y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsymv_v2(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int               n,
                                const hipComplex* alpha,
                                const hipComplex* A,
                                int               lda,
                                const hipComplex* x,
                                int               incx,
                                const hipComplex* beta,
                                hipComplex*       y,
                                int               incy)
try
{
    return hipblasConvertStatus(cublasCsymv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            (cuComplex*)alpha,
                                            (cuComplex*)A,
                                            lda,
                                            (cuComplex*)x,
                                            incx,
                                            (cuComplex*)beta,
                                            (cuComplex*)y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsymv_v2(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                int                     n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* A,
                                int                     lda,
                                const hipDoubleComplex* x,
                                int                     incx,
                                const hipDoubleComplex* beta,
                                hipDoubleComplex*       y,
                                int                     incy)
try
{
    return hipblasConvertStatus(cublasZsymv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            (cuDoubleComplex*)alpha,
                                            (cuDoubleComplex*)A,
                                            lda,
                                            (cuDoubleComplex*)x,
                                            incx,
                                            (cuDoubleComplex*)beta,
                                            (cuDoubleComplex*)y,
                                            incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// symv_64
hipblasStatus_t hipblasSsymv_64(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int64_t           n,
                                const float*      alpha,
                                const float*      A,
                                int64_t           lda,
                                const float*      x,
                                int64_t           incx,
                                const float*      beta,
                                float*            y,
                                int64_t           incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasSsymv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               alpha,
                                               A,
                                               lda,
                                               x,
                                               incx,
                                               beta,
                                               y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsymv_64(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int64_t           n,
                                const double*     alpha,
                                const double*     A,
                                int64_t           lda,
                                const double*     x,
                                int64_t           incx,
                                const double*     beta,
                                double*           y,
                                int64_t           incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasDsymv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               alpha,
                                               A,
                                               lda,
                                               x,
                                               incx,
                                               beta,
                                               y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsymv_64(hipblasHandle_t       handle,
                                hipblasFillMode_t     uplo,
                                int64_t               n,
                                const hipblasComplex* alpha,
                                const hipblasComplex* A,
                                int64_t               lda,
                                const hipblasComplex* x,
                                int64_t               incx,
                                const hipblasComplex* beta,
                                hipblasComplex*       y,
                                int64_t               incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCsymv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               (cuComplex*)alpha,
                                               (cuComplex*)A,
                                               lda,
                                               (cuComplex*)x,
                                               incx,
                                               (cuComplex*)beta,
                                               (cuComplex*)y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsymv_64(hipblasHandle_t             handle,
                                hipblasFillMode_t           uplo,
                                int64_t                     n,
                                const hipblasDoubleComplex* alpha,
                                const hipblasDoubleComplex* A,
                                int64_t                     lda,
                                const hipblasDoubleComplex* x,
                                int64_t                     incx,
                                const hipblasDoubleComplex* beta,
                                hipblasDoubleComplex*       y,
                                int64_t                     incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZsymv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               (cuDoubleComplex*)alpha,
                                               (cuDoubleComplex*)A,
                                               lda,
                                               (cuDoubleComplex*)x,
                                               incx,
                                               (cuDoubleComplex*)beta,
                                               (cuDoubleComplex*)y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsymv_v2_64(hipblasHandle_t   handle,
                                   hipblasFillMode_t uplo,
                                   int64_t           n,
                                   const hipComplex* alpha,
                                   const hipComplex* A,
                                   int64_t           lda,
                                   const hipComplex* x,
                                   int64_t           incx,
                                   const hipComplex* beta,
                                   hipComplex*       y,
                                   int64_t           incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCsymv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               (cuComplex*)alpha,
                                               (cuComplex*)A,
                                               lda,
                                               (cuComplex*)x,
                                               incx,
                                               (cuComplex*)beta,
                                               (cuComplex*)y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsymv_v2_64(hipblasHandle_t         handle,
                                   hipblasFillMode_t       uplo,
                                   int64_t                 n,
                                   const hipDoubleComplex* alpha,
                                   const hipDoubleComplex* A,
                                   int64_t                 lda,
                                   const hipDoubleComplex* x,
                                   int64_t                 incx,
                                   const hipDoubleComplex* beta,
                                   hipDoubleComplex*       y,
                                   int64_t                 incy)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZsymv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               (cuDoubleComplex*)alpha,
                                               (cuDoubleComplex*)A,
                                               lda,
                                               (cuDoubleComplex*)x,
                                               incx,
                                               (cuDoubleComplex*)beta,
                                               (cuDoubleComplex*)y,
                                               incy));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// symv_batched
hipblasStatus_t hipblasSsymvBatched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    int                n,
                                    const float*       alpha,
                                    const float* const A[],
                                    int                lda,
                                    const float* const x[],
                                    int                incx,
                                    const float*       beta,
                                    float* const       y[],
                                    int                incy,
                                    int                batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDsymvBatched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    int                 n,
                                    const double*       alpha,
                                    const double* const A[],
                                    int                 lda,
                                    const double* const x[],
                                    int                 incx,
                                    const double*       beta,
                                    double* const       y[],
                                    int                 incy,
                                    int                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsymvBatched(hipblasHandle_t             handle,
                                    hipblasFillMode_t           uplo,
                                    int                         n,
                                    const hipblasComplex*       alpha,
                                    const hipblasComplex* const A[],
                                    int                         lda,
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    const hipblasComplex*       beta,
                                    hipblasComplex* const       y[],
                                    int                         incy,
                                    int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsymvBatched(hipblasHandle_t                   handle,
                                    hipblasFillMode_t                 uplo,
                                    int                               n,
                                    const hipblasDoubleComplex*       alpha,
                                    const hipblasDoubleComplex* const A[],
                                    int                               lda,
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    const hipblasDoubleComplex*       beta,
                                    hipblasDoubleComplex* const       y[],
                                    int                               incy,
                                    int                               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsymvBatched_v2(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       int                     n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const A[],
                                       int                     lda,
                                       const hipComplex* const x[],
                                       int                     incx,
                                       const hipComplex*       beta,
                                       hipComplex* const       y[],
                                       int                     incy,
                                       int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsymvBatched_v2(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       int                           n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const A[],
                                       int                           lda,
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       const hipDoubleComplex*       beta,
                                       hipDoubleComplex* const       y[],
                                       int                           incy,
                                       int                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// symv_batched_64
hipblasStatus_t hipblasSsymvBatched_64(hipblasHandle_t    handle,
                                       hipblasFillMode_t  uplo,
                                       int64_t            n,
                                       const float*       alpha,
                                       const float* const A[],
                                       int64_t            lda,
                                       const float* const x[],
                                       int64_t            incx,
                                       const float*       beta,
                                       float* const       y[],
                                       int64_t            incy,
                                       int64_t            batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDsymvBatched_64(hipblasHandle_t     handle,
                                       hipblasFillMode_t   uplo,
                                       int64_t             n,
                                       const double*       alpha,
                                       const double* const A[],
                                       int64_t             lda,
                                       const double* const x[],
                                       int64_t             incx,
                                       const double*       beta,
                                       double* const       y[],
                                       int64_t             incy,
                                       int64_t             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsymvBatched_64(hipblasHandle_t             handle,
                                       hipblasFillMode_t           uplo,
                                       int64_t                     n,
                                       const hipblasComplex*       alpha,
                                       const hipblasComplex* const A[],
                                       int64_t                     lda,
                                       const hipblasComplex* const x[],
                                       int64_t                     incx,
                                       const hipblasComplex*       beta,
                                       hipblasComplex* const       y[],
                                       int64_t                     incy,
                                       int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsymvBatched_64(hipblasHandle_t                   handle,
                                       hipblasFillMode_t                 uplo,
                                       int64_t                           n,
                                       const hipblasDoubleComplex*       alpha,
                                       const hipblasDoubleComplex* const A[],
                                       int64_t                           lda,
                                       const hipblasDoubleComplex* const x[],
                                       int64_t                           incx,
                                       const hipblasDoubleComplex*       beta,
                                       hipblasDoubleComplex* const       y[],
                                       int64_t                           incy,
                                       int64_t                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsymvBatched_v2_64(hipblasHandle_t         handle,
                                          hipblasFillMode_t       uplo,
                                          int64_t                 n,
                                          const hipComplex*       alpha,
                                          const hipComplex* const A[],
                                          int64_t                 lda,
                                          const hipComplex* const x[],
                                          int64_t                 incx,
                                          const hipComplex*       beta,
                                          hipComplex* const       y[],
                                          int64_t                 incy,
                                          int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsymvBatched_v2_64(hipblasHandle_t               handle,
                                          hipblasFillMode_t             uplo,
                                          int64_t                       n,
                                          const hipDoubleComplex*       alpha,
                                          const hipDoubleComplex* const A[],
                                          int64_t                       lda,
                                          const hipDoubleComplex* const x[],
                                          int64_t                       incx,
                                          const hipDoubleComplex*       beta,
                                          hipDoubleComplex* const       y[],
                                          int64_t                       incy,
                                          int64_t                       batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// symv_strided_batched
hipblasStatus_t hipblasSsymvStridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const float*      alpha,
                                           const float*      A,
                                           int               lda,
                                           hipblasStride     strideA,
                                           const float*      x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const float*      beta,
                                           float*            y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDsymvStridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const double*     alpha,
                                           const double*     A,
                                           int               lda,
                                           hipblasStride     strideA,
                                           const double*     x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const double*     beta,
                                           double*           y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsymvStridedBatched(hipblasHandle_t       handle,
                                           hipblasFillMode_t     uplo,
                                           int                   n,
                                           const hipblasComplex* alpha,
                                           const hipblasComplex* A,
                                           int                   lda,
                                           hipblasStride         strideA,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           const hipblasComplex* beta,
                                           hipblasComplex*       y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsymvStridedBatched(hipblasHandle_t             handle,
                                           hipblasFillMode_t           uplo,
                                           int                         n,
                                           const hipblasDoubleComplex* alpha,
                                           const hipblasDoubleComplex* A,
                                           int                         lda,
                                           hipblasStride               strideA,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           const hipblasDoubleComplex* beta,
                                           hipblasDoubleComplex*       y,
                                           int                         incy,
                                           hipblasStride               stridey,
                                           int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsymvStridedBatched_v2(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int               n,
                                              const hipComplex* alpha,
                                              const hipComplex* A,
                                              int               lda,
                                              hipblasStride     strideA,
                                              const hipComplex* x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              const hipComplex* beta,
                                              hipComplex*       y,
                                              int               incy,
                                              hipblasStride     stridey,
                                              int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsymvStridedBatched_v2(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              int                     n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* A,
                                              int                     lda,
                                              hipblasStride           strideA,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* beta,
                                              hipDoubleComplex*       y,
                                              int                     incy,
                                              hipblasStride           stridey,
                                              int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// symv_strided_batched_64
hipblasStatus_t hipblasSsymvStridedBatched_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              const float*      alpha,
                                              const float*      A,
                                              int64_t           lda,
                                              hipblasStride     strideA,
                                              const float*      x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const float*      beta,
                                              float*            y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDsymvStridedBatched_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              const double*     alpha,
                                              const double*     A,
                                              int64_t           lda,
                                              hipblasStride     strideA,
                                              const double*     x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const double*     beta,
                                              double*           y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsymvStridedBatched_64(hipblasHandle_t       handle,
                                              hipblasFillMode_t     uplo,
                                              int64_t               n,
                                              const hipblasComplex* alpha,
                                              const hipblasComplex* A,
                                              int64_t               lda,
                                              hipblasStride         strideA,
                                              const hipblasComplex* x,
                                              int64_t               incx,
                                              hipblasStride         stridex,
                                              const hipblasComplex* beta,
                                              hipblasComplex*       y,
                                              int64_t               incy,
                                              hipblasStride         stridey,
                                              int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsymvStridedBatched_64(hipblasHandle_t             handle,
                                              hipblasFillMode_t           uplo,
                                              int64_t                     n,
                                              const hipblasDoubleComplex* alpha,
                                              const hipblasDoubleComplex* A,
                                              int64_t                     lda,
                                              hipblasStride               strideA,
                                              const hipblasDoubleComplex* x,
                                              int64_t                     incx,
                                              hipblasStride               stridex,
                                              const hipblasDoubleComplex* beta,
                                              hipblasDoubleComplex*       y,
                                              int64_t                     incy,
                                              hipblasStride               stridey,
                                              int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsymvStridedBatched_v2_64(hipblasHandle_t   handle,
                                                 hipblasFillMode_t uplo,
                                                 int64_t           n,
                                                 const hipComplex* alpha,
                                                 const hipComplex* A,
                                                 int64_t           lda,
                                                 hipblasStride     strideA,
                                                 const hipComplex* x,
                                                 int64_t           incx,
                                                 hipblasStride     stridex,
                                                 const hipComplex* beta,
                                                 hipComplex*       y,
                                                 int64_t           incy,
                                                 hipblasStride     stridey,
                                                 int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsymvStridedBatched_v2_64(hipblasHandle_t         handle,
                                                 hipblasFillMode_t       uplo,
                                                 int64_t                 n,
                                                 const hipDoubleComplex* alpha,
                                                 const hipDoubleComplex* A,
                                                 int64_t                 lda,
                                                 hipblasStride           strideA,
                                                 const hipDoubleComplex* x,
                                                 int64_t                 incx,
                                                 hipblasStride           stridex,
                                                 const hipDoubleComplex* beta,
                                                 hipDoubleComplex*       y,
                                                 int64_t                 incy,
                                                 hipblasStride           stridey,
                                                 int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// syr
hipblasStatus_t hipblasSsyr(hipblasHandle_t   handle,
                            hipblasFillMode_t uplo,
                            int               n,
                            const float*      alpha,
                            const float*      x,
                            int               incx,
                            float*            A,
                            int               lda)
try
{
    return hipblasConvertStatus(
        cublasSsyr((cublasHandle_t)handle, hipblasConvertFill(uplo), n, alpha, x, incx, A, lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsyr(hipblasHandle_t   handle,
                            hipblasFillMode_t uplo,
                            int               n,
                            const double*     alpha,
                            const double*     x,
                            int               incx,
                            double*           A,
                            int               lda)
try
{
    return hipblasConvertStatus(
        cublasDsyr((cublasHandle_t)handle, hipblasConvertFill(uplo), n, alpha, x, incx, A, lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsyr(hipblasHandle_t       handle,
                            hipblasFillMode_t     uplo,
                            int                   n,
                            const hipblasComplex* alpha,
                            const hipblasComplex* x,
                            int                   incx,
                            hipblasComplex*       A,
                            int                   lda)
try
{
    return hipblasConvertStatus(cublasCsyr((cublasHandle_t)handle,
                                           hipblasConvertFill(uplo),
                                           n,
                                           (cuComplex*)alpha,
                                           (cuComplex*)x,
                                           incx,
                                           (cuComplex*)A,
                                           lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsyr(hipblasHandle_t             handle,
                            hipblasFillMode_t           uplo,
                            int                         n,
                            const hipblasDoubleComplex* alpha,
                            const hipblasDoubleComplex* x,
                            int                         incx,
                            hipblasDoubleComplex*       A,
                            int                         lda)
try
{
    return hipblasConvertStatus(cublasZsyr((cublasHandle_t)handle,
                                           hipblasConvertFill(uplo),
                                           n,
                                           (cuDoubleComplex*)alpha,
                                           (cuDoubleComplex*)x,
                                           incx,
                                           (cuDoubleComplex*)A,
                                           lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsyr_v2(hipblasHandle_t   handle,
                               hipblasFillMode_t uplo,
                               int               n,
                               const hipComplex* alpha,
                               const hipComplex* x,
                               int               incx,
                               hipComplex*       A,
                               int               lda)
try
{
    return hipblasConvertStatus(cublasCsyr((cublasHandle_t)handle,
                                           hipblasConvertFill(uplo),
                                           n,
                                           (cuComplex*)alpha,
                                           (cuComplex*)x,
                                           incx,
                                           (cuComplex*)A,
                                           lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsyr_v2(hipblasHandle_t         handle,
                               hipblasFillMode_t       uplo,
                               int                     n,
                               const hipDoubleComplex* alpha,
                               const hipDoubleComplex* x,
                               int                     incx,
                               hipDoubleComplex*       A,
                               int                     lda)
try
{
    return hipblasConvertStatus(cublasZsyr((cublasHandle_t)handle,
                                           hipblasConvertFill(uplo),
                                           n,
                                           (cuDoubleComplex*)alpha,
                                           (cuDoubleComplex*)x,
                                           incx,
                                           (cuDoubleComplex*)A,
                                           lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// syr_64
hipblasStatus_t hipblasSsyr_64(hipblasHandle_t   handle,
                               hipblasFillMode_t uplo,
                               int64_t           n,
                               const float*      alpha,
                               const float*      x,
                               int64_t           incx,
                               float*            A,
                               int64_t           lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasSsyr_64((cublasHandle_t)handle, hipblasConvertFill(uplo), n, alpha, x, incx, A, lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsyr_64(hipblasHandle_t   handle,
                               hipblasFillMode_t uplo,
                               int64_t           n,
                               const double*     alpha,
                               const double*     x,
                               int64_t           incx,
                               double*           A,
                               int64_t           lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(
        cublasDsyr_64((cublasHandle_t)handle, hipblasConvertFill(uplo), n, alpha, x, incx, A, lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsyr_64(hipblasHandle_t       handle,
                               hipblasFillMode_t     uplo,
                               int64_t               n,
                               const hipblasComplex* alpha,
                               const hipblasComplex* x,
                               int64_t               incx,
                               hipblasComplex*       A,
                               int64_t               lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCsyr_64((cublasHandle_t)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              (cuComplex*)alpha,
                                              (cuComplex*)x,
                                              incx,
                                              (cuComplex*)A,
                                              lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsyr_64(hipblasHandle_t             handle,
                               hipblasFillMode_t           uplo,
                               int64_t                     n,
                               const hipblasDoubleComplex* alpha,
                               const hipblasDoubleComplex* x,
                               int64_t                     incx,
                               hipblasDoubleComplex*       A,
                               int64_t                     lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZsyr_64((cublasHandle_t)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              (cuDoubleComplex*)alpha,
                                              (cuDoubleComplex*)x,
                                              incx,
                                              (cuDoubleComplex*)A,
                                              lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsyr_v2_64(hipblasHandle_t   handle,
                                  hipblasFillMode_t uplo,
                                  int64_t           n,
                                  const hipComplex* alpha,
                                  const hipComplex* x,
                                  int64_t           incx,
                                  hipComplex*       A,
                                  int64_t           lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCsyr_64((cublasHandle_t)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              (cuComplex*)alpha,
                                              (cuComplex*)x,
                                              incx,
                                              (cuComplex*)A,
                                              lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsyr_v2_64(hipblasHandle_t         handle,
                                  hipblasFillMode_t       uplo,
                                  int64_t                 n,
                                  const hipDoubleComplex* alpha,
                                  const hipDoubleComplex* x,
                                  int64_t                 incx,
                                  hipDoubleComplex*       A,
                                  int64_t                 lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZsyr_64((cublasHandle_t)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              (cuDoubleComplex*)alpha,
                                              (cuDoubleComplex*)x,
                                              incx,
                                              (cuDoubleComplex*)A,
                                              lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// syr_batched
hipblasStatus_t hipblasSsyrBatched(hipblasHandle_t    handle,
                                   hipblasFillMode_t  uplo,
                                   int                n,
                                   const float*       alpha,
                                   const float* const x[],
                                   int                incx,
                                   float* const       A[],
                                   int                lda,
                                   int                batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDsyrBatched(hipblasHandle_t     handle,
                                   hipblasFillMode_t   uplo,
                                   int                 n,
                                   const double*       alpha,
                                   const double* const x[],
                                   int                 incx,
                                   double* const       A[],
                                   int                 lda,
                                   int                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsyrBatched(hipblasHandle_t             handle,
                                   hipblasFillMode_t           uplo,
                                   int                         n,
                                   const hipblasComplex*       alpha,
                                   const hipblasComplex* const x[],
                                   int                         incx,
                                   hipblasComplex* const       A[],
                                   int                         lda,
                                   int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsyrBatched(hipblasHandle_t                   handle,
                                   hipblasFillMode_t                 uplo,
                                   int                               n,
                                   const hipblasDoubleComplex*       alpha,
                                   const hipblasDoubleComplex* const x[],
                                   int                               incx,
                                   hipblasDoubleComplex* const       A[],
                                   int                               lda,
                                   int                               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsyrBatched_v2(hipblasHandle_t         handle,
                                      hipblasFillMode_t       uplo,
                                      int                     n,
                                      const hipComplex*       alpha,
                                      const hipComplex* const x[],
                                      int                     incx,
                                      hipComplex* const       A[],
                                      int                     lda,
                                      int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsyrBatched_v2(hipblasHandle_t               handle,
                                      hipblasFillMode_t             uplo,
                                      int                           n,
                                      const hipDoubleComplex*       alpha,
                                      const hipDoubleComplex* const x[],
                                      int                           incx,
                                      hipDoubleComplex* const       A[],
                                      int                           lda,
                                      int                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// syr_batched_64
hipblasStatus_t hipblasSsyrBatched_64(hipblasHandle_t    handle,
                                      hipblasFillMode_t  uplo,
                                      int64_t            n,
                                      const float*       alpha,
                                      const float* const x[],
                                      int64_t            incx,
                                      float* const       A[],
                                      int64_t            lda,
                                      int64_t            batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDsyrBatched_64(hipblasHandle_t     handle,
                                      hipblasFillMode_t   uplo,
                                      int64_t             n,
                                      const double*       alpha,
                                      const double* const x[],
                                      int64_t             incx,
                                      double* const       A[],
                                      int64_t             lda,
                                      int64_t             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsyrBatched_64(hipblasHandle_t             handle,
                                      hipblasFillMode_t           uplo,
                                      int64_t                     n,
                                      const hipblasComplex*       alpha,
                                      const hipblasComplex* const x[],
                                      int64_t                     incx,
                                      hipblasComplex* const       A[],
                                      int64_t                     lda,
                                      int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsyrBatched_64(hipblasHandle_t                   handle,
                                      hipblasFillMode_t                 uplo,
                                      int64_t                           n,
                                      const hipblasDoubleComplex*       alpha,
                                      const hipblasDoubleComplex* const x[],
                                      int64_t                           incx,
                                      hipblasDoubleComplex* const       A[],
                                      int64_t                           lda,
                                      int64_t                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsyrBatched_v2_64(hipblasHandle_t         handle,
                                         hipblasFillMode_t       uplo,
                                         int64_t                 n,
                                         const hipComplex*       alpha,
                                         const hipComplex* const x[],
                                         int64_t                 incx,
                                         hipComplex* const       A[],
                                         int64_t                 lda,
                                         int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsyrBatched_v2_64(hipblasHandle_t               handle,
                                         hipblasFillMode_t             uplo,
                                         int64_t                       n,
                                         const hipDoubleComplex*       alpha,
                                         const hipDoubleComplex* const x[],
                                         int64_t                       incx,
                                         hipDoubleComplex* const       A[],
                                         int64_t                       lda,
                                         int64_t                       batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// syr_strided_batched
hipblasStatus_t hipblasSsyrStridedBatched(hipblasHandle_t   handle,
                                          hipblasFillMode_t uplo,
                                          int               n,
                                          const float*      alpha,
                                          const float*      x,
                                          int               incx,
                                          hipblasStride     stridex,
                                          float*            A,
                                          int               lda,
                                          hipblasStride     strideA,
                                          int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDsyrStridedBatched(hipblasHandle_t   handle,
                                          hipblasFillMode_t uplo,
                                          int               n,
                                          const double*     alpha,
                                          const double*     x,
                                          int               incx,
                                          hipblasStride     stridex,
                                          double*           A,
                                          int               lda,
                                          hipblasStride     strideA,
                                          int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsyrStridedBatched(hipblasHandle_t       handle,
                                          hipblasFillMode_t     uplo,
                                          int                   n,
                                          const hipblasComplex* alpha,
                                          const hipblasComplex* x,
                                          int                   incx,
                                          hipblasStride         stridex,
                                          hipblasComplex*       A,
                                          int                   lda,
                                          hipblasStride         strideA,
                                          int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsyrStridedBatched(hipblasHandle_t             handle,
                                          hipblasFillMode_t           uplo,
                                          int                         n,
                                          const hipblasDoubleComplex* alpha,
                                          const hipblasDoubleComplex* x,
                                          int                         incx,
                                          hipblasStride               stridex,
                                          hipblasDoubleComplex*       A,
                                          int                         lda,
                                          hipblasStride               strideA,
                                          int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsyrStridedBatched_v2(hipblasHandle_t   handle,
                                             hipblasFillMode_t uplo,
                                             int               n,
                                             const hipComplex* alpha,
                                             const hipComplex* x,
                                             int               incx,
                                             hipblasStride     stridex,
                                             hipComplex*       A,
                                             int               lda,
                                             hipblasStride     strideA,
                                             int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsyrStridedBatched_v2(hipblasHandle_t         handle,
                                             hipblasFillMode_t       uplo,
                                             int                     n,
                                             const hipDoubleComplex* alpha,
                                             const hipDoubleComplex* x,
                                             int                     incx,
                                             hipblasStride           stridex,
                                             hipDoubleComplex*       A,
                                             int                     lda,
                                             hipblasStride           strideA,
                                             int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// syr_strided_batched_64
hipblasStatus_t hipblasSsyrStridedBatched_64(hipblasHandle_t   handle,
                                             hipblasFillMode_t uplo,
                                             int64_t           n,
                                             const float*      alpha,
                                             const float*      x,
                                             int64_t           incx,
                                             hipblasStride     stridex,
                                             float*            A,
                                             int64_t           lda,
                                             hipblasStride     strideA,
                                             int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDsyrStridedBatched_64(hipblasHandle_t   handle,
                                             hipblasFillMode_t uplo,
                                             int64_t           n,
                                             const double*     alpha,
                                             const double*     x,
                                             int64_t           incx,
                                             hipblasStride     stridex,
                                             double*           A,
                                             int64_t           lda,
                                             hipblasStride     strideA,
                                             int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsyrStridedBatched_64(hipblasHandle_t       handle,
                                             hipblasFillMode_t     uplo,
                                             int64_t               n,
                                             const hipblasComplex* alpha,
                                             const hipblasComplex* x,
                                             int64_t               incx,
                                             hipblasStride         stridex,
                                             hipblasComplex*       A,
                                             int64_t               lda,
                                             hipblasStride         strideA,
                                             int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsyrStridedBatched_64(hipblasHandle_t             handle,
                                             hipblasFillMode_t           uplo,
                                             int64_t                     n,
                                             const hipblasDoubleComplex* alpha,
                                             const hipblasDoubleComplex* x,
                                             int64_t                     incx,
                                             hipblasStride               stridex,
                                             hipblasDoubleComplex*       A,
                                             int64_t                     lda,
                                             hipblasStride               strideA,
                                             int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsyrStridedBatched_v2_64(hipblasHandle_t   handle,
                                                hipblasFillMode_t uplo,
                                                int64_t           n,
                                                const hipComplex* alpha,
                                                const hipComplex* x,
                                                int64_t           incx,
                                                hipblasStride     stridex,
                                                hipComplex*       A,
                                                int64_t           lda,
                                                hipblasStride     strideA,
                                                int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsyrStridedBatched_v2_64(hipblasHandle_t         handle,
                                                hipblasFillMode_t       uplo,
                                                int64_t                 n,
                                                const hipDoubleComplex* alpha,
                                                const hipDoubleComplex* x,
                                                int64_t                 incx,
                                                hipblasStride           stridex,
                                                hipDoubleComplex*       A,
                                                int64_t                 lda,
                                                hipblasStride           strideA,
                                                int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// syr2
hipblasStatus_t hipblasSsyr2(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const float*      alpha,
                             const float*      x,
                             int               incx,
                             const float*      y,
                             int               incy,
                             float*            A,
                             int               lda)
try
{
    return hipblasConvertStatus(cublasSsyr2(
        (cublasHandle_t)handle, hipblasConvertFill(uplo), n, alpha, x, incx, y, incy, A, lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsyr2(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const double*     alpha,
                             const double*     x,
                             int               incx,
                             const double*     y,
                             int               incy,
                             double*           A,
                             int               lda)
try
{
    return hipblasConvertStatus(cublasDsyr2(
        (cublasHandle_t)handle, hipblasConvertFill(uplo), n, alpha, x, incx, y, incy, A, lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsyr2(hipblasHandle_t       handle,
                             hipblasFillMode_t     uplo,
                             int                   n,
                             const hipblasComplex* alpha,
                             const hipblasComplex* x,
                             int                   incx,
                             const hipblasComplex* y,
                             int                   incy,
                             hipblasComplex*       A,
                             int                   lda)
try
{
    return hipblasConvertStatus(cublasCsyr2((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            (cuComplex*)alpha,
                                            (cuComplex*)x,
                                            incx,
                                            (cuComplex*)y,
                                            incy,
                                            (cuComplex*)A,
                                            lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsyr2(hipblasHandle_t             handle,
                             hipblasFillMode_t           uplo,
                             int                         n,
                             const hipblasDoubleComplex* alpha,
                             const hipblasDoubleComplex* x,
                             int                         incx,
                             const hipblasDoubleComplex* y,
                             int                         incy,
                             hipblasDoubleComplex*       A,
                             int                         lda)
try
{
    return hipblasConvertStatus(cublasZsyr2((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            (cuDoubleComplex*)alpha,
                                            (cuDoubleComplex*)x,
                                            incx,
                                            (cuDoubleComplex*)y,
                                            incy,
                                            (cuDoubleComplex*)A,
                                            lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsyr2_v2(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int               n,
                                const hipComplex* alpha,
                                const hipComplex* x,
                                int               incx,
                                const hipComplex* y,
                                int               incy,
                                hipComplex*       A,
                                int               lda)
try
{
    return hipblasConvertStatus(cublasCsyr2((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            (cuComplex*)alpha,
                                            (cuComplex*)x,
                                            incx,
                                            (cuComplex*)y,
                                            incy,
                                            (cuComplex*)A,
                                            lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsyr2_v2(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                int                     n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* x,
                                int                     incx,
                                const hipDoubleComplex* y,
                                int                     incy,
                                hipDoubleComplex*       A,
                                int                     lda)
try
{
    return hipblasConvertStatus(cublasZsyr2((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            n,
                                            (cuDoubleComplex*)alpha,
                                            (cuDoubleComplex*)x,
                                            incx,
                                            (cuDoubleComplex*)y,
                                            incy,
                                            (cuDoubleComplex*)A,
                                            lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// syr2_64
hipblasStatus_t hipblasSsyr2_64(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int64_t           n,
                                const float*      alpha,
                                const float*      x,
                                int64_t           incx,
                                const float*      y,
                                int64_t           incy,
                                float*            A,
                                int64_t           lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasSsyr2_64(
        (cublasHandle_t)handle, hipblasConvertFill(uplo), n, alpha, x, incx, y, incy, A, lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsyr2_64(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int64_t           n,
                                const double*     alpha,
                                const double*     x,
                                int64_t           incx,
                                const double*     y,
                                int64_t           incy,
                                double*           A,
                                int64_t           lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasDsyr2_64(
        (cublasHandle_t)handle, hipblasConvertFill(uplo), n, alpha, x, incx, y, incy, A, lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsyr2_64(hipblasHandle_t       handle,
                                hipblasFillMode_t     uplo,
                                int64_t               n,
                                const hipblasComplex* alpha,
                                const hipblasComplex* x,
                                int64_t               incx,
                                const hipblasComplex* y,
                                int64_t               incy,
                                hipblasComplex*       A,
                                int64_t               lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCsyr2_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               (cuComplex*)alpha,
                                               (cuComplex*)x,
                                               incx,
                                               (cuComplex*)y,
                                               incy,
                                               (cuComplex*)A,
                                               lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsyr2_64(hipblasHandle_t             handle,
                                hipblasFillMode_t           uplo,
                                int64_t                     n,
                                const hipblasDoubleComplex* alpha,
                                const hipblasDoubleComplex* x,
                                int64_t                     incx,
                                const hipblasDoubleComplex* y,
                                int64_t                     incy,
                                hipblasDoubleComplex*       A,
                                int64_t                     lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZsyr2_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               (cuDoubleComplex*)alpha,
                                               (cuDoubleComplex*)x,
                                               incx,
                                               (cuDoubleComplex*)y,
                                               incy,
                                               (cuDoubleComplex*)A,
                                               lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsyr2_v2_64(hipblasHandle_t   handle,
                                   hipblasFillMode_t uplo,
                                   int64_t           n,
                                   const hipComplex* alpha,
                                   const hipComplex* x,
                                   int64_t           incx,
                                   const hipComplex* y,
                                   int64_t           incy,
                                   hipComplex*       A,
                                   int64_t           lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCsyr2_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               (cuComplex*)alpha,
                                               (cuComplex*)x,
                                               incx,
                                               (cuComplex*)y,
                                               incy,
                                               (cuComplex*)A,
                                               lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsyr2_v2_64(hipblasHandle_t         handle,
                                   hipblasFillMode_t       uplo,
                                   int64_t                 n,
                                   const hipDoubleComplex* alpha,
                                   const hipDoubleComplex* x,
                                   int64_t                 incx,
                                   const hipDoubleComplex* y,
                                   int64_t                 incy,
                                   hipDoubleComplex*       A,
                                   int64_t                 lda)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZsyr2_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               n,
                                               (cuDoubleComplex*)alpha,
                                               (cuDoubleComplex*)x,
                                               incx,
                                               (cuDoubleComplex*)y,
                                               incy,
                                               (cuDoubleComplex*)A,
                                               lda));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// syr2_batched
hipblasStatus_t hipblasSsyr2Batched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    int                n,
                                    const float*       alpha,
                                    const float* const x[],
                                    int                incx,
                                    const float* const y[],
                                    int                incy,
                                    float* const       A[],
                                    int                lda,
                                    int                batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDsyr2Batched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    int                 n,
                                    const double*       alpha,
                                    const double* const x[],
                                    int                 incx,
                                    const double* const y[],
                                    int                 incy,
                                    double* const       A[],
                                    int                 lda,
                                    int                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsyr2Batched(hipblasHandle_t             handle,
                                    hipblasFillMode_t           uplo,
                                    int                         n,
                                    const hipblasComplex*       alpha,
                                    const hipblasComplex* const x[],
                                    int                         incx,
                                    const hipblasComplex* const y[],
                                    int                         incy,
                                    hipblasComplex* const       A[],
                                    int                         lda,
                                    int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsyr2Batched(hipblasHandle_t                   handle,
                                    hipblasFillMode_t                 uplo,
                                    int                               n,
                                    const hipblasDoubleComplex*       alpha,
                                    const hipblasDoubleComplex* const x[],
                                    int                               incx,
                                    const hipblasDoubleComplex* const y[],
                                    int                               incy,
                                    hipblasDoubleComplex* const       A[],
                                    int                               lda,
                                    int                               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsyr2Batched_v2(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       int                     n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const x[],
                                       int                     incx,
                                       const hipComplex* const y[],
                                       int                     incy,
                                       hipComplex* const       A[],
                                       int                     lda,
                                       int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsyr2Batched_v2(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       int                           n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const x[],
                                       int                           incx,
                                       const hipDoubleComplex* const y[],
                                       int                           incy,
                                       hipDoubleComplex* const       A[],
                                       int                           lda,
                                       int                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// syr2_batched_64
hipblasStatus_t hipblasSsyr2Batched_64(hipblasHandle_t    handle,
                                       hipblasFillMode_t  uplo,
                                       int64_t            n,
                                       const float*       alpha,
                                       const float* const x[],
                                       int64_t            incx,
                                       const float* const y[],
                                       int64_t            incy,
                                       float* const       A[],
                                       int64_t            lda,
                                       int64_t            batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDsyr2Batched_64(hipblasHandle_t     handle,
                                       hipblasFillMode_t   uplo,
                                       int64_t             n,
                                       const double*       alpha,
                                       const double* const x[],
                                       int64_t             incx,
                                       const double* const y[],
                                       int64_t             incy,
                                       double* const       A[],
                                       int64_t             lda,
                                       int64_t             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsyr2Batched_64(hipblasHandle_t             handle,
                                       hipblasFillMode_t           uplo,
                                       int64_t                     n,
                                       const hipblasComplex*       alpha,
                                       const hipblasComplex* const x[],
                                       int64_t                     incx,
                                       const hipblasComplex* const y[],
                                       int64_t                     incy,
                                       hipblasComplex* const       A[],
                                       int64_t                     lda,
                                       int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsyr2Batched_64(hipblasHandle_t                   handle,
                                       hipblasFillMode_t                 uplo,
                                       int64_t                           n,
                                       const hipblasDoubleComplex*       alpha,
                                       const hipblasDoubleComplex* const x[],
                                       int64_t                           incx,
                                       const hipblasDoubleComplex* const y[],
                                       int64_t                           incy,
                                       hipblasDoubleComplex* const       A[],
                                       int64_t                           lda,
                                       int64_t                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsyr2Batched_v2_64(hipblasHandle_t         handle,
                                          hipblasFillMode_t       uplo,
                                          int64_t                 n,
                                          const hipComplex*       alpha,
                                          const hipComplex* const x[],
                                          int64_t                 incx,
                                          const hipComplex* const y[],
                                          int64_t                 incy,
                                          hipComplex* const       A[],
                                          int64_t                 lda,
                                          int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsyr2Batched_v2_64(hipblasHandle_t               handle,
                                          hipblasFillMode_t             uplo,
                                          int64_t                       n,
                                          const hipDoubleComplex*       alpha,
                                          const hipDoubleComplex* const x[],
                                          int64_t                       incx,
                                          const hipDoubleComplex* const y[],
                                          int64_t                       incy,
                                          hipDoubleComplex* const       A[],
                                          int64_t                       lda,
                                          int64_t                       batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// syr2_strided_batched
hipblasStatus_t hipblasSsyr2StridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const float*      alpha,
                                           const float*      x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const float*      y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           float*            A,
                                           int               lda,
                                           hipblasStride     strideA,
                                           int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDsyr2StridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const double*     alpha,
                                           const double*     x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const double*     y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           double*           A,
                                           int               lda,
                                           hipblasStride     strideA,
                                           int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsyr2StridedBatched(hipblasHandle_t       handle,
                                           hipblasFillMode_t     uplo,
                                           int                   n,
                                           const hipblasComplex* alpha,
                                           const hipblasComplex* x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           const hipblasComplex* y,
                                           int                   incy,
                                           hipblasStride         stridey,
                                           hipblasComplex*       A,
                                           int                   lda,
                                           hipblasStride         strideA,
                                           int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsyr2StridedBatched(hipblasHandle_t             handle,
                                           hipblasFillMode_t           uplo,
                                           int                         n,
                                           const hipblasDoubleComplex* alpha,
                                           const hipblasDoubleComplex* x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           const hipblasDoubleComplex* y,
                                           int                         incy,
                                           hipblasStride               stridey,
                                           hipblasDoubleComplex*       A,
                                           int                         lda,
                                           hipblasStride               strideA,
                                           int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsyr2StridedBatched_v2(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int               n,
                                              const hipComplex* alpha,
                                              const hipComplex* x,
                                              int               incx,
                                              hipblasStride     stridex,
                                              const hipComplex* y,
                                              int               incy,
                                              hipblasStride     stridey,
                                              hipComplex*       A,
                                              int               lda,
                                              hipblasStride     strideA,
                                              int               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsyr2StridedBatched_v2(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              int                     n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* y,
                                              int                     incy,
                                              hipblasStride           stridey,
                                              hipDoubleComplex*       A,
                                              int                     lda,
                                              hipblasStride           strideA,
                                              int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// syr2_strided_batched_64
hipblasStatus_t hipblasSsyr2StridedBatched_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              const float*      alpha,
                                              const float*      x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const float*      y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              float*            A,
                                              int64_t           lda,
                                              hipblasStride     strideA,
                                              int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDsyr2StridedBatched_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              const double*     alpha,
                                              const double*     x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const double*     y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              double*           A,
                                              int64_t           lda,
                                              hipblasStride     strideA,
                                              int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsyr2StridedBatched_64(hipblasHandle_t       handle,
                                              hipblasFillMode_t     uplo,
                                              int64_t               n,
                                              const hipblasComplex* alpha,
                                              const hipblasComplex* x,
                                              int64_t               incx,
                                              hipblasStride         stridex,
                                              const hipblasComplex* y,
                                              int64_t               incy,
                                              hipblasStride         stridey,
                                              hipblasComplex*       A,
                                              int64_t               lda,
                                              hipblasStride         strideA,
                                              int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsyr2StridedBatched_64(hipblasHandle_t             handle,
                                              hipblasFillMode_t           uplo,
                                              int64_t                     n,
                                              const hipblasDoubleComplex* alpha,
                                              const hipblasDoubleComplex* x,
                                              int64_t                     incx,
                                              hipblasStride               stridex,
                                              const hipblasDoubleComplex* y,
                                              int64_t                     incy,
                                              hipblasStride               stridey,
                                              hipblasDoubleComplex*       A,
                                              int64_t                     lda,
                                              hipblasStride               strideA,
                                              int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCsyr2StridedBatched_v2_64(hipblasHandle_t   handle,
                                                 hipblasFillMode_t uplo,
                                                 int64_t           n,
                                                 const hipComplex* alpha,
                                                 const hipComplex* x,
                                                 int64_t           incx,
                                                 hipblasStride     stridex,
                                                 const hipComplex* y,
                                                 int64_t           incy,
                                                 hipblasStride     stridey,
                                                 hipComplex*       A,
                                                 int64_t           lda,
                                                 hipblasStride     strideA,
                                                 int64_t           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZsyr2StridedBatched_v2_64(hipblasHandle_t         handle,
                                                 hipblasFillMode_t       uplo,
                                                 int64_t                 n,
                                                 const hipDoubleComplex* alpha,
                                                 const hipDoubleComplex* x,
                                                 int64_t                 incx,
                                                 hipblasStride           stridex,
                                                 const hipDoubleComplex* y,
                                                 int64_t                 incy,
                                                 hipblasStride           stridey,
                                                 hipDoubleComplex*       A,
                                                 int64_t                 lda,
                                                 hipblasStride           strideA,
                                                 int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// tbmv
hipblasStatus_t hipblasStbmv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             int                k,
                             const float*       A,
                             int                lda,
                             float*             x,
                             int                incx)
try
{
    return hipblasConvertStatus(cublasStbmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            k,
                                            A,
                                            lda,
                                            x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtbmv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             int                k,
                             const double*      A,
                             int                lda,
                             double*            x,
                             int                incx)
try
{
    return hipblasConvertStatus(cublasDtbmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            k,
                                            A,
                                            lda,
                                            x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtbmv(hipblasHandle_t       handle,
                             hipblasFillMode_t     uplo,
                             hipblasOperation_t    transA,
                             hipblasDiagType_t     diag,
                             int                   n,
                             int                   k,
                             const hipblasComplex* A,
                             int                   lda,
                             hipblasComplex*       x,
                             int                   incx)
try
{
    return hipblasConvertStatus(cublasCtbmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            k,
                                            (cuComplex*)A,
                                            lda,
                                            (cuComplex*)x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtbmv(hipblasHandle_t             handle,
                             hipblasFillMode_t           uplo,
                             hipblasOperation_t          transA,
                             hipblasDiagType_t           diag,
                             int                         n,
                             int                         k,
                             const hipblasDoubleComplex* A,
                             int                         lda,
                             hipblasDoubleComplex*       x,
                             int                         incx)
try
{
    return hipblasConvertStatus(cublasZtbmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            k,
                                            (cuDoubleComplex*)A,
                                            lda,
                                            (cuDoubleComplex*)x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtbmv_v2(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int                n,
                                int                k,
                                const hipComplex*  A,
                                int                lda,
                                hipComplex*        x,
                                int                incx)
try
{
    return hipblasConvertStatus(cublasCtbmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            k,
                                            (cuComplex*)A,
                                            lda,
                                            (cuComplex*)x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtbmv_v2(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                hipblasOperation_t      transA,
                                hipblasDiagType_t       diag,
                                int                     n,
                                int                     k,
                                const hipDoubleComplex* A,
                                int                     lda,
                                hipDoubleComplex*       x,
                                int                     incx)
try
{
    return hipblasConvertStatus(cublasZtbmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            k,
                                            (cuDoubleComplex*)A,
                                            lda,
                                            (cuDoubleComplex*)x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tbmv_64
hipblasStatus_t hipblasStbmv_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int64_t            n,
                                int64_t            k,
                                const float*       A,
                                int64_t            lda,
                                float*             x,
                                int64_t            incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasStbmv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               k,
                                               A,
                                               lda,
                                               x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtbmv_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int64_t            n,
                                int64_t            k,
                                const double*      A,
                                int64_t            lda,
                                double*            x,
                                int64_t            incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasDtbmv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               k,
                                               A,
                                               lda,
                                               x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtbmv_64(hipblasHandle_t       handle,
                                hipblasFillMode_t     uplo,
                                hipblasOperation_t    transA,
                                hipblasDiagType_t     diag,
                                int64_t               n,
                                int64_t               k,
                                const hipblasComplex* A,
                                int64_t               lda,
                                hipblasComplex*       x,
                                int64_t               incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCtbmv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               k,
                                               (cuComplex*)A,
                                               lda,
                                               (cuComplex*)x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtbmv_64(hipblasHandle_t             handle,
                                hipblasFillMode_t           uplo,
                                hipblasOperation_t          transA,
                                hipblasDiagType_t           diag,
                                int64_t                     n,
                                int64_t                     k,
                                const hipblasDoubleComplex* A,
                                int64_t                     lda,
                                hipblasDoubleComplex*       x,
                                int64_t                     incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZtbmv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               k,
                                               (cuDoubleComplex*)A,
                                               lda,
                                               (cuDoubleComplex*)x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtbmv_v2_64(hipblasHandle_t    handle,
                                   hipblasFillMode_t  uplo,
                                   hipblasOperation_t transA,
                                   hipblasDiagType_t  diag,
                                   int64_t            n,
                                   int64_t            k,
                                   const hipComplex*  A,
                                   int64_t            lda,
                                   hipComplex*        x,
                                   int64_t            incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCtbmv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               k,
                                               (cuComplex*)A,
                                               lda,
                                               (cuComplex*)x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtbmv_v2_64(hipblasHandle_t         handle,
                                   hipblasFillMode_t       uplo,
                                   hipblasOperation_t      transA,
                                   hipblasDiagType_t       diag,
                                   int64_t                 n,
                                   int64_t                 k,
                                   const hipDoubleComplex* A,
                                   int64_t                 lda,
                                   hipDoubleComplex*       x,
                                   int64_t                 incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZtbmv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               k,
                                               (cuDoubleComplex*)A,
                                               lda,
                                               (cuDoubleComplex*)x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tbmv_batched
hipblasStatus_t hipblasStbmvBatched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    hipblasOperation_t transA,
                                    hipblasDiagType_t  diag,
                                    int                n,
                                    int                k,
                                    const float* const A[],
                                    int                lda,
                                    float* const       x[],
                                    int                incx,
                                    int                batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDtbmvBatched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    hipblasOperation_t  transA,
                                    hipblasDiagType_t   diag,
                                    int                 n,
                                    int                 k,
                                    const double* const A[],
                                    int                 lda,
                                    double* const       x[],
                                    int                 incx,
                                    int                 batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtbmvBatched(hipblasHandle_t             handle,
                                    hipblasFillMode_t           uplo,
                                    hipblasOperation_t          transA,
                                    hipblasDiagType_t           diag,
                                    int                         n,
                                    int                         k,
                                    const hipblasComplex* const A[],
                                    int                         lda,
                                    hipblasComplex* const       x[],
                                    int                         incx,
                                    int                         batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtbmvBatched(hipblasHandle_t                   handle,
                                    hipblasFillMode_t                 uplo,
                                    hipblasOperation_t                transA,
                                    hipblasDiagType_t                 diag,
                                    int                               n,
                                    int                               k,
                                    const hipblasDoubleComplex* const A[],
                                    int                               lda,
                                    hipblasDoubleComplex* const       x[],
                                    int                               incx,
                                    int                               batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtbmvBatched_v2(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       hipblasOperation_t      transA,
                                       hipblasDiagType_t       diag,
                                       int                     n,
                                       int                     k,
                                       const hipComplex* const A[],
                                       int                     lda,
                                       hipComplex* const       x[],
                                       int                     incx,
                                       int                     batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtbmvBatched_v2(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       hipblasOperation_t            transA,
                                       hipblasDiagType_t             diag,
                                       int                           n,
                                       int                           k,
                                       const hipDoubleComplex* const A[],
                                       int                           lda,
                                       hipDoubleComplex* const       x[],
                                       int                           incx,
                                       int                           batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// tbmv_batched_64
hipblasStatus_t hipblasStbmvBatched_64(hipblasHandle_t    handle,
                                       hipblasFillMode_t  uplo,
                                       hipblasOperation_t transA,
                                       hipblasDiagType_t  diag,
                                       int64_t            n,
                                       int64_t            k,
                                       const float* const A[],
                                       int64_t            lda,
                                       float* const       x[],
                                       int64_t            incx,
                                       int64_t            batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDtbmvBatched_64(hipblasHandle_t     handle,
                                       hipblasFillMode_t   uplo,
                                       hipblasOperation_t  transA,
                                       hipblasDiagType_t   diag,
                                       int64_t             n,
                                       int64_t             k,
                                       const double* const A[],
                                       int64_t             lda,
                                       double* const       x[],
                                       int64_t             incx,
                                       int64_t             batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtbmvBatched_64(hipblasHandle_t             handle,
                                       hipblasFillMode_t           uplo,
                                       hipblasOperation_t          transA,
                                       hipblasDiagType_t           diag,
                                       int64_t                     n,
                                       int64_t                     k,
                                       const hipblasComplex* const A[],
                                       int64_t                     lda,
                                       hipblasComplex* const       x[],
                                       int64_t                     incx,
                                       int64_t                     batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtbmvBatched_64(hipblasHandle_t                   handle,
                                       hipblasFillMode_t                 uplo,
                                       hipblasOperation_t                transA,
                                       hipblasDiagType_t                 diag,
                                       int64_t                           n,
                                       int64_t                           k,
                                       const hipblasDoubleComplex* const A[],
                                       int64_t                           lda,
                                       hipblasDoubleComplex* const       x[],
                                       int64_t                           incx,
                                       int64_t                           batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtbmvBatched_v2_64(hipblasHandle_t         handle,
                                          hipblasFillMode_t       uplo,
                                          hipblasOperation_t      transA,
                                          hipblasDiagType_t       diag,
                                          int64_t                 n,
                                          int64_t                 k,
                                          const hipComplex* const A[],
                                          int64_t                 lda,
                                          hipComplex* const       x[],
                                          int64_t                 incx,
                                          int64_t                 batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtbmvBatched_v2_64(hipblasHandle_t               handle,
                                          hipblasFillMode_t             uplo,
                                          hipblasOperation_t            transA,
                                          hipblasDiagType_t             diag,
                                          int64_t                       n,
                                          int64_t                       k,
                                          const hipDoubleComplex* const A[],
                                          int64_t                       lda,
                                          hipDoubleComplex* const       x[],
                                          int64_t                       incx,
                                          int64_t                       batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// tbmv_strided_batched
hipblasStatus_t hipblasStbmvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           int                k,
                                           const float*       A,
                                           int                lda,
                                           hipblasStride      stride_a,
                                           float*             x,
                                           int                incx,
                                           hipblasStride      stride_x,
                                           int                batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDtbmvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           int                k,
                                           const double*      A,
                                           int                lda,
                                           hipblasStride      stride_a,
                                           double*            x,
                                           int                incx,
                                           hipblasStride      stride_x,
                                           int                batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtbmvStridedBatched(hipblasHandle_t       handle,
                                           hipblasFillMode_t     uplo,
                                           hipblasOperation_t    transA,
                                           hipblasDiagType_t     diag,
                                           int                   n,
                                           int                   k,
                                           const hipblasComplex* A,
                                           int                   lda,
                                           hipblasStride         stride_a,
                                           hipblasComplex*       x,
                                           int                   incx,
                                           hipblasStride         stride_x,
                                           int                   batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtbmvStridedBatched(hipblasHandle_t             handle,
                                           hipblasFillMode_t           uplo,
                                           hipblasOperation_t          transA,
                                           hipblasDiagType_t           diag,
                                           int                         n,
                                           int                         k,
                                           const hipblasDoubleComplex* A,
                                           int                         lda,
                                           hipblasStride               stride_a,
                                           hipblasDoubleComplex*       x,
                                           int                         incx,
                                           hipblasStride               stride_x,
                                           int                         batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtbmvStridedBatched_v2(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int                n,
                                              int                k,
                                              const hipComplex*  A,
                                              int                lda,
                                              hipblasStride      stride_a,
                                              hipComplex*        x,
                                              int                incx,
                                              hipblasStride      stride_x,
                                              int                batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtbmvStridedBatched_v2(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              hipblasOperation_t      transA,
                                              hipblasDiagType_t       diag,
                                              int                     n,
                                              int                     k,
                                              const hipDoubleComplex* A,
                                              int                     lda,
                                              hipblasStride           stride_a,
                                              hipDoubleComplex*       x,
                                              int                     incx,
                                              hipblasStride           stride_x,
                                              int                     batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// tbmv_strided_batched_64
hipblasStatus_t hipblasStbmvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int64_t            n,
                                              int64_t            k,
                                              const float*       A,
                                              int64_t            lda,
                                              hipblasStride      stride_a,
                                              float*             x,
                                              int64_t            incx,
                                              hipblasStride      stride_x,
                                              int64_t            batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDtbmvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int64_t            n,
                                              int64_t            k,
                                              const double*      A,
                                              int64_t            lda,
                                              hipblasStride      stride_a,
                                              double*            x,
                                              int64_t            incx,
                                              hipblasStride      stride_x,
                                              int64_t            batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtbmvStridedBatched_64(hipblasHandle_t       handle,
                                              hipblasFillMode_t     uplo,
                                              hipblasOperation_t    transA,
                                              hipblasDiagType_t     diag,
                                              int64_t               n,
                                              int64_t               k,
                                              const hipblasComplex* A,
                                              int64_t               lda,
                                              hipblasStride         stride_a,
                                              hipblasComplex*       x,
                                              int64_t               incx,
                                              hipblasStride         stride_x,
                                              int64_t               batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtbmvStridedBatched_64(hipblasHandle_t             handle,
                                              hipblasFillMode_t           uplo,
                                              hipblasOperation_t          transA,
                                              hipblasDiagType_t           diag,
                                              int64_t                     n,
                                              int64_t                     k,
                                              const hipblasDoubleComplex* A,
                                              int64_t                     lda,
                                              hipblasStride               stride_a,
                                              hipblasDoubleComplex*       x,
                                              int64_t                     incx,
                                              hipblasStride               stride_x,
                                              int64_t                     batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtbmvStridedBatched_v2_64(hipblasHandle_t    handle,
                                                 hipblasFillMode_t  uplo,
                                                 hipblasOperation_t transA,
                                                 hipblasDiagType_t  diag,
                                                 int64_t            n,
                                                 int64_t            k,
                                                 const hipComplex*  A,
                                                 int64_t            lda,
                                                 hipblasStride      stride_a,
                                                 hipComplex*        x,
                                                 int64_t            incx,
                                                 hipblasStride      stride_x,
                                                 int64_t            batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtbmvStridedBatched_v2_64(hipblasHandle_t         handle,
                                                 hipblasFillMode_t       uplo,
                                                 hipblasOperation_t      transA,
                                                 hipblasDiagType_t       diag,
                                                 int64_t                 n,
                                                 int64_t                 k,
                                                 const hipDoubleComplex* A,
                                                 int64_t                 lda,
                                                 hipblasStride           stride_a,
                                                 hipDoubleComplex*       x,
                                                 int64_t                 incx,
                                                 hipblasStride           stride_x,
                                                 int64_t                 batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// tbsv
hipblasStatus_t hipblasStbsv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             int                k,
                             const float*       A,
                             int                lda,
                             float*             x,
                             int                incx)
try
{
    return hipblasConvertStatus(cublasStbsv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            k,
                                            A,
                                            lda,
                                            x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtbsv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             int                k,
                             const double*      A,
                             int                lda,
                             double*            x,
                             int                incx)
try
{
    return hipblasConvertStatus(cublasDtbsv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            k,
                                            A,
                                            lda,
                                            x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtbsv(hipblasHandle_t       handle,
                             hipblasFillMode_t     uplo,
                             hipblasOperation_t    transA,
                             hipblasDiagType_t     diag,
                             int                   n,
                             int                   k,
                             const hipblasComplex* A,
                             int                   lda,
                             hipblasComplex*       x,
                             int                   incx)
try
{
    return hipblasConvertStatus(cublasCtbsv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            k,
                                            (cuComplex*)A,
                                            lda,
                                            (cuComplex*)x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtbsv(hipblasHandle_t             handle,
                             hipblasFillMode_t           uplo,
                             hipblasOperation_t          transA,
                             hipblasDiagType_t           diag,
                             int                         n,
                             int                         k,
                             const hipblasDoubleComplex* A,
                             int                         lda,
                             hipblasDoubleComplex*       x,
                             int                         incx)
try
{
    return hipblasConvertStatus(cublasZtbsv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            k,
                                            (cuDoubleComplex*)A,
                                            lda,
                                            (cuDoubleComplex*)x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtbsv_v2(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int                n,
                                int                k,
                                const hipComplex*  A,
                                int                lda,
                                hipComplex*        x,
                                int                incx)
try
{
    return hipblasConvertStatus(cublasCtbsv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            k,
                                            (cuComplex*)A,
                                            lda,
                                            (cuComplex*)x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtbsv_v2(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                hipblasOperation_t      transA,
                                hipblasDiagType_t       diag,
                                int                     n,
                                int                     k,
                                const hipDoubleComplex* A,
                                int                     lda,
                                hipDoubleComplex*       x,
                                int                     incx)
try
{
    return hipblasConvertStatus(cublasZtbsv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            k,
                                            (cuDoubleComplex*)A,
                                            lda,
                                            (cuDoubleComplex*)x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tbsv_64
hipblasStatus_t hipblasStbsv_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int64_t            n,
                                int64_t            k,
                                const float*       A,
                                int64_t            lda,
                                float*             x,
                                int64_t            incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasStbsv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               k,
                                               A,
                                               lda,
                                               x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtbsv_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int64_t            n,
                                int64_t            k,
                                const double*      A,
                                int64_t            lda,
                                double*            x,
                                int64_t            incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasDtbsv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               k,
                                               A,
                                               lda,
                                               x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtbsv_64(hipblasHandle_t       handle,
                                hipblasFillMode_t     uplo,
                                hipblasOperation_t    transA,
                                hipblasDiagType_t     diag,
                                int64_t               n,
                                int64_t               k,
                                const hipblasComplex* A,
                                int64_t               lda,
                                hipblasComplex*       x,
                                int64_t               incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCtbsv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               k,
                                               (cuComplex*)A,
                                               lda,
                                               (cuComplex*)x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtbsv_64(hipblasHandle_t             handle,
                                hipblasFillMode_t           uplo,
                                hipblasOperation_t          transA,
                                hipblasDiagType_t           diag,
                                int64_t                     n,
                                int64_t                     k,
                                const hipblasDoubleComplex* A,
                                int64_t                     lda,
                                hipblasDoubleComplex*       x,
                                int64_t                     incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZtbsv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               k,
                                               (cuDoubleComplex*)A,
                                               lda,
                                               (cuDoubleComplex*)x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtbsv_v2_64(hipblasHandle_t    handle,
                                   hipblasFillMode_t  uplo,
                                   hipblasOperation_t transA,
                                   hipblasDiagType_t  diag,
                                   int64_t            n,
                                   int64_t            k,
                                   const hipComplex*  A,
                                   int64_t            lda,
                                   hipComplex*        x,
                                   int64_t            incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCtbsv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               k,
                                               (cuComplex*)A,
                                               lda,
                                               (cuComplex*)x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtbsv_v2_64(hipblasHandle_t         handle,
                                   hipblasFillMode_t       uplo,
                                   hipblasOperation_t      transA,
                                   hipblasDiagType_t       diag,
                                   int64_t                 n,
                                   int64_t                 k,
                                   const hipDoubleComplex* A,
                                   int64_t                 lda,
                                   hipDoubleComplex*       x,
                                   int64_t                 incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZtbsv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               k,
                                               (cuDoubleComplex*)A,
                                               lda,
                                               (cuDoubleComplex*)x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tbsv_batched
hipblasStatus_t hipblasStbsvBatched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    hipblasOperation_t transA,
                                    hipblasDiagType_t  diag,
                                    int                n,
                                    int                k,
                                    const float* const A[],
                                    int                lda,
                                    float* const       x[],
                                    int                incx,
                                    int                batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDtbsvBatched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    hipblasOperation_t  transA,
                                    hipblasDiagType_t   diag,
                                    int                 n,
                                    int                 k,
                                    const double* const A[],
                                    int                 lda,
                                    double* const       x[],
                                    int                 incx,
                                    int                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtbsvBatched(hipblasHandle_t             handle,
                                    hipblasFillMode_t           uplo,
                                    hipblasOperation_t          transA,
                                    hipblasDiagType_t           diag,
                                    int                         n,
                                    int                         k,
                                    const hipblasComplex* const A[],
                                    int                         lda,
                                    hipblasComplex* const       x[],
                                    int                         incx,
                                    int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtbsvBatched(hipblasHandle_t                   handle,
                                    hipblasFillMode_t                 uplo,
                                    hipblasOperation_t                transA,
                                    hipblasDiagType_t                 diag,
                                    int                               n,
                                    int                               k,
                                    const hipblasDoubleComplex* const A[],
                                    int                               lda,
                                    hipblasDoubleComplex* const       x[],
                                    int                               incx,
                                    int                               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtbsvBatched_v2(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       hipblasOperation_t      transA,
                                       hipblasDiagType_t       diag,
                                       int                     n,
                                       int                     k,
                                       const hipComplex* const A[],
                                       int                     lda,
                                       hipComplex* const       x[],
                                       int                     incx,
                                       int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtbsvBatched_v2(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       hipblasOperation_t            transA,
                                       hipblasDiagType_t             diag,
                                       int                           n,
                                       int                           k,
                                       const hipDoubleComplex* const A[],
                                       int                           lda,
                                       hipDoubleComplex* const       x[],
                                       int                           incx,
                                       int                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// tbsv_batched_64
hipblasStatus_t hipblasStbsvBatched_64(hipblasHandle_t    handle,
                                       hipblasFillMode_t  uplo,
                                       hipblasOperation_t transA,
                                       hipblasDiagType_t  diag,
                                       int64_t            n,
                                       int64_t            k,
                                       const float* const A[],
                                       int64_t            lda,
                                       float* const       x[],
                                       int64_t            incx,
                                       int64_t            batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDtbsvBatched_64(hipblasHandle_t     handle,
                                       hipblasFillMode_t   uplo,
                                       hipblasOperation_t  transA,
                                       hipblasDiagType_t   diag,
                                       int64_t             n,
                                       int64_t             k,
                                       const double* const A[],
                                       int64_t             lda,
                                       double* const       x[],
                                       int64_t             incx,
                                       int64_t             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtbsvBatched_64(hipblasHandle_t             handle,
                                       hipblasFillMode_t           uplo,
                                       hipblasOperation_t          transA,
                                       hipblasDiagType_t           diag,
                                       int64_t                     n,
                                       int64_t                     k,
                                       const hipblasComplex* const A[],
                                       int64_t                     lda,
                                       hipblasComplex* const       x[],
                                       int64_t                     incx,
                                       int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtbsvBatched_64(hipblasHandle_t                   handle,
                                       hipblasFillMode_t                 uplo,
                                       hipblasOperation_t                transA,
                                       hipblasDiagType_t                 diag,
                                       int64_t                           n,
                                       int64_t                           k,
                                       const hipblasDoubleComplex* const A[],
                                       int64_t                           lda,
                                       hipblasDoubleComplex* const       x[],
                                       int64_t                           incx,
                                       int64_t                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtbsvBatched_v2_64(hipblasHandle_t         handle,
                                          hipblasFillMode_t       uplo,
                                          hipblasOperation_t      transA,
                                          hipblasDiagType_t       diag,
                                          int64_t                 n,
                                          int64_t                 k,
                                          const hipComplex* const A[],
                                          int64_t                 lda,
                                          hipComplex* const       x[],
                                          int64_t                 incx,
                                          int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtbsvBatched_v2_64(hipblasHandle_t               handle,
                                          hipblasFillMode_t             uplo,
                                          hipblasOperation_t            transA,
                                          hipblasDiagType_t             diag,
                                          int64_t                       n,
                                          int64_t                       k,
                                          const hipDoubleComplex* const A[],
                                          int64_t                       lda,
                                          hipDoubleComplex* const       x[],
                                          int64_t                       incx,
                                          int64_t                       batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// tbsv_strided_batched
hipblasStatus_t hipblasStbsvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           int                k,
                                           const float*       A,
                                           int                lda,
                                           hipblasStride      strideA,
                                           float*             x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDtbsvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           int                k,
                                           const double*      A,
                                           int                lda,
                                           hipblasStride      strideA,
                                           double*            x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtbsvStridedBatched(hipblasHandle_t       handle,
                                           hipblasFillMode_t     uplo,
                                           hipblasOperation_t    transA,
                                           hipblasDiagType_t     diag,
                                           int                   n,
                                           int                   k,
                                           const hipblasComplex* A,
                                           int                   lda,
                                           hipblasStride         strideA,
                                           hipblasComplex*       x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtbsvStridedBatched(hipblasHandle_t             handle,
                                           hipblasFillMode_t           uplo,
                                           hipblasOperation_t          transA,
                                           hipblasDiagType_t           diag,
                                           int                         n,
                                           int                         k,
                                           const hipblasDoubleComplex* A,
                                           int                         lda,
                                           hipblasStride               strideA,
                                           hipblasDoubleComplex*       x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtbsvStridedBatched_v2(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int                n,
                                              int                k,
                                              const hipComplex*  A,
                                              int                lda,
                                              hipblasStride      strideA,
                                              hipComplex*        x,
                                              int                incx,
                                              hipblasStride      stridex,
                                              int                batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtbsvStridedBatched_v2(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              hipblasOperation_t      transA,
                                              hipblasDiagType_t       diag,
                                              int                     n,
                                              int                     k,
                                              const hipDoubleComplex* A,
                                              int                     lda,
                                              hipblasStride           strideA,
                                              hipDoubleComplex*       x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// tbsv_strided_batched_64
hipblasStatus_t hipblasStbsvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int64_t            n,
                                              int64_t            k,
                                              const float*       A,
                                              int64_t            lda,
                                              hipblasStride      strideA,
                                              float*             x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              int64_t            batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDtbsvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int64_t            n,
                                              int64_t            k,
                                              const double*      A,
                                              int64_t            lda,
                                              hipblasStride      strideA,
                                              double*            x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              int64_t            batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtbsvStridedBatched_64(hipblasHandle_t       handle,
                                              hipblasFillMode_t     uplo,
                                              hipblasOperation_t    transA,
                                              hipblasDiagType_t     diag,
                                              int64_t               n,
                                              int64_t               k,
                                              const hipblasComplex* A,
                                              int64_t               lda,
                                              hipblasStride         strideA,
                                              hipblasComplex*       x,
                                              int64_t               incx,
                                              hipblasStride         stridex,
                                              int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtbsvStridedBatched_64(hipblasHandle_t             handle,
                                              hipblasFillMode_t           uplo,
                                              hipblasOperation_t          transA,
                                              hipblasDiagType_t           diag,
                                              int64_t                     n,
                                              int64_t                     k,
                                              const hipblasDoubleComplex* A,
                                              int64_t                     lda,
                                              hipblasStride               strideA,
                                              hipblasDoubleComplex*       x,
                                              int64_t                     incx,
                                              hipblasStride               stridex,
                                              int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtbsvStridedBatched_v2_64(hipblasHandle_t    handle,
                                                 hipblasFillMode_t  uplo,
                                                 hipblasOperation_t transA,
                                                 hipblasDiagType_t  diag,
                                                 int64_t            n,
                                                 int64_t            k,
                                                 const hipComplex*  A,
                                                 int64_t            lda,
                                                 hipblasStride      strideA,
                                                 hipComplex*        x,
                                                 int64_t            incx,
                                                 hipblasStride      stridex,
                                                 int64_t            batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtbsvStridedBatched_v2_64(hipblasHandle_t         handle,
                                                 hipblasFillMode_t       uplo,
                                                 hipblasOperation_t      transA,
                                                 hipblasDiagType_t       diag,
                                                 int64_t                 n,
                                                 int64_t                 k,
                                                 const hipDoubleComplex* A,
                                                 int64_t                 lda,
                                                 hipblasStride           strideA,
                                                 hipDoubleComplex*       x,
                                                 int64_t                 incx,
                                                 hipblasStride           stridex,
                                                 int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// tpmv
hipblasStatus_t hipblasStpmv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             const float*       AP,
                             float*             x,
                             int                incx)
try
{
    return hipblasConvertStatus(cublasStpmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            AP,
                                            x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtpmv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             const double*      AP,
                             double*            x,
                             int                incx)
try
{
    return hipblasConvertStatus(cublasDtpmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            AP,
                                            x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtpmv(hipblasHandle_t       handle,
                             hipblasFillMode_t     uplo,
                             hipblasOperation_t    transA,
                             hipblasDiagType_t     diag,
                             int                   n,
                             const hipblasComplex* AP,
                             hipblasComplex*       x,
                             int                   incx)
try
{
    return hipblasConvertStatus(cublasCtpmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            (cuComplex*)AP,
                                            (cuComplex*)x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtpmv(hipblasHandle_t             handle,
                             hipblasFillMode_t           uplo,
                             hipblasOperation_t          transA,
                             hipblasDiagType_t           diag,
                             int                         n,
                             const hipblasDoubleComplex* AP,
                             hipblasDoubleComplex*       x,
                             int                         incx)
try
{
    return hipblasConvertStatus(cublasZtpmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            (cuDoubleComplex*)AP,
                                            (cuDoubleComplex*)x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtpmv_v2(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int                n,
                                const hipComplex*  AP,
                                hipComplex*        x,
                                int                incx)
try
{
    return hipblasConvertStatus(cublasCtpmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            (cuComplex*)AP,
                                            (cuComplex*)x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtpmv_v2(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                hipblasOperation_t      transA,
                                hipblasDiagType_t       diag,
                                int                     n,
                                const hipDoubleComplex* AP,
                                hipDoubleComplex*       x,
                                int                     incx)
try
{
    return hipblasConvertStatus(cublasZtpmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            (cuDoubleComplex*)AP,
                                            (cuDoubleComplex*)x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tpmv_64
hipblasStatus_t hipblasStpmv_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int64_t            n,
                                const float*       AP,
                                float*             x,
                                int64_t            incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasStpmv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               AP,
                                               x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtpmv_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int64_t            n,
                                const double*      AP,
                                double*            x,
                                int64_t            incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasDtpmv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               AP,
                                               x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtpmv_64(hipblasHandle_t       handle,
                                hipblasFillMode_t     uplo,
                                hipblasOperation_t    transA,
                                hipblasDiagType_t     diag,
                                int64_t               n,
                                const hipblasComplex* AP,
                                hipblasComplex*       x,
                                int64_t               incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCtpmv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               (cuComplex*)AP,
                                               (cuComplex*)x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtpmv_64(hipblasHandle_t             handle,
                                hipblasFillMode_t           uplo,
                                hipblasOperation_t          transA,
                                hipblasDiagType_t           diag,
                                int64_t                     n,
                                const hipblasDoubleComplex* AP,
                                hipblasDoubleComplex*       x,
                                int64_t                     incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZtpmv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               (cuDoubleComplex*)AP,
                                               (cuDoubleComplex*)x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtpmv_v2_64(hipblasHandle_t    handle,
                                   hipblasFillMode_t  uplo,
                                   hipblasOperation_t transA,
                                   hipblasDiagType_t  diag,
                                   int64_t            n,
                                   const hipComplex*  AP,
                                   hipComplex*        x,
                                   int64_t            incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCtpmv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               (cuComplex*)AP,
                                               (cuComplex*)x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtpmv_v2_64(hipblasHandle_t         handle,
                                   hipblasFillMode_t       uplo,
                                   hipblasOperation_t      transA,
                                   hipblasDiagType_t       diag,
                                   int64_t                 n,
                                   const hipDoubleComplex* AP,
                                   hipDoubleComplex*       x,
                                   int64_t                 incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZtpmv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               (cuDoubleComplex*)AP,
                                               (cuDoubleComplex*)x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tpmv_batched
hipblasStatus_t hipblasStpmvBatched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    hipblasOperation_t transA,
                                    hipblasDiagType_t  diag,
                                    int                n,
                                    const float* const AP[],
                                    float* const       x[],
                                    int                incx,
                                    int                batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDtpmvBatched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    hipblasOperation_t  transA,
                                    hipblasDiagType_t   diag,
                                    int                 n,
                                    const double* const AP[],
                                    double* const       x[],
                                    int                 incx,
                                    int                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtpmvBatched(hipblasHandle_t             handle,
                                    hipblasFillMode_t           uplo,
                                    hipblasOperation_t          transA,
                                    hipblasDiagType_t           diag,
                                    int                         n,
                                    const hipblasComplex* const AP[],
                                    hipblasComplex* const       x[],
                                    int                         incx,
                                    int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtpmvBatched(hipblasHandle_t                   handle,
                                    hipblasFillMode_t                 uplo,
                                    hipblasOperation_t                transA,
                                    hipblasDiagType_t                 diag,
                                    int                               n,
                                    const hipblasDoubleComplex* const AP[],
                                    hipblasDoubleComplex* const       x[],
                                    int                               incx,
                                    int                               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtpmvBatched_v2(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       hipblasOperation_t      transA,
                                       hipblasDiagType_t       diag,
                                       int                     n,
                                       const hipComplex* const AP[],
                                       hipComplex* const       x[],
                                       int                     incx,
                                       int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtpmvBatched_v2(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       hipblasOperation_t            transA,
                                       hipblasDiagType_t             diag,
                                       int                           n,
                                       const hipDoubleComplex* const AP[],
                                       hipDoubleComplex* const       x[],
                                       int                           incx,
                                       int                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// tpmv_batched_64
hipblasStatus_t hipblasStpmvBatched_64(hipblasHandle_t    handle,
                                       hipblasFillMode_t  uplo,
                                       hipblasOperation_t transA,
                                       hipblasDiagType_t  diag,
                                       int64_t            n,
                                       const float* const AP[],
                                       float* const       x[],
                                       int64_t            incx,
                                       int64_t            batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDtpmvBatched_64(hipblasHandle_t     handle,
                                       hipblasFillMode_t   uplo,
                                       hipblasOperation_t  transA,
                                       hipblasDiagType_t   diag,
                                       int64_t             n,
                                       const double* const AP[],
                                       double* const       x[],
                                       int64_t             incx,
                                       int64_t             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtpmvBatched_64(hipblasHandle_t             handle,
                                       hipblasFillMode_t           uplo,
                                       hipblasOperation_t          transA,
                                       hipblasDiagType_t           diag,
                                       int64_t                     n,
                                       const hipblasComplex* const AP[],
                                       hipblasComplex* const       x[],
                                       int64_t                     incx,
                                       int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtpmvBatched_64(hipblasHandle_t                   handle,
                                       hipblasFillMode_t                 uplo,
                                       hipblasOperation_t                transA,
                                       hipblasDiagType_t                 diag,
                                       int64_t                           n,
                                       const hipblasDoubleComplex* const AP[],
                                       hipblasDoubleComplex* const       x[],
                                       int64_t                           incx,
                                       int64_t                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtpmvBatched_v2_64(hipblasHandle_t         handle,
                                          hipblasFillMode_t       uplo,
                                          hipblasOperation_t      transA,
                                          hipblasDiagType_t       diag,
                                          int64_t                 n,
                                          const hipComplex* const AP[],
                                          hipComplex* const       x[],
                                          int64_t                 incx,
                                          int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtpmvBatched_v2_64(hipblasHandle_t               handle,
                                          hipblasFillMode_t             uplo,
                                          hipblasOperation_t            transA,
                                          hipblasDiagType_t             diag,
                                          int64_t                       n,
                                          const hipDoubleComplex* const AP[],
                                          hipDoubleComplex* const       x[],
                                          int64_t                       incx,
                                          int64_t                       batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// tpmv_strided_batched
hipblasStatus_t hipblasStpmvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           const float*       AP,
                                           hipblasStride      strideAP,
                                           float*             x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDtpmvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           const double*      AP,
                                           hipblasStride      strideAP,
                                           double*            x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtpmvStridedBatched(hipblasHandle_t       handle,
                                           hipblasFillMode_t     uplo,
                                           hipblasOperation_t    transA,
                                           hipblasDiagType_t     diag,
                                           int                   n,
                                           const hipblasComplex* AP,
                                           hipblasStride         strideAP,
                                           hipblasComplex*       x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtpmvStridedBatched(hipblasHandle_t             handle,
                                           hipblasFillMode_t           uplo,
                                           hipblasOperation_t          transA,
                                           hipblasDiagType_t           diag,
                                           int                         n,
                                           const hipblasDoubleComplex* AP,
                                           hipblasStride               strideAP,
                                           hipblasDoubleComplex*       x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtpmvStridedBatched_v2(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int                n,
                                              const hipComplex*  AP,
                                              hipblasStride      strideAP,
                                              hipComplex*        x,
                                              int                incx,
                                              hipblasStride      stridex,
                                              int                batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtpmvStridedBatched_v2(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              hipblasOperation_t      transA,
                                              hipblasDiagType_t       diag,
                                              int                     n,
                                              const hipDoubleComplex* AP,
                                              hipblasStride           strideAP,
                                              hipDoubleComplex*       x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// tpmv_strided_batched_64
hipblasStatus_t hipblasStpmvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int64_t            n,
                                              const float*       AP,
                                              hipblasStride      strideAP,
                                              float*             x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              int64_t            batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDtpmvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int64_t            n,
                                              const double*      AP,
                                              hipblasStride      strideAP,
                                              double*            x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              int64_t            batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtpmvStridedBatched_64(hipblasHandle_t       handle,
                                              hipblasFillMode_t     uplo,
                                              hipblasOperation_t    transA,
                                              hipblasDiagType_t     diag,
                                              int64_t               n,
                                              const hipblasComplex* AP,
                                              hipblasStride         strideAP,
                                              hipblasComplex*       x,
                                              int64_t               incx,
                                              hipblasStride         stridex,
                                              int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtpmvStridedBatched_64(hipblasHandle_t             handle,
                                              hipblasFillMode_t           uplo,
                                              hipblasOperation_t          transA,
                                              hipblasDiagType_t           diag,
                                              int64_t                     n,
                                              const hipblasDoubleComplex* AP,
                                              hipblasStride               strideAP,
                                              hipblasDoubleComplex*       x,
                                              int64_t                     incx,
                                              hipblasStride               stridex,
                                              int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtpmvStridedBatched_v2_64(hipblasHandle_t    handle,
                                                 hipblasFillMode_t  uplo,
                                                 hipblasOperation_t transA,
                                                 hipblasDiagType_t  diag,
                                                 int64_t            n,
                                                 const hipComplex*  AP,
                                                 hipblasStride      strideAP,
                                                 hipComplex*        x,
                                                 int64_t            incx,
                                                 hipblasStride      stridex,
                                                 int64_t            batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtpmvStridedBatched_v2_64(hipblasHandle_t         handle,
                                                 hipblasFillMode_t       uplo,
                                                 hipblasOperation_t      transA,
                                                 hipblasDiagType_t       diag,
                                                 int64_t                 n,
                                                 const hipDoubleComplex* AP,
                                                 hipblasStride           strideAP,
                                                 hipDoubleComplex*       x,
                                                 int64_t                 incx,
                                                 hipblasStride           stridex,
                                                 int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// tpsv
hipblasStatus_t hipblasStpsv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             const float*       AP,
                             float*             x,
                             int                incx)
try
{
    return hipblasConvertStatus(cublasStpsv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            AP,
                                            x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtpsv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             const double*      AP,
                             double*            x,
                             int                incx)
try
{
    return hipblasConvertStatus(cublasDtpsv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            AP,
                                            x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtpsv(hipblasHandle_t       handle,
                             hipblasFillMode_t     uplo,
                             hipblasOperation_t    transA,
                             hipblasDiagType_t     diag,
                             int                   n,
                             const hipblasComplex* AP,
                             hipblasComplex*       x,
                             int                   incx)
try
{
    return hipblasConvertStatus(cublasCtpsv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            (cuComplex*)AP,
                                            (cuComplex*)x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtpsv(hipblasHandle_t             handle,
                             hipblasFillMode_t           uplo,
                             hipblasOperation_t          transA,
                             hipblasDiagType_t           diag,
                             int                         n,
                             const hipblasDoubleComplex* AP,
                             hipblasDoubleComplex*       x,
                             int                         incx)
try
{
    return hipblasConvertStatus(cublasZtpsv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            (cuDoubleComplex*)AP,
                                            (cuDoubleComplex*)x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtpsv_v2(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int                n,
                                const hipComplex*  AP,
                                hipComplex*        x,
                                int                incx)
try
{
    return hipblasConvertStatus(cublasCtpsv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            (cuComplex*)AP,
                                            (cuComplex*)x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtpsv_v2(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                hipblasOperation_t      transA,
                                hipblasDiagType_t       diag,
                                int                     n,
                                const hipDoubleComplex* AP,
                                hipDoubleComplex*       x,
                                int                     incx)
try
{
    return hipblasConvertStatus(cublasZtpsv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            (cuDoubleComplex*)AP,
                                            (cuDoubleComplex*)x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tpsv_64
hipblasStatus_t hipblasStpsv_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int64_t            n,
                                const float*       AP,
                                float*             x,
                                int64_t            incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasStpsv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               AP,
                                               x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtpsv_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int64_t            n,
                                const double*      AP,
                                double*            x,
                                int64_t            incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasDtpsv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               AP,
                                               x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtpsv_64(hipblasHandle_t       handle,
                                hipblasFillMode_t     uplo,
                                hipblasOperation_t    transA,
                                hipblasDiagType_t     diag,
                                int64_t               n,
                                const hipblasComplex* AP,
                                hipblasComplex*       x,
                                int64_t               incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCtpsv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               (cuComplex*)AP,
                                               (cuComplex*)x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtpsv_64(hipblasHandle_t             handle,
                                hipblasFillMode_t           uplo,
                                hipblasOperation_t          transA,
                                hipblasDiagType_t           diag,
                                int64_t                     n,
                                const hipblasDoubleComplex* AP,
                                hipblasDoubleComplex*       x,
                                int64_t                     incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZtpsv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               (cuDoubleComplex*)AP,
                                               (cuDoubleComplex*)x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtpsv_v2_64(hipblasHandle_t    handle,
                                   hipblasFillMode_t  uplo,
                                   hipblasOperation_t transA,
                                   hipblasDiagType_t  diag,
                                   int64_t            n,
                                   const hipComplex*  AP,
                                   hipComplex*        x,
                                   int64_t            incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCtpsv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               (cuComplex*)AP,
                                               (cuComplex*)x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtpsv_v2_64(hipblasHandle_t         handle,
                                   hipblasFillMode_t       uplo,
                                   hipblasOperation_t      transA,
                                   hipblasDiagType_t       diag,
                                   int64_t                 n,
                                   const hipDoubleComplex* AP,
                                   hipDoubleComplex*       x,
                                   int64_t                 incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZtpsv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               (cuDoubleComplex*)AP,
                                               (cuDoubleComplex*)x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tpsv_batched
hipblasStatus_t hipblasStpsvBatched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    hipblasOperation_t transA,
                                    hipblasDiagType_t  diag,
                                    int                n,
                                    const float* const AP[],
                                    float* const       x[],
                                    int                incx,
                                    int                batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDtpsvBatched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    hipblasOperation_t  transA,
                                    hipblasDiagType_t   diag,
                                    int                 n,
                                    const double* const AP[],
                                    double* const       x[],
                                    int                 incx,
                                    int                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtpsvBatched(hipblasHandle_t             handle,
                                    hipblasFillMode_t           uplo,
                                    hipblasOperation_t          transA,
                                    hipblasDiagType_t           diag,
                                    int                         n,
                                    const hipblasComplex* const AP[],
                                    hipblasComplex* const       x[],
                                    int                         incx,
                                    int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtpsvBatched(hipblasHandle_t                   handle,
                                    hipblasFillMode_t                 uplo,
                                    hipblasOperation_t                transA,
                                    hipblasDiagType_t                 diag,
                                    int                               n,
                                    const hipblasDoubleComplex* const AP[],
                                    hipblasDoubleComplex* const       x[],
                                    int                               incx,
                                    int                               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtpsvBatched_v2(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       hipblasOperation_t      transA,
                                       hipblasDiagType_t       diag,
                                       int                     n,
                                       const hipComplex* const AP[],
                                       hipComplex* const       x[],
                                       int                     incx,
                                       int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtpsvBatched_v2(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       hipblasOperation_t            transA,
                                       hipblasDiagType_t             diag,
                                       int                           n,
                                       const hipDoubleComplex* const AP[],
                                       hipDoubleComplex* const       x[],
                                       int                           incx,
                                       int                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// tpsv_batched_64
hipblasStatus_t hipblasStpsvBatched_64(hipblasHandle_t    handle,
                                       hipblasFillMode_t  uplo,
                                       hipblasOperation_t transA,
                                       hipblasDiagType_t  diag,
                                       int64_t            n,
                                       const float* const AP[],
                                       float* const       x[],
                                       int64_t            incx,
                                       int64_t            batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDtpsvBatched_64(hipblasHandle_t     handle,
                                       hipblasFillMode_t   uplo,
                                       hipblasOperation_t  transA,
                                       hipblasDiagType_t   diag,
                                       int64_t             n,
                                       const double* const AP[],
                                       double* const       x[],
                                       int64_t             incx,
                                       int64_t             batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtpsvBatched_64(hipblasHandle_t             handle,
                                       hipblasFillMode_t           uplo,
                                       hipblasOperation_t          transA,
                                       hipblasDiagType_t           diag,
                                       int64_t                     n,
                                       const hipblasComplex* const AP[],
                                       hipblasComplex* const       x[],
                                       int64_t                     incx,
                                       int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtpsvBatched_64(hipblasHandle_t                   handle,
                                       hipblasFillMode_t                 uplo,
                                       hipblasOperation_t                transA,
                                       hipblasDiagType_t                 diag,
                                       int64_t                           n,
                                       const hipblasDoubleComplex* const AP[],
                                       hipblasDoubleComplex* const       x[],
                                       int64_t                           incx,
                                       int64_t                           batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtpsvBatched_v2_64(hipblasHandle_t         handle,
                                          hipblasFillMode_t       uplo,
                                          hipblasOperation_t      transA,
                                          hipblasDiagType_t       diag,
                                          int64_t                 n,
                                          const hipComplex* const AP[],
                                          hipComplex* const       x[],
                                          int64_t                 incx,
                                          int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtpsvBatched_v2_64(hipblasHandle_t               handle,
                                          hipblasFillMode_t             uplo,
                                          hipblasOperation_t            transA,
                                          hipblasDiagType_t             diag,
                                          int64_t                       n,
                                          const hipDoubleComplex* const AP[],
                                          hipDoubleComplex* const       x[],
                                          int64_t                       incx,
                                          int64_t                       batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// tpsv_strided_batched
hipblasStatus_t hipblasStpsvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           const float*       AP,
                                           hipblasStride      strideAP,
                                           float*             x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDtpsvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           const double*      AP,
                                           hipblasStride      strideAP,
                                           double*            x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtpsvStridedBatched(hipblasHandle_t       handle,
                                           hipblasFillMode_t     uplo,
                                           hipblasOperation_t    transA,
                                           hipblasDiagType_t     diag,
                                           int                   n,
                                           const hipblasComplex* AP,
                                           hipblasStride         strideAP,
                                           hipblasComplex*       x,
                                           int                   incx,
                                           hipblasStride         stridex,
                                           int                   batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtpsvStridedBatched(hipblasHandle_t             handle,
                                           hipblasFillMode_t           uplo,
                                           hipblasOperation_t          transA,
                                           hipblasDiagType_t           diag,
                                           int                         n,
                                           const hipblasDoubleComplex* AP,
                                           hipblasStride               strideAP,
                                           hipblasDoubleComplex*       x,
                                           int                         incx,
                                           hipblasStride               stridex,
                                           int                         batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtpsvStridedBatched_v2(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int                n,
                                              const hipComplex*  AP,
                                              hipblasStride      strideAP,
                                              hipComplex*        x,
                                              int                incx,
                                              hipblasStride      stridex,
                                              int                batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtpsvStridedBatched_v2(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              hipblasOperation_t      transA,
                                              hipblasDiagType_t       diag,
                                              int                     n,
                                              const hipDoubleComplex* AP,
                                              hipblasStride           strideAP,
                                              hipDoubleComplex*       x,
                                              int                     incx,
                                              hipblasStride           stridex,
                                              int                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// tpsv_strided_batched_64
hipblasStatus_t hipblasStpsvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int64_t            n,
                                              const float*       AP,
                                              hipblasStride      strideAP,
                                              float*             x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              int64_t            batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDtpsvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int64_t            n,
                                              const double*      AP,
                                              hipblasStride      strideAP,
                                              double*            x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              int64_t            batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtpsvStridedBatched_64(hipblasHandle_t       handle,
                                              hipblasFillMode_t     uplo,
                                              hipblasOperation_t    transA,
                                              hipblasDiagType_t     diag,
                                              int64_t               n,
                                              const hipblasComplex* AP,
                                              hipblasStride         strideAP,
                                              hipblasComplex*       x,
                                              int64_t               incx,
                                              hipblasStride         stridex,
                                              int64_t               batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtpsvStridedBatched_64(hipblasHandle_t             handle,
                                              hipblasFillMode_t           uplo,
                                              hipblasOperation_t          transA,
                                              hipblasDiagType_t           diag,
                                              int64_t                     n,
                                              const hipblasDoubleComplex* AP,
                                              hipblasStride               strideAP,
                                              hipblasDoubleComplex*       x,
                                              int64_t                     incx,
                                              hipblasStride               stridex,
                                              int64_t                     batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtpsvStridedBatched_v2_64(hipblasHandle_t    handle,
                                                 hipblasFillMode_t  uplo,
                                                 hipblasOperation_t transA,
                                                 hipblasDiagType_t  diag,
                                                 int64_t            n,
                                                 const hipComplex*  AP,
                                                 hipblasStride      strideAP,
                                                 hipComplex*        x,
                                                 int64_t            incx,
                                                 hipblasStride      stridex,
                                                 int64_t            batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtpsvStridedBatched_v2_64(hipblasHandle_t         handle,
                                                 hipblasFillMode_t       uplo,
                                                 hipblasOperation_t      transA,
                                                 hipblasDiagType_t       diag,
                                                 int64_t                 n,
                                                 const hipDoubleComplex* AP,
                                                 hipblasStride           strideAP,
                                                 hipDoubleComplex*       x,
                                                 int64_t                 incx,
                                                 hipblasStride           stridex,
                                                 int64_t                 batchCount)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// trmv
hipblasStatus_t hipblasStrmv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             const float*       A,
                             int                lda,
                             float*             x,
                             int                incx)
try
{
    return hipblasConvertStatus(cublasStrmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            A,
                                            lda,
                                            x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtrmv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             const double*      A,
                             int                lda,
                             double*            x,
                             int                incx)
try
{
    return hipblasConvertStatus(cublasDtrmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            A,
                                            lda,
                                            x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtrmv(hipblasHandle_t       handle,
                             hipblasFillMode_t     uplo,
                             hipblasOperation_t    transA,
                             hipblasDiagType_t     diag,
                             int                   n,
                             const hipblasComplex* A,
                             int                   lda,
                             hipblasComplex*       x,
                             int                   incx)
try
{
    return hipblasConvertStatus(cublasCtrmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            (cuComplex*)A,
                                            lda,
                                            (cuComplex*)x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtrmv(hipblasHandle_t             handle,
                             hipblasFillMode_t           uplo,
                             hipblasOperation_t          transA,
                             hipblasDiagType_t           diag,
                             int                         n,
                             const hipblasDoubleComplex* A,
                             int                         lda,
                             hipblasDoubleComplex*       x,
                             int                         incx)
try
{
    return hipblasConvertStatus(cublasZtrmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            (cuDoubleComplex*)A,
                                            lda,
                                            (cuDoubleComplex*)x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtrmv_v2(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int                n,
                                const hipComplex*  A,
                                int                lda,
                                hipComplex*        x,
                                int                incx)
try
{
    return hipblasConvertStatus(cublasCtrmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            (cuComplex*)A,
                                            lda,
                                            (cuComplex*)x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtrmv_v2(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                hipblasOperation_t      transA,
                                hipblasDiagType_t       diag,
                                int                     n,
                                const hipDoubleComplex* A,
                                int                     lda,
                                hipDoubleComplex*       x,
                                int                     incx)
try
{
    return hipblasConvertStatus(cublasZtrmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            (cuDoubleComplex*)A,
                                            lda,
                                            (cuDoubleComplex*)x,
                                            incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// trmv_64
hipblasStatus_t hipblasStrmv_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int64_t            n,
                                const float*       A,
                                int64_t            lda,
                                float*             x,
                                int64_t            incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasStrmv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               A,
                                               lda,
                                               x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtrmv_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int64_t            n,
                                const double*      A,
                                int64_t            lda,
                                double*            x,
                                int64_t            incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasDtrmv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               A,
                                               lda,
                                               x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtrmv_64(hipblasHandle_t       handle,
                                hipblasFillMode_t     uplo,
                                hipblasOperation_t    transA,
                                hipblasDiagType_t     diag,
                                int64_t               n,
                                const hipblasComplex* A,
                                int64_t               lda,
                                hipblasComplex*       x,
                                int64_t               incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCtrmv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               (cuComplex*)A,
                                               lda,
                                               (cuComplex*)x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtrmv_64(hipblasHandle_t             handle,
                                hipblasFillMode_t           uplo,
                                hipblasOperation_t          transA,
                                hipblasDiagType_t           diag,
                                int64_t                     n,
                                const hipblasDoubleComplex* A,
                                int64_t                     lda,
                                hipblasDoubleComplex*       x,
                                int64_t                     incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZtrmv((cublasHandle_t)handle,
                                            hipblasConvertFill(uplo),
                                            hipblasConvertOperation(transA),
                                            hipblasConvertDiag(diag),
                                            n,
                                            (cuDoubleComplex*)A,
                                            lda,
                                            (cuDoubleComplex*)x,
                                            incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtrmv_v2_64(hipblasHandle_t    handle,
                                   hipblasFillMode_t  uplo,
                                   hipblasOperation_t transA,
                                   hipblasDiagType_t  diag,
                                   int64_t            n,
                                   const hipComplex*  A,
                                   int64_t            lda,
                                   hipComplex*        x,
                                   int64_t            incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasCtrmv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               (cuComplex*)A,
                                               lda,
                                               (cuComplex*)x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtrmv_v2_64(hipblasHandle_t         handle,
                                   hipblasFillMode_t       uplo,
                                   hipblasOperation_t      transA,
                                   hipblasDiagType_t       diag,
                                   int64_t                 n,
                                   const hipDoubleComplex* A,
                                   int64_t                 lda,
                                   hipDoubleComplex*       x,
                                   int64_t                 incx)
try
{
#if CUBLAS_VER_MAJOR >= 12
    return hipblasConvertStatus(cublasZtrmv_64((cublasHandle_t)handle,
                                               hipblasConvertFill(uplo),
                                               hipblasConvertOperation(transA),
                                               hipblasConvertDiag(diag),
                                               n,
                                               (cuDoubleComplex*)A,
                                               lda,
                                               (cuDoubleComplex*)x,
                                               incx));
#else
    return HIPBLAS_STATUS_NOT_SUPPORTED;
#endif
}
catch(...)
{
    return hipblas_exception_to_status();
}

// trmv_batched
hipblasStatus_t hipblasStrmvBatched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    hipblasOperation_t transA,
                                    hipblasDiagType_t  diag,
                                    int                n,
                                    const float* const A[],
                                    int                lda,
                                    float* const       x[],
                                    int                incx,
                                    int                batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDtrmvBatched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    hipblasOperation_t  transA,
                                    hipblasDiagType_t   diag,
                                    int                 n,
                                    const double* const A[],
                                    int                 lda,
                                    double* const       x[],
                                    int                 incx,
                                    int                 batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtrmvBatched(hipblasHandle_t             handle,
                                    hipblasFillMode_t           uplo,
                                    hipblasOperation_t          transA,
                                    hipblasDiagType_t           diag,
                                    int                         n,
                                    const hipblasComplex* const A[],
                                    int                         lda,
                                    hipblasComplex* const       x[],
                                    int                         incx,
                                    int                         batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtrmvBatched(hipblasHandle_t                   handle,
                                    hipblasFillMode_t                 uplo,
                                    hipblasOperation_t                transA,
                                    hipblasDiagType_t                 diag,
                                    int                               n,
                                    const hipblasDoubleComplex* const A[],
                                    int                               lda,
                                    hipblasDoubleComplex* const       x[],
                                    int                               incx,
                                    int                               batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtrmvBatched_v2(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       hipblasOperation_t      transA,
                                       hipblasDiagType_t       diag,
                                       int                     n,
                                       const hipComplex* const A[],
                                       int                     lda,
                                       hipComplex* const       x[],
                                       int                     incx,
                                       int                     batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtrmvBatched_v2(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       hipblasOperation_t            transA,
                                       hipblasDiagType_t             diag,
                                       int                           n,
                                       const hipDoubleComplex* const A[],
                                       int                           lda,
                                       hipDoubleComplex* const       x[],
                                       int                           incx,
                                       int                           batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// trmv_batched_64
hipblasStatus_t hipblasStrmvBatched_64(hipblasHandle_t    handle,
                                       hipblasFillMode_t  uplo,
                                       hipblasOperation_t transA,
                                       hipblasDiagType_t  diag,
                                       int64_t            n,
                                       const float* const A[],
                                       int64_t            lda,
                                       float* const       x[],
                                       int64_t            incx,
                                       int64_t            batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDtrmvBatched_64(hipblasHandle_t     handle,
                                       hipblasFillMode_t   uplo,
                                       hipblasOperation_t  transA,
                                       hipblasDiagType_t   diag,
                                       int64_t             n,
                                       const double* const A[],
                                       int64_t             lda,
                                       double* const       x[],
                                       int64_t             incx,
                                       int64_t             batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtrmvBatched_64(hipblasHandle_t             handle,
                                       hipblasFillMode_t           uplo,
                                       hipblasOperation_t          transA,
                                       hipblasDiagType_t           diag,
                                       int64_t                     n,
                                       const hipblasComplex* const A[],
                                       int64_t                     lda,
                                       hipblasComplex* const       x[],
                                       int64_t                     incx,
                                       int64_t                     batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtrmvBatched_64(hipblasHandle_t                   handle,
                                       hipblasFillMode_t                 uplo,
                                       hipblasOperation_t                transA,
                                       hipblasDiagType_t                 diag,
                                       int64_t                           n,
                                       const hipblasDoubleComplex* const A[],
                                       int64_t                           lda,
                                       hipblasDoubleComplex* const       x[],
                                       int64_t                           incx,
                                       int64_t                           batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtrmvBatched_v2_64(hipblasHandle_t         handle,
                                          hipblasFillMode_t       uplo,
                                          hipblasOperation_t      transA,
                                          hipblasDiagType_t       diag,
                                          int64_t                 n,
                                          const hipComplex* const A[],
                                          int64_t                 lda,
                                          hipComplex* const       x[],
                                          int64_t                 incx,
                                          int64_t                 batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtrmvBatched_v2_64(hipblasHandle_t               handle,
                                          hipblasFillMode_t             uplo,
                                          hipblasOperation_t            transA,
                                          hipblasDiagType_t             diag,
                                          int64_t                       n,
                                          const hipDoubleComplex* const A[],
                                          int64_t                       lda,
                                          hipDoubleComplex* const       x[],
                                          int64_t                       incx,
                                          int64_t                       batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// trmv_strided_batched
hipblasStatus_t hipblasStrmvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           const float*       A,
                                           int                lda,
                                           hipblasStride      stride_a,
                                           float*             x,
                                           int                incx,
                                           hipblasStride      stride_x,
                                           int                batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDtrmvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           const double*      A,
                                           int                lda,
                                           hipblasStride      stride_a,
                                           double*            x,
                                           int                incx,
                                           hipblasStride      stride_x,
                                           int                batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtrmvStridedBatched(hipblasHandle_t       handle,
                                           hipblasFillMode_t     uplo,
                                           hipblasOperation_t    transA,
                                           hipblasDiagType_t     diag,
                                           int                   n,
                                           const hipblasComplex* A,
                                           int                   lda,
                                           hipblasStride         stride_a,
                                           hipblasComplex*       x,
                                           int                   incx,
                                           hipblasStride         stride_x,
                                           int                   batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtrmvStridedBatched(hipblasHandle_t             handle,
                                           hipblasFillMode_t           uplo,
                                           hipblasOperation_t          transA,
                                           hipblasDiagType_t           diag,
                                           int                         n,
                                           const hipblasDoubleComplex* A,
                                           int                         lda,
                                           hipblasStride               stride_a,
                                           hipblasDoubleComplex*       x,
                                           int                         incx,
                                           hipblasStride               stride_x,
                                           int                         batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtrmvStridedBatched_v2(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int                n,
                                              const hipComplex*  A,
                                              int                lda,
                                              hipblasStride      stride_a,
                                              hipComplex*        x,
                                              int                incx,
                                              hipblasStride      stride_x,
                                              int                batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtrmvStridedBatched_v2(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              hipblasOperation_t      transA,
                                              hipblasDiagType_t       diag,
                                              int                     n,
                                              const hipDoubleComplex* A,
                                              int                     lda,
                                              hipblasStride           stride_a,
                                              hipDoubleComplex*       x,
                                              int                     incx,
                                              hipblasStride           stride_x,
                                              int                     batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// trmv_strided_batched_64
hipblasStatus_t hipblasStrmvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int64_t            n,
                                              const float*       A,
                                              int64_t            lda,
                                              hipblasStride      stride_a,
                                              float*             x,
                                              int64_t            incx,
                                              hipblasStride      stride_x,
                                              int64_t            batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasDtrmvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int64_t            n,
                                              const double*      A,
                                              int64_t            lda,
                                              hipblasStride      stride_a,
                                              double*            x,
                                              int64_t            incx,
                                              hipblasStride      stride_x,
                                              int64_t            batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtrmvStridedBatched_64(hipblasHandle_t       handle,
                                              hipblasFillMode_t     uplo,
                                              hipblasOperation_t    transA,
                                              hipblasDiagType_t     diag,
                                              int64_t               n,
                                              const hipblasComplex* A,
                                              int64_t               lda,
                                              hipblasStride         stride_a,
                                              hipblasComplex*       x,
                                              int64_t               incx,
                                              hipblasStride         stride_x,
                                              int64_t               batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtrmvStridedBatched_64(hipblasHandle_t             handle,
                                              hipblasFillMode_t           uplo,
                                              hipblasOperation_t          transA,
                                              hipblasDiagType_t           diag,
                                              int64_t                     n,
                                              const hipblasDoubleComplex* A,
                                              int64_t                     lda,
                                              hipblasStride               stride_a,
                                              hipblasDoubleComplex*       x,
                                              int64_t                     incx,
                                              hipblasStride               stride_x,
                                              int64_t                     batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasCtrmvStridedBatched_v2_64(hipblasHandle_t    handle,
                                                 hipblasFillMode_t  uplo,
                                                 hipblasOperation_t transA,
                                                 hipblasDiagType_t  diag,
                                                 int64_t            n,
                                                 const hipComplex*  A,
                                                 int64_t            lda,
                                                 hipblasStride      stride_a,
                                                 hipComplex*        x,
                                                 int64_t            incx,
                                                 hipblasStride      stride_x,
                                                 int64_t            batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasStatus_t hipblasZtrmvStridedBatched_v2_64(hipblasHandle_t         handle,
                                                 hipblasFillMode_t       uplo,
                                                 hipblasOperation_t      transA,
                                                 hipblasDiagType_t       diag,
                                                 int64_t                 n,
                                                 const hipDoubleComplex* A,
                                                 int64_t                 lda,
                                                 hipblasStride           stride_a,
                                                 hipDoubleComplex*       x,
                                                 int64_t                 incx,
                                                 hipblasStride           stride_x,
                                                 int64_t                 batch_count)
{
    return HIPBLAS_STATUS_NOT_SUPPORTED;
}

// trsv
hipblasStatus_t hipblasStrsv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             const float*       A,
                             int                lda,
                             float*             x,
                             int                incx)
try
{
    return hipblasConvertStatus(cublasStrsv((cublasHandle_t)handle,
                                            hipb