---
title: 0.285.0 Breaking Changes
slug: breaking-changes/0.285.0
---

# v0.285.0 Breaking Changes

This release removes support for Apollo Federation v1 and updates the federation
API to always use Federation v2.

## What Changed

The `enable_federation_2` parameter has been removed and replaced with
`federation_version`. Federation v2 is now always enabled, with version 2.11 as
the default.

```python
# Before (0.284.x and earlier)
schema = strawberry.federation.Schema(
    query=Query, enable_federation_2=True  # Opt-in to Federation 2
)

# After (0.285.0+)
schema = strawberry.federation.Schema(
    query=Query, federation_version="2.11"  # Defaults to "2.11", always Federation 2
)
```

## Impact on Your Code

### Federation v2 Users

If you were already using `enable_federation_2=True`, you can remove that
parameter:

```python
# Before
schema = strawberry.federation.Schema(query=Query, enable_federation_2=True)

# After
schema = strawberry.federation.Schema(query=Query)
```

Your schemas will continue to work without changes and will now default to
Federation v2.11.

### Federation v1 Users

If you were using Federation v1 (without `enable_federation_2`), you **must**
migrate to Federation v2. Apollo Federation v1 is no longer supported by
Strawberry.

Federation v2 provides better schema composition and additional features. Most
v1 schemas can be migrated with minimal changes:

```python
# Before (Federation v1)
schema = strawberry.federation.Schema(query=Query)

# After (Federation v2)
schema = strawberry.federation.Schema(query=Query)  # Now uses v2.11
```

Key differences in Federation v2:

1. **No `extend` keyword needed**: Types no longer need to be marked as
   extensions
2. **No `@external` required**: The `@key` directive alone is sufficient in most
   cases
3. **`@shareable` directive**: Use this to indicate fields that can be resolved
   by multiple subgraphs
4. **`@link` directive**: Automatically added to declare federation spec version

See the
[Apollo Federation v2 migration guide](https://www.apollographql.com/docs/federation/federation-2/moving-to-federation-2/)
for more details.

## Specifying Federation Version

You can now specify which Federation v2 version to use:

```python
schema = strawberry.federation.Schema(
    query=Query, federation_version="2.5"  # Use a specific version
)
```

Supported versions: 2.0, 2.1, 2.2, 2.3, 2.4, 2.5, 2.6, 2.7, 2.8, 2.9, 2.10, 2.11

This is useful if you need to ensure compatibility with a specific Apollo Router
or Gateway version, or if you want to avoid using newer directives.

### Version Validation

Strawberry validates that directives are compatible with your chosen federation
version:

```python
from strawberry.federation.schema_directives import Cost


@strawberry.federation.type
class Product:
    name: str = strawberry.federation.field(
        directives=[Cost(weight=10)]  # Requires v2.9+
    )


# This will raise an error because @cost requires v2.9+
schema = strawberry.federation.Schema(
    query=Query, federation_version="2.5"  # Too old for @cost
)
```

## Why This Change?

1. **Apollo Federation v1 is deprecated**: Apollo has moved to Federation v2 as
   the standard
2. **Simpler API**: Removes the boolean flag in favor of explicit version
   control
3. **Better features**: Federation v2 provides improved schema composition and
   new directives
4. **Version control**: Users can now specify exact federation versions for
   compatibility

## Migration Steps

### Step 1: Check Your Current Usage

If you're using `enable_federation_2=False` or not setting it at all, you're
using Federation v1 and need to migrate.

### Step 2: Remove Federation v1 Code

```python
# Remove this parameter
schema = strawberry.federation.Schema(
    query=Query, enable_federation_2=True  # Remove this line
)
```

### Step 3: Update Federation v1 Patterns

If you were using Federation v1 patterns like `extend=True` and explicit
`@external` fields, you can simplify:

```python
# Before (Federation v1)
@strawberry.federation.type(keys=["id"], extend=True)
class Product:
    id: strawberry.ID = strawberry.federation.field(external=True)
    reviews: list[Review]


# After (Federation v2)
@strawberry.federation.type(keys=["id"])
class Product:
    id: strawberry.ID
    reviews: list[Review]
```

### Step 4: Test Your Schema

Run your schema composition and tests to ensure everything works correctly with
Federation v2.

## Need Help?

- See the [Federation guide](../guides/federation.md) for comprehensive usage
  examples
- Review the
  [Apollo Federation v2 documentation](https://www.apollographql.com/docs/federation/federation-2/new-in-federation-2/)
- If you encounter issues, please
  [report them on GitHub](https://github.com/strawberry-graphql/strawberry/issues)
